package presentation.twitter;

import com.google.common.collect.ImmutableList;
import org.codehaus.jackson.annotate.JsonProperty;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

public class Tweet implements Comparable<Tweet>, Serializable {

	@JsonProperty("tweetId")
	private final long tweetId;

	@JsonProperty("username")
	private final String username;

	@JsonProperty("message")
	private final String message;

	@JsonProperty("date")
	private final Date date;

	private final List<Comment> comments = new ArrayList<Comment>();

	public Tweet(long tweetId, String username, String message, Date date) {
		this.tweetId = tweetId;
		this.username = username;
		this.message = message;
		this.date = date == null ? new Date() : date;
	}

	public long getTweetId() {
		return tweetId;
	}

	public String getUsername() {
		return username;
	}

	public String getMessage() {
		return message;
	}

	public Date getDate() {
		return date;
	}

	@JsonProperty("comments")
	public List<Comment> getComments() {
		return ImmutableList.copyOf(comments);
	}

	public void addComment(Comment comment) {
		comments.add(comment);
	}

	@Override
	public int compareTo(Tweet other) {

		int comparison = -date.compareTo(other.date);

		if (comparison == 0) {
			comparison = username.compareTo(other.username);

			if (comparison == 0) {
				comparison = message.compareTo(other.message);

				if (comparison == 0) {
					comparison = Long.valueOf(tweetId).compareTo(other.tweetId);
				}
			}
		}

		return comparison;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null || getClass() != obj.getClass()) {
			return false;
		}

		Tweet tweet = (Tweet) obj;

		if (tweetId != tweet.tweetId) {
			return false;
		}

		return true;
	}

	@Override
	public int hashCode() {
		return (int) (tweetId ^ (tweetId >>> 32));
	}

	@Override
	public String toString() {
		final StringBuilder sb = new StringBuilder();
		sb.append("Tweet");
		sb.append("{tweetId=").append(tweetId);
		sb.append(", username='").append(username).append('\'');
		sb.append(", message='").append(message).append('\'');
		sb.append(", date=").append(date);
		sb.append(", comments=").append(comments);
		sb.append('}');
		return sb.toString();
	}

	public static Builder newBuilder() {
		return new Builder();
	}

	public static class Builder {

		private long tweetId;

		@JsonProperty("username")
		private String username;

		@JsonProperty("message")
		private String message;

		@JsonProperty("date")
		private Date date;

		private Builder() {
		}

		public long getTweetId() {
			return tweetId;
		}

		public void setTweetId(long tweetId) {
			this.tweetId = tweetId;
		}

		public Builder withTweetId(long tweetId) {
			this.tweetId = tweetId;
			return this;
		}

		public String getUsername() {
			return username;
		}

		public void setUsername(String username) {
			this.username = username;
		}

		public Builder withUsername(String username) {
			this.username = username;
			return this;
		}

		public String getMessage() {
			return message;
		}

		public void setMessage(String message) {
			this.message = message;
		}

		public Builder withMessage(String message) {
			this.message = message;
			return this;
		}

		public Date getDate() {
			return date;
		}

		public void setDate(Date date) {
			this.date = date;
		}

		public Builder withDate(Date date) {
			this.date = date;
			return this;
		}

		@Override
		public String toString() {
			final StringBuilder sb = new StringBuilder();
			sb.append("Builder");
			sb.append("{tweetId=").append(tweetId);
			sb.append(", username='").append(username).append('\'');
			sb.append(", message='").append(message).append('\'');
			sb.append(", date=").append(date);
			sb.append('}');
			return sb.toString();
		}

		public Tweet build() {
			return new Tweet(tweetId, username, message, date);
		}

	}
}
