/*******************************************************************************
 * Portions created by Sebastian Thomschke are copyright (c) 2005-2010 Sebastian
 * Thomschke.
 * 
 * All Rights Reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sebastian Thomschke - initial implementation.
 *******************************************************************************/
package wrm.saferJava.oval.localization.context;

import java.util.Enumeration;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import wrm.saferJava.oval.context.ClassContext;
import wrm.saferJava.oval.context.ConstructorParameterContext;
import wrm.saferJava.oval.context.FieldContext;
import wrm.saferJava.oval.context.MethodEntryContext;
import wrm.saferJava.oval.context.MethodExitContext;
import wrm.saferJava.oval.context.MethodParameterContext;
import wrm.saferJava.oval.context.MethodReturnValueContext;
import wrm.saferJava.oval.context.OValContext;
import wrm.saferJava.oval.internal.Log;


/**
 * This renderer searches for a resource file that is in the same package and has the same name as the validated class.
 * It then tries to lookup a localized version of the validation context, e.g.<br>
 * <b>com.acme.model.Person.java<br>
 * com.acme.model.Person.properties<br>
 * com.acme.model.Person_de.properties<br>
 * com.acme.model.Person_fr.properties</b>
 * 
 * <p>
 * The properties file is expected to have values following this scheme
 * <pre>
 * label.class=My translated name of the class name
 * label.field.firstname=My translated name of the field "firstname"
 * label.field.lastname=My translated name of the field "lastname"
 * label.parameter.amount=My translated name of a constructor/method parameter "amount"
 * label.method.increase=My translated name of the method "increase"
 * </pre>
 * @author Sebastian Thomschke
 */
public class ResourceBundleValidationContextRenderer implements OValContextRenderer
{
	private static final Log  LOG = Log.getLog(ResourceBundleValidationContextRenderer.class);

	public static final ResourceBundleValidationContextRenderer INSTANCE = new ResourceBundleValidationContextRenderer();

	private boolean containsKey(final ResourceBundle bundle, final String key)
	{
		for (final Enumeration<String> en = bundle.getKeys(); en.hasMoreElements();)
		{
			if (en.nextElement().equals(key)) return true;
		}
		return false;
	}

	/**
	 * {@inheritDoc}
	 */
	public String render(final OValContext ovalContext)
	{
		final String baseName;
		final String key;
		if (ovalContext instanceof ClassContext)
		{
			final ClassContext ctx = (ClassContext) ovalContext;
			baseName = ctx.getClazz().getName();
			key = "label.class";
		}
		else if (ovalContext instanceof FieldContext)
		{
			final FieldContext ctx = (FieldContext) ovalContext;
			baseName = ctx.getField().getDeclaringClass().getName();
			final String fieldName = ctx.getField().getName();
			key = "label.field." + fieldName;
		}
		else if (ovalContext instanceof ConstructorParameterContext)
		{
			final ConstructorParameterContext ctx = (ConstructorParameterContext) ovalContext;
			baseName = ctx.getConstructor().getDeclaringClass().getName();
			key = "label.parameter." + ctx.getParameterName();
		}
		else if (ovalContext instanceof MethodParameterContext)
		{
			final MethodParameterContext ctx = (MethodParameterContext) ovalContext;
			baseName = ctx.getMethod().getDeclaringClass().getName();
			key = "label.parameter." + ctx.getParameterName();
		}
		else if (ovalContext instanceof MethodEntryContext)
		{
			final MethodEntryContext ctx = (MethodEntryContext) ovalContext;
			baseName = ctx.getMethod().getDeclaringClass().getName();
			key = "label.method." + ctx.getMethod().getName();
		}
		else if (ovalContext instanceof MethodExitContext)
		{
			final MethodExitContext ctx = (MethodExitContext) ovalContext;
			baseName = ctx.getMethod().getDeclaringClass().getName();
			key = "label.method." + ctx.getMethod().getName();
		}
		else if (ovalContext instanceof MethodReturnValueContext)
		{
			final MethodReturnValueContext ctx = (MethodReturnValueContext) ovalContext;
			baseName = ctx.getMethod().getDeclaringClass().getName();
			key = "label.method." + ctx.getMethod().getName();
		}
		else
			return ovalContext.toString();

		try
		{
			final ResourceBundle bundle = ResourceBundle.getBundle(baseName);
			if (containsKey(bundle, key)) return bundle.getString(key);
			LOG.debug("Key {1} not found in bundle {2}", key, baseName);
		}
		catch (final MissingResourceException ex)
		{
			LOG.debug("Bundle {1} not found", baseName, ex);
		}
		return ovalContext.toString();
	}
}
