/**
 * Bachelorarbeit: Crowdsourcing- Systeme Manipulation und Resistenz
 * von Jens Bothur
 */
package jens.bothur.occt.automatons;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import jens.bothur.occt.automatons.security_framework.group_calculators.DeviationSpamScoreCalculator;
import jens.bothur.occt.automatons.security_framework.group_calculators.EarlyTimeFrameSpamScoreCalculator;
import jens.bothur.occt.automatons.security_framework.group_calculators.GroupSizeRatioSpamScoreCalculator;
import jens.bothur.occt.automatons.security_framework.group_calculators.GroupSizeSpamScoreCalculator;
import jens.bothur.occt.automatons.security_framework.group_calculators.GroupSupportCountSpamScoreCalculator;
import jens.bothur.occt.automatons.security_framework.group_calculators.TimeWindowSpamScoreCalculator;
import jens.bothur.occt.automatons.security_framework.single_calculators.EarlyDeviationCalculator;
import jens.bothur.occt.automatons.security_framework.single_calculators.GeneralDeviationCalculator;
import jens.bothur.occt.automatons.security_framework.single_calculators.HighAndLowRatingSpamScoreCalculator;
import jens.bothur.occt.automatons.security_framework.single_calculators.RatingSpamScoreCalculator;
import jens.bothur.occt.domainobjects.GroupOfRateableObjects;
import jens.bothur.occt.domainobjects.RateableObject;
import jens.bothur.occt.domainobjects.ReviewerGroup;
import jens.bothur.occt.domainobjects.User;
import jens.bothur.occt.domainvalues.ConfidenceLevel;
import jens.bothur.occt.services.IRateableObjectService;
import jens.bothur.occt.services.IUserService;

/**
 * Diese Klasse reprsentiert den Berechner fr die Vertrauenslevel der
 * Benutzer. Diese Klasse hat nur eine nach auen sichtbare Methode. Stt man
 * diese Methode an werden ALLE Vertrauenslevel von ALLEN Benutzern im System
 * mit dem aktuellen Datenstand neu berechnet.
 * 
 * Es wird standardmig von einem Review-Spammer Prozentsatz von 10%
 * ausgegangen. Dieser kann aber ber eine statische Methode gendert werden.
 * 
 * @author Jens Bothur
 */
public class ConfidenceLevelCalculator {

	/**
	 * Der aktuell angenommene Prozentsatz der Benutzer welche Review-Spammer
	 * sind.
	 */
	private static int _currentlyAssumedPercentageOfReviewSpammers = 10;

	/**
	 * Gibt den aktuell angenommene Prozentsatz der Benutzer welche
	 * Review-Spammer sind zurck.
	 * 
	 * @return der aktuell angenommene Prozentsatz der Benutzer welche
	 *         Review-Spammer sind .
	 */
	public static int getCurrentlyAssumedPercentageOfReviewSpammers() {
		return _currentlyAssumedPercentageOfReviewSpammers;
	}

	/**
	 * Setz den wert des aktuell angenommen Prozentsatz der Benutzer welche
	 * Review-Spamer sind auf einen bergebenen Wert.
	 * 
	 * @param newPercentage
	 *            Der neue Wert der fr den aktuell angenommen Prozentsatz der
	 *            Benutzer welche Review-Spammer sind gesetzt werden soll.
	 * 
	 * @require newPercentage <= 100;
	 * @reqzure newPercentage >= 0;
	 */
	public static void setNewAssumedPercentageOfReviewSpammers(int newPercentage) {
		assert newPercentage <= 100;
		assert newPercentage >= 0;
		_currentlyAssumedPercentageOfReviewSpammers = newPercentage;
	}

	/**
	 * Diese Variable gibt an ob fr einen Benutzer der maximale(
	 * <code>true</code>) oder der durchschnittliche(<code>false</code>)
	 * Gruppen-Spamming-Score benutzt werden soll.
	 */
	private static boolean _useMaximumGroupSpammingScore = true;

	/**
	 * Gibt an ob fr einen Benutzer der maximale( <code>true</code>) oder der
	 * durchschnittliche(<code>false</code>) Gruppen-Spamming-Score benutzt
	 * werden soll.
	 * 
	 * @return <code>true</code> wenn der maximale, <code>false</code> wenn der
	 *         durchschnittliche Gruppen-Spamming-Score zur Berechnung genutzt
	 *         werden soll.
	 */
	public static boolean useMaximumGroupSpammingScore() {
		return _useMaximumGroupSpammingScore;
	}

	/**
	 * Setz ob der maximale( <code>true</code>) oder der durchschnittliche(
	 * <code>false</code>) Gruppen-Spamming-Score benutzt werden soll.
	 * 
	 * @param useMaximumGroupSpamminScore
	 *            <code>true</code> damit der maximale, <code>false</code> damit
	 *            der durchschnittliche Gruppen-Spamming-Score zur Berechnung
	 *            genutzt wird.
	 */
	public static void setUseMaximumGroupSpammingScore(
			boolean useMaximumGroupSpamminScore) {
		_useMaximumGroupSpammingScore = useMaximumGroupSpamminScore;
	}

	/**
	 * Eine Sammlung mit allen bewertbaren Objekten des Systems.
	 */
	private Collection<RateableObject> _ratebaleObjects;

	/**
	 * Eine Sammlung mit allen Benutzern des Systems.
	 */
	private Collection<User> _users;

	/**
	 * Eine Menge mit allen Benutzergruppen des Systems.
	 */
	private Set<ReviewerGroup> _reviewerGroups;

	/**
	 * Eine Map mit allen Benutzern zu ihren bewerteten Objekten.
	 */
	private Map<User, Set<RateableObject>> _usersToRatedObjects;

	/**
	 * Eine Map mit allen Benutzern zu ihren jeweiligen Gruppen.
	 */
	private Map<User, Set<ReviewerGroup>> _usersToTheirGroups;

	/**
	 * Der zu benutzende {@link IUserService}.
	 */
	private IUserService _userService;

	/**
	 * Der zu benutzende {@link IRateableObjectService}.
	 */
	private IRateableObjectService _objectService;

	/**
	 * Der zu benutzende {@link IReviewerGroupDataMiner}.
	 */
	private IReviewerGroupDataMiner _raterGroupMiner;

	/**
	 * Der zu benutzende {@link IGroupOfRateableObjectsDataMiner}.
	 */
	private IGroupOfRateableObjectsDataMiner _objectGroupMiner;

	/**
	 * Standard-Konstruktor fr {@link ConfidenceLevelCalculator}.
	 * 
	 * @param userservice
	 *            Der zu benutzende {@link IUserService}.
	 * @param objectService
	 *            Der zu benutzende {@link IRateableObjectService}.
	 * @param groupminer
	 *            Der zu benutzende {@link IReviewerGroupDataMiner}.
	 */
	public ConfidenceLevelCalculator(IUserService userservice,
			IRateableObjectService objectService,
			IReviewerGroupDataMiner raterGroupminer,
			IGroupOfRateableObjectsDataMiner objectGroupMiner) {
		_userService = userservice;
		_objectService = objectService;
		_raterGroupMiner = raterGroupminer;
		_objectGroupMiner = objectGroupMiner;
	}

	/**
	 * Diese Methode veranlasst eine Neubrechnung aller Vertrauenslevel aller
	 * Benutzer im System.
	 */
	public void calculateConfidenceLevels() {

		_ratebaleObjects = _objectService.getAllRateableObjects();
		_users = _userService.getAllUsers();
		_reviewerGroups = _raterGroupMiner.getAllReviewerGroups();
		_usersToRatedObjects = mapUsersToRatedObjects();
		_usersToTheirGroups = mapUsersToTheirGroups();

		Map<User, Double> usersToSingleSpamScore = calculateUserSingleSpamScores();
		Map<ReviewerGroup, Double> groupsToGroupSpamScore = calculateGroupSpamScores();

		List<User> usersSpammerOrder = calculateUserSpammerOrder(
				usersToSingleSpamScore, groupsToGroupSpamScore);
		calculateAndSetNewConfidenceLevels(usersSpammerOrder);
	}

	/**
	 * Diese Methode berechnet anhand einer bergebenen Reihenfolge fr alle
	 * Benutzer ein neues Vertrauenslevel.
	 * 
	 * @param usersSpammerOrder
	 *            Eine {@link List} welche die Reihenfolge innerhalb der
	 *            {@link User} angibt nach der Wahrscheinlichkeit ob sie
	 *            Review-Spammer sind.
	 */
	private void calculateAndSetNewConfidenceLevels(List<User> usersSpammerOrder) {
		int userCount = usersSpammerOrder.size();
		int reviewSpammerCount = (userCount * _currentlyAssumedPercentageOfReviewSpammers) / 100;
		int normalUserCount = userCount - reviewSpammerCount;

		for (int i = 0; i < reviewSpammerCount; i++) {
			usersSpammerOrder.get(i).setConfidenceLevel(
					ConfidenceLevel.getConfidenceLevelByIntTimes10(0));
		}
		for (int i = reviewSpammerCount; i < userCount; i++) {
			double confLevel = ((i - reviewSpammerCount + 1.0) / (double) normalUserCount) * 10.0;
			int confLevelTimes10 = (int) (confLevel * 10.0);
			User user = usersSpammerOrder.get(i);
			user.setConfidenceLevel(ConfidenceLevel
					.getConfidenceLevelByIntTimes10(confLevelTimes10));

		}

	}

	/**
	 * Dieser Methode berechnet die Reihenfolge unter den Benutzern entsprechend
	 * der Wahrscheinlichkeit ob sie ein Review-Spammer sind oder nicht. Die
	 * Benutzer mit der grten Wahrscheinlichkeit sind vorne in der Liste zu
	 * finden.
	 * 
	 * @param usersToSingleSpamScore
	 *            Eine {@link Map} welche alle {@link User} zu ihren jeweiligen
	 *            einzelnen Spamming-Scores als {@link Double} zuordnet.
	 * @param groupsToGroupSpamScore
	 *            Eine {@link Map} welche alle {@link ReviewerGroup}s zu ihren
	 *            jeweiligen Gruppen-Spamming-Scores als {@link Double}
	 *            zuordnet.
	 * @return Eine {@link List} von {@link User} in der entsprechenden
	 *         Reihenfolge der Wahrscheinlichkeit ob sie Review-Spammer sind.
	 */
	private List<User> calculateUserSpammerOrder(
			Map<User, Double> usersToSingleSpamScore,
			Map<ReviewerGroup, Double> groupsToGroupSpamScore) {
		List<User> result = new ArrayList<User>();

		Map<User, Double> totalSpammingScores = new HashMap<User, Double>();
		for (User user : _users) {
			double singleSpamScore = usersToSingleSpamScore.get(user);
			Set<ReviewerGroup> usersGroups = _usersToTheirGroups.get(user);

			double totalSpammingScore = calculateTotalSpammingScore(
					singleSpamScore, usersGroups, groupsToGroupSpamScore);
			totalSpammingScores.put(user, totalSpammingScore);
		}

		// De-Comment this for debugging
		//
		// System.out.println("All users and their new total spamming score:");
		// System.out.println("---------------------------------------------");
		// for (Map.Entry<User,Double> entry : totalSpammingScores.entrySet()) {
		// System.out.println("User : " + entry.getKey() +
		// " with spamming score: " + entry.getValue());
		//
		// }
		// System.out.println("---------------------------------------------");

		for (int i = 0; i < _users.size(); i++) {
			double maximum = 0.0;
			User maximumHolder = null;

			for (Map.Entry<User, Double> entry : totalSpammingScores.entrySet()) {
				if (entry.getValue() >= maximum) {
					maximum = entry.getValue();
					maximumHolder = entry.getKey();
				}
			}
			totalSpammingScores.remove(maximumHolder);
			result.add(maximumHolder);
		}

		return result;
	}

	/**
	 * Diese Methode berechnet den kompletten Spamming-Score fr einen Benutzer
	 * aus einem bergebenen einzelnen Spamming-Score, einer Menge mit den
	 * Gruppen dieses Benutzers und einer Map welche allen Gruppen ihren
	 * Gruppen-Spamming-Score zuordnet.
	 * 
	 * @param singleSpamScore
	 *            Der einzelne Spamming-Score des Benutzers als {@link Double}.
	 * @param usersGroups
	 *            Ein {@link Set} mit allen {@link ReviewerGroup} des Benutzers.
	 * @param groupsToGroupSpamScore
	 *            Eine {@link Map} welche {@link ReviewerGroup}s ihrem
	 *            Gruppen-Spamming-Score als {@link Double} zuordnet.
	 * @return Den kompletten Spamming-Score des Benutzers als {@link Double}.
	 */
	private double calculateTotalSpammingScore(double singleSpamScore,
			Set<ReviewerGroup> usersGroups,
			Map<ReviewerGroup, Double> groupsToGroupSpamScore) {
		double groupSpamScore = 0.0;

		if (_useMaximumGroupSpammingScore) {
			for (ReviewerGroup reviewerGroup : usersGroups) {
				Double groupsSpamScore = groupsToGroupSpamScore
						.get(reviewerGroup);
				if (groupsSpamScore > groupSpamScore) {
					groupSpamScore = groupsSpamScore;
				}
			}
		} else {
			for (ReviewerGroup reviewerGroup : usersGroups) {
				groupSpamScore += groupsToGroupSpamScore.get(reviewerGroup);
			}
			groupSpamScore = groupSpamScore / usersGroups.size();
		}

		return 0.5 * (singleSpamScore + groupSpamScore);
	}

	/**
	 * Diese Methode berechnet den Gruppen-SpamScore eines jeden Benutzers. Es
	 * wird eine Map welche die Benutzer zu ihren jeweiligen Spam-Scores
	 * zuordnet zurck gegeben.
	 * 
	 * @return Eine {@link Map} welche {@link User} zu ihren jeweiligen
	 *         Gruppen-Spam-Scores als {@link Double} zuordnet.
	 */
	private Map<ReviewerGroup, Double> calculateGroupSpamScores() {
		Map<ReviewerGroup, Double> timeWindowSpamScore = TimeWindowSpamScoreCalculator
				.calculateTimeWindowSpamScore(_reviewerGroups);
		Map<ReviewerGroup, Double> earlyTimeFrameSpamScore = EarlyTimeFrameSpamScoreCalculator
				.calculateEarlyTimeFrameSpamScore(_reviewerGroups);
		Map<ReviewerGroup, Double> deviationSpamScore = DeviationSpamScoreCalculator
				.calculateDeviationSpamScore(_reviewerGroups);
		Map<ReviewerGroup, Double> groupSizeRatioSpamScore = GroupSizeRatioSpamScoreCalculator
				.calculateGroupSizeRatioSpamScore(_reviewerGroups);
		Map<ReviewerGroup, Double> groupSizeSpamScore = GroupSizeSpamScoreCalculator
				.calculateGroupSizeSpamScore(_reviewerGroups);
		Map<ReviewerGroup, Double> groupSupportCountSpamScore = GroupSupportCountSpamScoreCalculator
				.calculateGroupSupportCountSpamScore(_reviewerGroups);

		Map<ReviewerGroup, Double> result = new LinkedHashMap<ReviewerGroup, Double>();
		for (ReviewerGroup group : _reviewerGroups) {
			double score = (1 / 6)
					* (timeWindowSpamScore.get(group)
							+ earlyTimeFrameSpamScore.get(group)
							+ deviationSpamScore.get(group)
							+ groupSizeRatioSpamScore.get(group)
							+ groupSizeSpamScore.get(group) + groupSupportCountSpamScore
								.get(group));
			result.put(group, score);
		}

		return result;
	}

	/**
	 * Diese Methode berechnet den einzelnen Spam-Score eines jeden Benutzers.
	 * Es wird eine Map welche die Benutzer zu ihren jeweiligen Spam-Scores
	 * zuordnet zurck gegeben.
	 * 
	 * @return Eine {@link Map} welche {@link User} zu ihrem jeweiligen
	 *         SingleSpamScore als {@link Double} zuordnet.
	 */
	private Map<User, Double> calculateUserSingleSpamScores() {
		Set<GroupOfRateableObjects> allGroupsOfRateableObjects = _objectGroupMiner
				.getAllGroupsOfRateableObjects();

		Map<User, Double> ratingSpamScore = RatingSpamScoreCalculator
				.calculateRatingSpamScore(_users, allGroupsOfRateableObjects);
		Map<User, Double> highAndLowRatingScore = HighAndLowRatingSpamScoreCalculator
				.calculateHighAndLowRatingScore(_users,
						allGroupsOfRateableObjects);
		Map<User, Double> generalDeviation = GeneralDeviationCalculator
				.calculateGeneralDeviation(_usersToRatedObjects);
		Map<User, Double> earlyDeviation = EarlyDeviationCalculator
				.calculateEarlyDeviation(_usersToRatedObjects);

		Map<User, Double> result = new LinkedHashMap<User, Double>();
		for (User user : _users) {
			double singleSpamScore = 0.5 * ratingSpamScore.get(user) + 0.25
					* highAndLowRatingScore.get(user) + 0.125
					+ generalDeviation.get(user) + 0.125
					* earlyDeviation.get(user);
			result.put(user, singleSpamScore);
		}

		return result;
	}

	/**
	 * Diese Methode berechnet die Map welche Benutzer zu einer Menge ihrer
	 * jeweiligen Benutzer-Gruppen zuordnet.
	 * 
	 * @return Eine {@link Map} von {@link User}n zu einem {@link Set} ihrer
	 *         jeweiligen {@link ReviewerGroup}s.
	 */
	private Map<User, Set<ReviewerGroup>> mapUsersToTheirGroups() {
		Map<User, Set<ReviewerGroup>> result = new HashMap<User, Set<ReviewerGroup>>();

		for (User user : _users) {
			result.put(user, new LinkedHashSet<ReviewerGroup>());
		}

		for (ReviewerGroup group : _reviewerGroups) {
			for (User user : group.getMembers()) {
				result.get(user).add(group);
			}
		}

		return result;
	}

	/**
	 * Diese Methode berechnet die Map welche Benutzer zu einer Menge ihrer
	 * jeweiligen bewerteten Objekten zuordnet.
	 * 
	 * @return Eine {@link Map} von {@link User}n zu einem {@link Set} ihrer
	 *         jeweiligen bewerten {@link RateableObject}.
	 */
	private Map<User, Set<RateableObject>> mapUsersToRatedObjects() {
		Map<User, Set<RateableObject>> result = new HashMap<User, Set<RateableObject>>();

		for (User user : _users) {
			result.put(user, new LinkedHashSet<RateableObject>());
		}

		for (RateableObject object : _ratebaleObjects) {
			for (User user : object.getReviewers()) {
				result.get(user).add(object);
			}

		}

		return result;
	}

}
