/**
 * Bachelorarbeit: Crowdsourcing- Systeme Manipulation und Resistenz
 * von Jens Bothur
 */
package jens.bothur.occt.automatons.security_framework.group_calculators;

import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

import jens.bothur.occt.domainobjects.RateableObject;
import jens.bothur.occt.domainobjects.ReviewerGroup;
import jens.bothur.occt.domainobjects.User;
import jens.bothur.occt.domainvalues.AverageReview;
import jens.bothur.occt.domainvalues.Review;

/**
 * Diese Klasse reprsentiert den Berechner zur Berechnung des SpamScore
 * bezglich der Gruppen-Abweichung. Hierzu wird die durchschnittliche Bewertung
 * einer Gruppe fr ein Objekt mit der durchschnittlichen Bewertung aller
 * anderen Bewerter in Verhltnis gesetzt. Zur Bestimmung des Scores wird dann
 * das schlechteste Gruppen Verhalten benutzt.
 * 
 * @author Jens Bothur
 */
public class DeviationSpamScoreCalculator implements IGroupSpamScoreCalculator {

	/**
	 * Berechnet den Gruppen-Abweichungs-SpamScore fr eine Menge von
	 * Bewerter-Gruppen.
	 * 
	 * @param reviewerGroups
	 *            Ein {@link Set} von {@link ReviewerGroup}
	 * @return Eine {@link Map} welche die reviewerGroups ihren berechneten
	 *         Scores zuordnet.
	 */
	public static Map<ReviewerGroup, Double> calculateDeviationSpamScore(
			Set<ReviewerGroup> reviewerGroups) {
		Map<ReviewerGroup, Double> result = new LinkedHashMap<ReviewerGroup, Double>();

		for (ReviewerGroup reviewerGroup : reviewerGroups) {
			result.put(reviewerGroup,
					calculateDeviationSpamScoreForGroup(reviewerGroup));
		}

		return result;
	}

	/**
	 * Berechnet die Gruppen-Abweichung fr eine bergebene Grupp.
	 * 
	 * @param reviewerGroup
	 *            Die {@link ReviewerGroup} fr die berechnet werden soll.
	 * @return Die Gruppen-Abweichung als double.
	 */
	private static Double calculateDeviationSpamScoreForGroup(
			ReviewerGroup reviewerGroup) {
		double result = 0.0;

		for (RateableObject rateableObject : reviewerGroup
				.getCollaboratedObjects()) {
			double objectScore = calculateDeviationSpamScoreForGroupAndObject(
					reviewerGroup, rateableObject);
			if (objectScore > result) {
				result = objectScore;
			}

		}

		return result;
	}

	/**
	 * Berechnet die Gruppen-Abweichung fr eine bergebene Gruppe bezglich
	 * eines bewertbaren Objektes.
	 * 
	 * @param reviewerGroup
	 *            Die {@link ReviewerGroup} fr die berechnet werden soll.
	 * @param rateableObject
	 *            Das {@link RateableObject} fr das berechnet werden soll.
	 * @return Der berechnete Score als double.
	 */
	private static double calculateDeviationSpamScoreForGroupAndObject(
			ReviewerGroup reviewerGroup, RateableObject rateableObject) {
		Set<User> members = reviewerGroup.getMembers();
		Set<User> reviewers = rateableObject.getReviewers();
		Set<User> reviewersOtherThanMembers = new LinkedHashSet<User>();

		for (User user : reviewers) {
			if (!members.contains(user)) {
				reviewersOtherThanMembers.add(user);
			}
		}

		AverageReview membersAverageRating = calculateAverageRating(
				rateableObject, members);
		AverageReview notMembersAverageRating = calculateAverageRating(
				rateableObject, reviewersOtherThanMembers);

		double result = Math.abs(membersAverageRating.getNormalizedRating()
				- notMembersAverageRating.getNormalizedRating());
		return result;
	}

	/**
	 * Berechnet die durchschnittliche Bewertung von einer Menge von Bewertern
	 * an einem Objekt.
	 * 
	 * @param rateableObject
	 *            Das {@link RateableObject} welches die raters bewertet haben.
	 * @param raters
	 *            Ein {@link Set} von {@link User} welche rateableObject
	 *            bewertet haben.
	 * @return Das berechnete {@link AverageReview}.
	 */
	private static AverageReview calculateAverageRating(
			RateableObject rateableObject, Set<User> raters) {
		int sumOfWeights = 0;
		int sumOfRatings = 0;

		AverageReview result = AverageReview.getAverageReviewByIntTimes10(30);

		if (!raters.isEmpty()) {
			for (User user : raters) {
				int confidenceLevelValue = user.getConfidenceLevel()
						.getValueTimes10();
				Review review = rateableObject.getReviewForReviewer(user);
				sumOfWeights += confidenceLevelValue;
				sumOfRatings += confidenceLevelValue * review.getRating();
			}

			sumOfRatings = sumOfRatings * 10;
			int averageRatingTimes10 = 35;
			if (sumOfWeights != 0) {
				averageRatingTimes10 = sumOfRatings / sumOfWeights;
			}

			result = AverageReview
					.getAverageReviewByIntTimes10(averageRatingTimes10);
		}

		return result;
	}

}
