/**
 * Bachelorarbeit: Crowdsourcing- Systeme Manipulation und Resistenz
 * von Jens Bothur
 */
package jens.bothur.occt.automatons.security_framework.single_calculators;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import jens.bothur.occt.domainobjects.RateableObject;
import jens.bothur.occt.domainobjects.User;

/**
 * Diese Klasse reprsentiert einen Berechner fr die Frhe Abweichung eines
 * Benutzers und den daraus resultierenden Spamming-Score.
 * 
 * @author Jens Bothur
 */
public class EarlyDeviationCalculator implements ISingleSpamScoreCalculator {

	/**
	 * Berechnet die Frhe Abweichung fr eine Menge von Benutzern. Diese Nutzer
	 * werden in einer Map bergeben in der sie jeweils zu einer Menge ihrer
	 * bewerteten Objekte zugeordnet werden.
	 * 
	 * @param usersToRatedObjects
	 *            Eine {@link Map} in der die {@link User} einem {@link Set}
	 *            ihrer jeweiligen {@link RateableObject} zugeordnet werden
	 * 
	 * @return Eine {@link Map} in der die obigen {@link User} ihrer berechneten
	 *         Frhe Abweichung zugeordnet werden.
	 */
	public static Map<User, Double> calculateEarlyDeviation(
			Map<User, Set<RateableObject>> usersToRatedObjects) {
		Map<User, Double> result = new HashMap<User, Double>();

		for (User user : usersToRatedObjects.keySet()) {
			double earlyDeviationForUser = calculateEarlyDeviationForUser(user,
					usersToRatedObjects.get(user));
			result.put(user, earlyDeviationForUser);
		}

		return result;
	}

	/**
	 * Berechnet die Frhe Abweichung eines Benutzers und eine Menge aller von
	 * diesem Benutzer abgegebene Bewertungen.
	 * 
	 * @param id
	 *            Der {@link User}.
	 * @param set
	 *            Ein {@link Set} von {@link RateableObject} die der Benutzer
	 *            bewertet hat.
	 * @return Die berechnete Frhe Abweichung im Intervall [0;1] als
	 *         {@link Double}.
	 */
	private static double calculateEarlyDeviationForUser(User user,
			Set<RateableObject> set) {
		double result = 0.0;

		if (!set.isEmpty()) {
			for (RateableObject rateableObject : set) {
				double divergence = Math.abs(rateableObject
						.getReviewForReviewer(user).getNormalizedRating()
						- rateableObject.getAverageReview()
								.getNormalizedRating());
				double weigth = 1.0 / Math.pow(rank(user, rateableObject), 1.1);
				result += divergence * weigth;
			}
			result = result / set.size();
		}

		return result;
	}

	/**
	 * Diese Methode gibt den Rang einer Bewertung eines Benutzer zurck. Dies
	 * bedeutet die wievielste sie unter allen Bewertung fr das entsprechende
	 * bewertbare Objekt war.
	 * 
	 * @param user
	 *            Der bewertende {@link User}.
	 * @param rateableObject
	 *            Ein {@link RateableObject} das von dem Benutzer bewertet wurde
	 * @return Der Rank der Bewertung durch den Benutzer.
	 */
	private static double rank(User user, RateableObject rateableObject) {
		Date ratersTime = rateableObject.getDateForReviewer(user);
		List<Date> ratingDates = new ArrayList<Date>(
				rateableObject.getRatingDates());
		Collections.sort(ratingDates);
		return ratingDates.indexOf(ratersTime) + 1;
	}
}
