/**
 * Bachelorarbeit: Crowdsourcing- Systeme Manipulation und Resistenz
 * von Jens Bothur
 */
package jens.bothur.occt.automatons.security_framework.single_calculators;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import jens.bothur.occt.domainobjects.GroupOfRateableObjects;
import jens.bothur.occt.domainobjects.RateableObject;
import jens.bothur.occt.domainobjects.User;
import jens.bothur.occt.domainvalues.Review;

/**
 * Diese Klasse reprsentiert einen Hoch-und-Niedrig-Spam-Score Berechner.
 * Hierzu werden alle Objektgruppen untersucht, ob einer Benutzer hier mehrere
 * Objekte innerhalb eines Zeitfensters von {@value #TIME_WINDOW_LENGTH} Stunden
 * entweder sehr hoch ({@value #HIGH_REVIEWS}) oder sehr tief (
 * {@value #LOW_REVIEWS}) bewertet.
 * 
 * Ist dies der Fall werden alle Hoch- und Niedrig-Bewertungen dieses Benutzers
 * aggregiert udn die Anzahl dieser Bewertungen wird innerhalb aller Benutzer in
 * Relation gestellt.
 * 
 * @author Jens Bothur
 */
public class HighAndLowRatingSpamScoreCalculator implements
		ISingleSpamScoreCalculator {

	/**
	 * Die grte des Zeitfensters in Stunden.
	 */
	private static final int TIME_WINDOW_LENGTH;

	/**
	 * Eine Menge der Bewertungen, welche als hoch angesehen werden.
	 */
	private static final Set<Review> HIGH_REVIEWS;

	/**
	 * Eine Menge der Bewertungen, welche als niedrig angesehen werden.
	 */
	private static final Set<Review> LOW_REVIEWS;

	/**
	 * Der static-Block zur Initialisierung der obigen Variablen.
	 */
	static {
		TIME_WINDOW_LENGTH = 1;

		HIGH_REVIEWS = new LinkedHashSet<Review>();
		HIGH_REVIEWS.add(Review.FIVE_STAR_REVIEW);

		LOW_REVIEWS = new LinkedHashSet<Review>();
		LOW_REVIEWS.add(Review.ONE_STAR_REVIEW);
	}

	/**
	 * Eine Map in der der aktuelle Stand von hohen Bewertungen hinterlegt wird.
	 */
	private static Map<User, Integer> _highReviewCounts;

	/**
	 * Eine Map in der der aktuelle Stand von hohen Bewertungen hinterlegt wird.
	 */
	private static Map<User, Integer> _lowReviewCounts;

	/**
	 * Diese Methode berechnet den Hoch-Und-Niedrig-Bewertungs-Spam-Score fr
	 * alle Benutzer. Sie bentigt dazu eien Sammlung aller Benutzer und eine
	 * Menge von allen Objektgruppen. Das Ergebnis wird dann in einer Map
	 * geliefert, welche jeden Benutzer seinen Score zuordnet.
	 * 
	 * @param users
	 *            Eine {@link Collection} von allen {@link User}n des Systems.
	 * @param allGroupsOfRateableObjects
	 *            Ein {@link Set} von allen {@link GroupOfRateableObjects} des
	 *            Systems.
	 * @return Eine {@link Map}, welche allen Benutzer ihren jeweiligen Score
	 *         als double zuordnet.
	 */
	public static Map<User, Double> calculateHighAndLowRatingScore(
			Collection<User> users,
			Set<GroupOfRateableObjects> allGroupsOfRateableObjects) {
		_highReviewCounts = new LinkedHashMap<User, Integer>();
		_lowReviewCounts = new LinkedHashMap<User, Integer>();
		// zwischen Speicher intialisieren
		for (User user : users) {
			_highReviewCounts.put(user, 0);
			_lowReviewCounts.put(user, 0);
		}
		// Anzahl der hohen und niedrigen Bewertungen berechnen und
		// zwischenspeichern.
		for (GroupOfRateableObjects groupOfRateableObjects : allGroupsOfRateableObjects) {
			calculateAndInduceObjectSpecificHighAndLowReviewCounts(groupOfRateableObjects);
		}
		// Maximum von hohen und niedrigen Bewertungen ber alle Benutzer
		// bestimmten.
		int maximumHighReviewCount = 1;
		int maximumLowReviewCount = 1;
		for (User user : users) {
			int highReviewCount = _highReviewCounts.get(user);
			if (highReviewCount > maximumHighReviewCount) {
				maximumHighReviewCount = highReviewCount;
			}
			int lowReviewCount = _lowReviewCounts.get(user);
			if (lowReviewCount > maximumLowReviewCount) {
				maximumLowReviewCount = lowReviewCount;
			}
		}
		// Ergebnis Map anlegen und Scores fr alle Benutzer berechnen.
		Map<User, Double> result = new LinkedHashMap<User, Double>();
		for (User user : users) {
			double highRatingScore = (double) _highReviewCounts.get(user)
					/ (double) maximumHighReviewCount;
			double lowRatingScore = (double) _lowReviewCounts.get(user)
					/ (double) maximumLowReviewCount;
			double highAndLowRatingScore = (highRatingScore + lowRatingScore) * 0.5;
			result.put(user, highAndLowRatingScore);
		}

		return result;
	}

	/**
	 * Diese Methode berechnet fr eine bergene Objektgruppe alle Hoch- und
	 * Niedrig-Bewertungsanzahlen innerhalb des Zeitfensters fr alle Bewerter
	 * dieser Gruppe. Die Anzahlen werden in die Zwishcenspeicher eingefgt.
	 * 
	 * @param groupOfRateableObjects
	 *            Eine {@link GroupOfRateableObjects} fr die die Hoch- und
	 *            Niedrig-Bewertungsanzahlen bestimmt werden sollen.
	 */
	private static void calculateAndInduceObjectSpecificHighAndLowReviewCounts(
			GroupOfRateableObjects groupOfRateableObjects) {
		Set<User> raters = groupOfRateableObjects.getRaters();

		for (User rater : raters) {
			int highReviewCount = gatherReviewsCount(rater,
					groupOfRateableObjects, HIGH_REVIEWS, 3);
			_highReviewCounts.put(rater, _highReviewCounts.get(rater)
					+ highReviewCount);
			int lowReviewCount = gatherReviewsCount(rater,
					groupOfRateableObjects, LOW_REVIEWS, 2);
			_lowReviewCounts.put(rater, _lowReviewCounts.get(rater)
					+ lowReviewCount);
		}

	}

	/**
	 * Diese Methode bestimmt die Anzahl von Spammer-Bewertungen , welche ein
	 * bergebener Benutzer fr eine bergebene Objektgruppe abgegeben hat. Eine
	 * mindest Anzahl an Bewertungen mssen innerhalb des Zeitfenster liegen und
	 * mssen Teil einer bergebenen Menge von zulssigen Bewertungen liegen,
	 * damit sie gezhlt werden.
	 * 
	 * @param rater
	 *            Ein {@link User} welche die groupOfRateableObjects bewertet
	 *            hat.
	 * @param groupOfRateableObjects
	 *            Eine {@link GroupOfRateableObjects} die der rater Bewerter
	 *            hat.
	 * @param reviews
	 *            Ein {@link Set} von {@link Review}, welche angibt, welche
	 *            Bewertungen zugelassen sind. (Hoch oder Tief)
	 * @param minimumCount
	 *            Die mindest Anzahl von Bewertungen innerhalb eines
	 *            Zeitfenster.
	 * @return Die Anzahl der Bewertungen des rates and der
	 *         groupOfRateableObjects, welche die Bedingungen erfllen.
	 */
	private static int gatherReviewsCount(User rater,
			GroupOfRateableObjects groupOfRateableObjects, Set<Review> reviews,
			int minimumCount) {
		Set<RateableObject> objects = groupOfRateableObjects.getMembers();
		List<Date> ratedTimes = new ArrayList<Date>();
		for (RateableObject object : objects) {
			Review reviewForReviewerID = object.getReviewForReviewer(rater);
			if (reviewForReviewerID != null
					&& reviews.contains(reviewForReviewerID)) {
				ratedTimes.add(object.getDateForReviewer(rater));
			}
		}
		Collections.sort(ratedTimes);

		Set<Date> allSpammingTimes = new LinkedHashSet<Date>();
		if (ratedTimes.size() >= minimumCount) {
			for (Date date : ratedTimes) {
				Set<Date> spammingTimes = findAllTimesWithinTimeWindow(date,
						ratedTimes);
				if (spammingTimes.size() >= minimumCount) {
					allSpammingTimes.addAll(spammingTimes);
				}

			}
		}
		return allSpammingTimes.size();
	}

	/**
	 * Findet alle Zeiten einer bergebenen Menge, welche innerhalb des
	 * Zeitfensters liegen nach einer bergebenen Zeit liegen.
	 * 
	 * @param date
	 *            Ein {@link Date}. Der Startzeitpunkt.
	 * @param ratedTimes
	 *            Ein {@link Set} von {@link Date}s. Hierin werden Ergebnisse
	 *            gesucht.
	 * @return Ein {@link Set} von {@link Date}s welche innerhalb des
	 *         Zeitfensters nach date liegen.
	 */
	private static Set<Date> findAllTimesWithinTimeWindow(Date date,
			List<Date> ratedTimes) {
		int index = ratedTimes.indexOf(date);
		Set<Date> result = new LinkedHashSet<Date>();
		result.add(date);
		while (index < ratedTimes.size() - 1) {
			index++;
			Date nextDate = ratedTimes.get(index);
			if (!datesAreWithinTimeWindow(date, nextDate)) {
				break;
			}
			result.add(nextDate);
		}
		return result;
	}

	/**
	 * berprft ob zwei bergebene Zeiten innerhalb eines Zeitfensters liegen.
	 * 
	 * @param from
	 *            Ein {@link Date}. Der Startzeitpunkt.
	 * @param until
	 *            Ein {@link Date} fr das berpft werden soll, ob es im
	 *            Zeitfenster nach from liegt.
	 * @return <code>true</code> wenn until im Zeitfenster nach from liegt,
	 *         <code>false</code> wenn nicht.
	 */
	private static boolean datesAreWithinTimeWindow(Date from, Date until) {
		long hour = 60 * 60 * 1000;
		long timeWindow = TIME_WINDOW_LENGTH * hour;

		return (until.getTime() - from.getTime()) <= timeWindow;
	}

}
