/**
 * Bachelorarbeit: Crowdsourcing- Systeme Manipulation und Resistenz
 * von Jens Bothur
 */
package jens.bothur.occt.domainobjects;

import java.util.LinkedHashSet;
import java.util.Set;

import jens.bothur.occt.domainvalues.Review;

/**
 * Diese Klasse reprsentiert eine Gruppe von bewertbaren Objekten. Diese Gruppe
 * reprsentiert Objekte welche eine gemeinsame Eigenscahft habe. Im Falle des
 * OCCT knnten dies zum Beispiel sich wiederholende Termine sein. Sie muss
 * mindestens eine Gre von {@link #MINIMUM_SIZE} haben.
 * 
 * @author Jens Bothur
 */
public class GroupOfRateableObjects {

	/**
	 * Die globale minimale Gre einer Gruppe von bewertbaren Objekten
	 */
	public final int MINIMUM_SIZE = 2;

	/**
	 * Eine Menge mit allen Mitgliedern dieser Gruppe.
	 */
	private final Set<RateableObject> _members;

	/**
	 * Eine Menge von {@link User}n welche diese Objektgruppe bewertet haben.
	 * Ein Benutzer muss mindestens {@link #MINIMUM_SIZE} Objekte in dieser
	 * Gruppe bewertet haben.
	 */
	private final Set<User> _raters;

	/**
	 * Konstruktor fr Gruppenb von bewertbaren Objekten. Die Mitglieder dieser
	 * Gruppe und die Bewerter dieser Gruppe werden bergeben.
	 * 
	 * @param members
	 *            Ein {@link Set} von {@link RateableObject}, alle Mitglieder
	 *            dieser Gruppe.
	 * @param raters
	 *            Ein {@link Set} von {@link User}n, die Bewerter dieser Gruppe.
	 * 
	 * @require Jeder Bewerter muss mindestens {@link #MINIMUM_SIZE} Objekte
	 *          dieser Gruppe bewertet haben.
	 * @require members.size() >= {@link #MINIMUM_SIZE}.
	 */
	public GroupOfRateableObjects(Set<RateableObject> members, Set<User> raters) {
		assert members.size() >= MINIMUM_SIZE;
		for (User user : raters) {
			int counter = 0;
			for (RateableObject rateableObject : members) {
				Review reviewForReviewerID = rateableObject
						.getReviewForReviewer(user);
				if (reviewForReviewerID != null) {
					counter++;
				}
			}
			assert counter >= MINIMUM_SIZE;
		}
		_members = members;
		_raters = raters;
	}

	/**
	 * Gibt ein {@link Set} von {@link RateableObject}, alle Mitglieder dieser
	 * Gruppe, zurck.
	 * 
	 * @return Ein {@link Set} von {@link RateableObject}, alle Mitglieder
	 *         dieser Gruppe.
	 */
	public Set<RateableObject> getMembers() {
		return new LinkedHashSet<RateableObject>(_members);
	}

	/**
	 * Gibt ein {@link Set} von {@link User}n, die Bewerter dieser Gruppe,
	 * zurck
	 * 
	 * @return Ein {@link Set} von {@link User}n, die Bewerter dieser Gruppe.
	 */
	public Set<User> getRaters() {
		return new LinkedHashSet<User>(_raters);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result
				+ ((_members == null) ? 0 : _members.hashCode());
		result = prime * result + ((_raters == null) ? 0 : _raters.hashCode());
		return result;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		GroupOfRateableObjects other = (GroupOfRateableObjects) obj;
		if (!_members.equals(other._members))
			return false;
		if (!_raters.equals(other._raters))
			return false;
		return true;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return "GroupOfRateableObjects [_members=" + _members + ", _raters="
				+ _raters + "]";
	}

}
