/**
 * Bachelorarbeit: Crowdsourcing- Systeme Manipulation und Resistenz
 * von Jens Bothur
 */
package jens.bothur.occt.automatons;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

import java.util.ArrayList;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;

import jens.bothur.occt.domainobjects.GroupOfRateableObjects;
import jens.bothur.occt.domainobjects.RateableObject;
import jens.bothur.occt.domainobjects.ReviewerGroup;
import jens.bothur.occt.domainobjects.User;
import jens.bothur.occt.domainvalues.Review;
import jens.bothur.occt.services.DummyRateableObjectService;
import jens.bothur.occt.services.DummyUserService;
import jens.bothur.occt.services.IRateableObjectService;
import jens.bothur.occt.services.IUserService;

import org.junit.Before;
import org.junit.Test;

/**
 * Test fr die Klasse: {@link ConfidenceLevelCalculator}. Dieser Test ist sehr
 * umfangreich und testet das komplette Sicherheits-Framework des OCCTs.
 * 
 * @author Jens Bothur
 */
public class ConfidenceLevelCalculatorTest {

	private ConfidenceLevelCalculator _instance;
	private AverageRatingCalculator _averageRatingCalc;

	private IUserService _userService;
	private IRateableObjectService _objectService;
	private IReviewerGroupDataMiner _raterGroupMiner;
	private IGroupOfRateableObjectsDataMiner _objectGroupMiner;

	private List<User> _normalUsers;
	private List<User> _reviewSpammers;
	private ArrayList<User> _users;

	private ArrayList<RateableObject> _objects;

	private Review[] _reviews = { Review.ONE_STAR_REVIEW,
			Review.TWO_STAR_REVIEW, Review.THREE_STAR_REVIEW,
			Review.FOUR_STAR_REVIEW, Review.FIVE_STAR_REVIEW };

	private Random _random;

	/**
	 * @throws java.lang.Exception
	 */
	@Before
	public void setUp() throws Exception {
		_random = new Random(1);
		for (int i = 0; i < 10; i++) {
			for (int j = 0; j < _random.nextInt(100); j++) {
				_random.nextBoolean();
			}

		}

		_userService = new DummyUserService();
		_normalUsers = new ArrayList<User>();
		_reviewSpammers = new ArrayList<User>();
		setUpUsers();
		_users = new ArrayList<User>(_userService.getAllUsers());

		_objectService = new DummyRateableObjectService();
		for (int i = 0; i < 200; i++) {
			_objectService.registerRateableObject(new RateableObject(
					getRandomDate()));
		}
		_objects = new ArrayList<RateableObject>(
				_objectService.getAllRateableObjects());

		Set<ReviewerGroup> groups = setUpSpammerGroups();
		groups.addAll(setUpNonSpammerGroups());
		Set<GroupOfRateableObjects> objectGroups = setUpObjectGroups();

		_raterGroupMiner = new TestRaterGroupMinerForConfidenceLevelCalculatorTest(
				groups);
		_objectGroupMiner = new TestObjectGroupMinerForConfidenceLevelCalculatorTest(
				objectGroups);

		setUpSpamming();
		setUpNormalRating();

		_averageRatingCalc = new AverageRatingCalculator(_objectService);
		_instance = new ConfidenceLevelCalculator(_userService, _objectService,
				_raterGroupMiner, _objectGroupMiner);
	}

	/**
	 * Verpasst den Spammern bis zu 20 spammende Bewertungen und 10 normale
	 * Bewertungen.
	 */
	private void setUpSpamming() {
		// early deviation spammings
		int oneDayInMilliSecs = 24 * 60 * 60 * 1000;
		for (User spammer : _reviewSpammers) {
			int count = getNormalDistributedInteger(10);
			for (int i = 0; i < count; i++) {
				RateableObject object = getRandomListElement(_objects);
				Review review = Review.ONE_STAR_REVIEW;
				if (_random.nextBoolean()) {
					review = Review.FIVE_STAR_REVIEW;
				}
				int timeAfterFirstPossibleRating = _random
						.nextInt(oneDayInMilliSecs) + 1;
				long timeOfRating = object.getEarliestPossibleReviewDate()
						.getTime() + timeAfterFirstPossibleRating;
				Date date = new Date(timeOfRating);
				if (!object.rate(spammer, review, date)) {
					i--;
				}
			}
		}
		// deviation spammings
		for (User spammer : _reviewSpammers) {
			int count = getNormalDistributedInteger(10);
			for (int i = 0; i < count; i++) {
				RateableObject object = getRandomListElement(_objects);
				Review review = Review.ONE_STAR_REVIEW;
				if (_random.nextBoolean()) {
					review = Review.FIVE_STAR_REVIEW;
				}

				Date date = getRandomDateAfter(object
						.getEarliestPossibleReviewDate());
				if (!object.rate(spammer, review, date)) {
					i--;
				}
			}
		}
		// normal ratings
		for (User spammer : _reviewSpammers) {
			int count = getNormalDistributedInteger(10);
			for (int i = 0; i < count; i++) {
				RateableObject object = getRandomListElement(_objects);
				Review review = _reviews[_random.nextInt(5)];
				Date date = getRandomDateAfter(object
						.getEarliestPossibleReviewDate());
				if (!object.rate(spammer, review, date)) {
					i--;
				}
			}
		}
	}

	/**
	 * Verpasst allen normalen User bis zu 30 normalen Ratings.
	 */
	private void setUpNormalRating() {
		for (User spammer : _normalUsers) {
			int count = getNormalDistributedInteger(30);
			for (int i = 0; i < count; i++) {
				RateableObject object = getRandomListElement(_objects);
				Review review = _reviews[_random.nextInt(5)];
				Date date = getRandomDateAfter(object
						.getEarliestPossibleReviewDate());
				if (!object.rate(spammer, review, date)) {
					i--;
				}
			}
		}

	}

	private int getNormalDistributedInteger(int upperLevel) {
		double gauss = _random.nextGaussian();
		gauss = gauss / 4;
		gauss = gauss + 0.5;
		if (gauss > 1) {
			gauss = 1;
		} else if (gauss < 0) {
			gauss = 0;
		}
		return (int) (gauss * upperLevel);
	}

	/**
	 * Erstellt die Objektgruppen. Insgesamt 20 Stck mit 2 bis 10 Mitgliedern.
	 * Jedes Gruppe erhlt 0-3 Spammer und 0-6 normale Rater. Normale Rater
	 * haben eine zufllige Anzahl an Objekten zufllig bewertet. Spammer haben
	 * alle Objekte mit dem gleichen zufligen rating bewertet.
	 * 
	 * @return Ein {@link Set} mit {@link GroupOfRateableObjects}.
	 */
	private Set<GroupOfRateableObjects> setUpObjectGroups() {
		Set<GroupOfRateableObjects> result = new LinkedHashSet<GroupOfRateableObjects>();
		// objekt gruppen erstellen
		List<RateableObject> objects = new ArrayList<RateableObject>(_objects);
		List<Set<RateableObject>> objectGroups = new ArrayList<Set<RateableObject>>();
		for (int i = 0; i < 20; i++) {
			int numberOfMembers = _random.nextInt(8) + 2;
			Set<RateableObject> objectGroup = new LinkedHashSet<RateableObject>();
			for (int j = 0; j < numberOfMembers; j++) {
				RateableObject object = getRandomListElement(objects);
				if (objectGroup.add(object)) {
					objects.remove(object);
				} else {
					j--;
				}
			}
			if (!objectGroups.contains(objectGroup)) {
				objectGroups.add(objectGroup);
			} else {
				i--;
			}
		}
		// rate fr objekt gruppen finden, ratings aufhren und finale gruppen
		// erstellen
		for (Set<RateableObject> objectGroup : objectGroups) {
			Set<User> raters = new LinkedHashSet<User>();
			// normale Rater hinzufgen
			int numberOfNormalRaters = _random.nextInt(7);
			// ber die anzahl der normalen rater iterieren
			for (int i = 0; i < numberOfNormalRaters; i++) {
				User rater = getRandomListElement(_normalUsers);
				if (raters.add(rater)) {
					int numberOfRatedObjects = _random.nextInt(objectGroup
							.size());
					numberOfRatedObjects = Math.max(numberOfRatedObjects, 2);
					List<RateableObject> objectGroupAsList = new ArrayList<RateableObject>(
							objectGroup);
					// ber die anzahl der zu bewertenden objekte iterieren
					for (int j = 0; j < numberOfRatedObjects; j++) {
						RateableObject object = getRandomListElement(objectGroupAsList);
						objectGroupAsList.remove(object);
						Review review = _reviews[_random.nextInt(5)];
						Date date = getRandomDateAfter(object
								.getEarliestPossibleReviewDate());
						object.rate(rater, review, date);
					}

				} else {
					i--;
				}
			}
			// Spammer hinzufgen
			int spammerCount = _random.nextInt(4);
			// ber die anzahl der Spammer iterieren
			for (int i = 0; i < spammerCount; i++) {
				User spammer = getRandomListElement(_reviewSpammers);
				if (raters.add(spammer)) {
					Review review = _reviews[_random.nextInt(5)];
					for (RateableObject rateableObject : objectGroup) {
						Date date = getRandomDateAfter(rateableObject
								.getEarliestPossibleReviewDate());
						rateableObject.rate(spammer, review, date);
					}
				} else {
					i--;
				}
			}
			// GroupOfRateableObjects erzeugen und hinzufgen
			result.add(new GroupOfRateableObjects(objectGroup, raters));
		}

		return result;
	}

	/**
	 * Erzeugt 20 nicht spammer gruppen, mit den gleichen eigentschaften wie die
	 * spammer gruppen. Nur die vergebene Bewertung und der Zeitpunkt ndert
	 * sich.
	 * 
	 * @return Ein {@link Set} mit {@link ReviewerGroup} welche aber nicht
	 *         erkannt werden sollten.
	 */
	private Set<ReviewerGroup> setUpNonSpammerGroups() {
		Set<ReviewerGroup> result = new LinkedHashSet<ReviewerGroup>();

		for (int i = 0; i < 20; i++) {
			// find group members
			int memberCount = _random.nextInt(4) + 2;
			Set<User> members = new LinkedHashSet<User>();
			for (int j = 0; j < memberCount; j++) {
				User member = getRandomListElement(_normalUsers);
				if (!members.add(member)) {
					j--;
				}
			}
			// find group objects
			int objectCount = _random.nextInt(8) + 3;
			Set<RateableObject> collaboratedObjects = new LinkedHashSet<RateableObject>();
			for (int j = 0; j < objectCount; j++) {
				RateableObject object = getRandomListElement(_objects);
				if (!collaboratedObjects.add(object)) {
					j--;
				}
			}
			// create ratings
			for (RateableObject rateableObject : collaboratedObjects) {

				// date of this groups first review for this object
				Date earliestPossibleReviewDate = rateableObject
						.getEarliestPossibleReviewDate();
				for (User user : members) {
					Review review = _reviews[_random.nextInt(5)];
					Date ratingDate = getRandomDateAfter(earliestPossibleReviewDate);
					rateableObject.rate(user, review, ratingDate);
				}

			}
			// add group
			if (!result.add(new ReviewerGroup(members, collaboratedObjects))) {
				i--;
			}
		}

		return result;
	}

	/**
	 * Erzeugt 10 Spammer Gruppen mit einer zuflligen gre zwischen 2 und 5
	 * und einer zuflligen menge von bewerteten objekten von 3 bis 10.
	 * 
	 * @return Ein {@link Set} mit den erzeugten {@link ReviewerGroup}.
	 */
	private Set<ReviewerGroup> setUpSpammerGroups() {
		Set<ReviewerGroup> result = new LinkedHashSet<ReviewerGroup>();

		for (int i = 0; i < 10; i++) {
			// find group members
			int memberCount = _random.nextInt(4) + 2;
			Set<User> members = new LinkedHashSet<User>();
			for (int j = 0; j < memberCount; j++) {
				User member = getRandomListElement(_reviewSpammers);
				if (!members.add(member)) {
					j--;
				}
			}
			// find group objects
			int objectCount = _random.nextInt(8) + 3;
			Set<RateableObject> collaboratedObjects = new LinkedHashSet<RateableObject>();
			for (int j = 0; j < objectCount; j++) {
				RateableObject object = getRandomListElement(_objects);
				if (!collaboratedObjects.add(object)) {
					j--;
				}
			}
			// create ratings
			for (RateableObject rateableObject : collaboratedObjects) {
				// rate high or low
				Review review = Review.ONE_STAR_REVIEW;
				boolean rateHigh = _random.nextBoolean();
				if (rateHigh) {
					review = Review.FIVE_STAR_REVIEW;
				}
				// date of this groups first review for this object
				Date earliestPossibleReviewDate = rateableObject
						.getEarliestPossibleReviewDate();
				long timeFrameStart = earliestPossibleReviewDate.getTime() + 1;
				// time frame for this groups ratings
				int twentyFourHoursInMilliSec = 24 * 60 * 60 * 1000;
				int timeFrameEnd = _random.nextInt(twentyFourHoursInMilliSec);
				for (User user : members) {
					Date ratingDate = new Date(timeFrameStart
							+ _random.nextInt(timeFrameEnd));
					rateableObject.rate(user, review, ratingDate);
				}

			}
			// add group
			if (!result.add(new ReviewerGroup(members, collaboratedObjects))) {
				i--;
			}
		}

		return result;
	}

	private <T> T getRandomListElement(List<T> list) {
		return list.get(_random.nextInt(list.size()));
	}

	/**
	 * Erzeugt ein zuflliges Datum in 2013. Nicht im Dezember. Kein 29.,30.
	 * oder 31..
	 * 
	 * @return Das zufllig generierte {@link Date}.
	 */
	@SuppressWarnings("deprecation")
	private Date getRandomDate() {
		int year = 113;
		int month = _random.nextInt(11) + 1;
		int day = _random.nextInt(28) + 1;
		int hour = _random.nextInt(24);
		int min = _random.nextInt(60);
		int sec = _random.nextInt(60);
		return new Date(year, month, day, hour, min, sec);
	}

	/**
	 * Erzeugt ein zuflliges Datum in 2013 NACH einem bergebenen Datum. Kein
	 * 29.,30 oder 31..
	 * 
	 * @param date
	 *            Das {@link Date} nachdem das generierte liegen soll.
	 * @return Das generierte {@link Date} was nach date liegt.
	 */
	@SuppressWarnings("deprecation")
	private Date getRandomDateAfter(Date date) {
		Date result;
		do {
			int year = 113;
			int month = _random.nextInt(12) + 1;
			int day = _random.nextInt(28) + 1;
			int hour = _random.nextInt(24);
			int min = _random.nextInt(60);
			int sec = _random.nextInt(60);
			result = new Date(year, month, day, hour, min, sec);

		} while (!result.after(date));
		return result;
	}

	private void setUpUsers() {
		for (int i = 1; i <= 90; i++) {
			User user = new User("randomUser" + i);
			_userService.registerUser(user);
			_normalUsers.add(user);
		}
		for (int i = 1; i <= 10; i++) {
			User user = new User("spammer" + i);
			_userService.registerUser(user);
			_reviewSpammers.add(user);
		}
	}

	@Test
	public void testStatics() {
		assertTrue(ConfidenceLevelCalculator
				.getCurrentlyAssumedPercentageOfReviewSpammers() == 10);
		ConfidenceLevelCalculator.setNewAssumedPercentageOfReviewSpammers(50);
		assertTrue(ConfidenceLevelCalculator
				.getCurrentlyAssumedPercentageOfReviewSpammers() == 50);

		boolean assertionError = false;
		try {
			ConfidenceLevelCalculator
					.setNewAssumedPercentageOfReviewSpammers(-1);
		} catch (AssertionError e) {
			assertionError = true;
		}
		assertTrue("There should have been an assertion error.", assertionError);

		assertionError = false;
		try {
			ConfidenceLevelCalculator
					.setNewAssumedPercentageOfReviewSpammers(101);
		} catch (AssertionError e) {
			assertionError = true;
		}
		assertTrue("There should have been an assertion error.", assertionError);
		ConfidenceLevelCalculator.setNewAssumedPercentageOfReviewSpammers(10);

		assertTrue(ConfidenceLevelCalculator.useMaximumGroupSpammingScore());
		ConfidenceLevelCalculator.setUseMaximumGroupSpammingScore(false);
		assertFalse(ConfidenceLevelCalculator.useMaximumGroupSpammingScore());
		ConfidenceLevelCalculator.setUseMaximumGroupSpammingScore(true);

	}

	@Test
	public void testCalculateConfidenceLevels() {
		int counter = 0;
		_averageRatingCalc.calculateAverageRatings();
		_instance.calculateConfidenceLevels();
		Map<User, Double> totalSpammingScores1 = new LinkedHashMap<User, Double>();
		for (User user : _users) {
			totalSpammingScores1
					.put(user, user.getConfidenceLevel().getValue());
		}
		Map<User, Double> totalSpammingScores2 = new LinkedHashMap<User, Double>();
		do {
			counter++;
			_averageRatingCalc.calculateAverageRatings();
			_instance.calculateConfidenceLevels();

			for (User user : _users) {
				totalSpammingScores2.put(user, user.getConfidenceLevel()
						.getValue());
			}
			if (totalSpammingScores2.equals(totalSpammingScores1)) {
				break;
			} else {
				totalSpammingScores1 = totalSpammingScores2;
			}
		} while (true);
		Map<User, Double> result = new LinkedHashMap<User, Double>();
		for (int i = 0; i < _users.size(); i++) {
			double maximum = 0.0;
			User maximumHolder = null;

			for (Map.Entry<User, Double> entry : totalSpammingScores1
					.entrySet()) {
				if (entry.getValue() >= maximum) {
					maximum = entry.getValue();
					maximumHolder = entry.getKey();
				}
			}
			totalSpammingScores1.remove(maximumHolder);
			result.put(maximumHolder, maximum);
		}
		System.out.println("All users and their new total spamming score:");
		System.out.println("---------------------------------------------");
		for (Map.Entry<User, Double> entry : result.entrySet()) {
			System.out.println("User : " + entry.getKey()
					+ " with spamming score: " + entry.getValue());

		}
		System.out.println("---------------------------------------------");
		// System.out.println(counter);
	}

	@Test
	public void testTest() {
		// Dies und Zeilte 526 einkommentieren und Zeilen 503-525
		// auskommentieren fr intensieveren Test mit verschiedenen Seeds.
		// Set<Integer> seeds = new LinkedHashSet<Integer>();
		// for (int i = 0; i < 200000; i++) {
		// if (!seeds.add(_random.nextInt())) {
		// i--;
		// }
		// }
		// System.out.println(seeds.size());
		// try {
		// Thread.sleep(2000);
		// } catch (InterruptedException e1) {
		// e1.printStackTrace();
		// throw new RuntimeException();
		// }
		// for (int i = 0; i < 2000; i++) {
		// int seed = _random.nextInt();
		// try {
		// setUp();
		// } catch (Exception e) {
		// e.printStackTrace();
		// }
		// System.out.print(seed + " tries ");
		// _random = new Random(seed);
		// for (int k = 0; k < 10; k++) {
		// for (int j = 0; j < _random.nextInt(100); j++) {
		// _random.nextBoolean();
		// }
		//
		// }
		// testCalculateConfidenceLevels();
		// }
	}

	/**
	 * Eine Test-Implementation des Interfaces {@link IReviewerGroupDataMiner}
	 * zum Testen des {@link ConfidenceLevelCalculator}.
	 * 
	 * @author Jens Bothur
	 */
	private static class TestRaterGroupMinerForConfidenceLevelCalculatorTest
			implements IReviewerGroupDataMiner {

		/**
		 * Eine Menge mit allen Bewerter-Gruppen.
		 */
		private Set<ReviewerGroup> _groups;

		/**
		 * Konstruktor fr
		 * {@link TestRaterGroupMinerForConfidenceLevelCalculatorTest}.
		 * 
		 * @param groups
		 *            Ein {@link Set} von {@link ReviewerGroup}.
		 */
		public TestRaterGroupMinerForConfidenceLevelCalculatorTest(
				Set<ReviewerGroup> groups) {
			_groups = groups;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public Set<ReviewerGroup> getAllReviewerGroups() {
			return _groups;
		}

	}

	/**
	 * Eine Test-Implementation des Interfaces
	 * {@link IGroupOfRateableObjectsDataMiner} zum Testen des
	 * {@link ConfidenceLevelCalculator}.
	 * 
	 * @author Jens Bothur
	 */
	private static class TestObjectGroupMinerForConfidenceLevelCalculatorTest
			implements IGroupOfRateableObjectsDataMiner {

		/**
		 * Eine Menge mit allen Objekt-Gruppen.
		 */
		private Set<GroupOfRateableObjects> _objectGroups;

		/**
		 * Konstruktor fr
		 * {@link TestObjectGroupMinerForConfidenceLevelCalculatorTest}.
		 * 
		 * @param objectGroups
		 *            Ein {@link Set} von {@link GroupOfRateableObjects}.
		 */
		public TestObjectGroupMinerForConfidenceLevelCalculatorTest(
				Set<GroupOfRateableObjects> objectGroups) {
			_objectGroups = objectGroups;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public Set<GroupOfRateableObjects> getAllGroupsOfRateableObjects() {
			return _objectGroups;
		}

	}

}
