/**
 *  Serponix is an arcade game in focus to multiplayer based on the classic game Snake.
 *  Copyright (C) 2010 - 2011  Daniel Vala
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License,
 *  or  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *  If you have any question do not hesitate to contact author
 *  on e-mail address: danielvala42@gmail.com
 */
package com.serponix.game;

import com.serponix.net.Informace;

/**
 * Směr obsahuje hodnoty NAHORU, DOLU, VLEVO, VPRAVO a NIKAM.
 *
 * @author Daniel Vala
 */
public enum Direction {

	UP, DOWN, LEFT, RIGHT, NOWHERE;

	/**
	 * Vrátí směr opačnéhý od směru zadaného.
	 * Směr opačný od směru NIKAM je opět směr NIKAM.
	 *
	 * @param smer Směr NAHORU, DOLU, VLEVO, VPRAVO nebo NIKAM.
	 * @return směr opačnéhý od směru zadaného,
	 * nebo null, pokud je zadaný směr null.
	 */
	public static Direction obratSmer(Direction smer) {
		switch (smer) {
			case UP:
				return Direction.DOWN;
			case DOWN:
				return Direction.UP;
			case LEFT:
				return smer.RIGHT;
			case RIGHT:
				return smer.LEFT;
			case NOWHERE:
				return smer.NOWHERE;
		}
		return null;
	}

	/**
	 * Vrátí směr, který je od zadaného směru relativní.
	 * Např. při zadaném směru doprava vrátí směr nahoru,
	 * protože při zatočení doleva bude výsledný směr nahoru.
	 *
	 * @param smer Směr, od kterého se vrátí relativní směr doleva.
	 * @return Relativní směr od směru zadaného.
	 */
	public static Direction getSmerVlevo(Direction smer) {
		switch (smer) {
			case UP:
				return Direction.LEFT;
			case DOWN:
				return Direction.RIGHT;
			case LEFT:
				return Direction.DOWN;
			case RIGHT:
				return Direction.UP;
			case NOWHERE:
				return NOWHERE;
		}
		return null;
	}

	/**
	 * Vrátí směr, který je od zadaného směru relativní.
	 * Např. při zadaném směru doprava vrátí směr dolu,
	 * protože při zatočení doprava bude výsledný směr dolu.
	 *
	 * @param smer Směr, od kterého se vrátí relativní směr doprava.
	 * @return Relativní směr od směru zadaného.
	 */
	public static Direction getSmerVpravo(Direction smer) {
		switch (smer) {
			case UP:
				return Direction.RIGHT;
			case DOWN:
				return Direction.LEFT;
			case LEFT:
				return Direction.UP;
			case RIGHT:
				return Direction.DOWN;
			case NOWHERE:
				return NOWHERE;
		}
		return null;
	}

	/**
	 * Převede směr na číslo, které lze poslat po síti.
	 *
	 * @param smer Směr.
	 * @return Číslo směru. Při směru NIKAM nebo null vrátí -1.
	 */
	public static int prevedSmerNaCislo(Direction smer) {
		switch (smer) {
			case UP:
				return Informace.NAHORU;
			case DOWN:
				return Informace.DOLU;
			case LEFT:
				return Informace.VLEVO;
			case RIGHT:
				return Informace.VPRAVO;
		}
		return -1;
	}

	/**
	 * Převede číslo, které lze přijmout ze sítě, na směr.
	 *
	 * @param cislo Číslo směru.
	 * @return Směr.
	 */
	public static Direction prevedCisloNaSmer(int cislo) {
		switch (cislo) {
			case Informace.DOLU:
				return Direction.DOWN;
			case Informace.NAHORU:
				return Direction.UP;
			case Informace.VLEVO:
				return Direction.LEFT;
			case Informace.VPRAVO:
				return Direction.RIGHT;
		}
		return null;
	}
}
