package org.simplextensions.ui.preferences;

import org.simplextensions.di.annotations.Point;
import org.simplextensions.di.annotations.ServiceDependency;
import org.simplextensions.registry.IServiceRegistry;
import org.simplextensions.registry.services.Service;
import org.simplextensions.registry.services.Start;
import org.simplextensions.ui.ISimpleXtensionsUIService;
import org.simplextensions.ui.preferences.xml.XMLPreferencesPersister;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 
 * This implementation of {@link IPreferencesService} uses
 * {@link XMLPreferencesPersister} as default properties persister. <br />
 * <br />
 * If you woulde like to change default persistence mechanism you have to
 * <ul>
 * <li>implement your own {@link IPreferencesPersister} and {@link IPreferences}
 * <li>create <b>simplextensions-ui.properties</b> file and inside this file set
 * property <b>preferencesservice.persister.className</b> or
 * <b>preferencesservice.persister.factoryClassName</b> with a value of your own
 * persister class name (if both of these properties are specified
 * <b>preferencesservice.persister.className</b> is prefered)
 * </ul>
 * 
 * 
 * @author krzyzak
 * 
 */
@Service(IPreferencesService.class)
public class PreferencesService implements IPreferencesService {

	private static final Logger log = LoggerFactory.getLogger(PreferencesService.class);

	public static final String PersisterPropertyName = "preferencesservice.persister.className";
	public static final String PersisterFactoryPropertyName = "preferencesservice.persister.factoryClassName";

	private IPreferencesPersister persister;

	@ServiceDependency
	private ISimpleXtensionsUIService sxUIService;
	
	@Point
	private IServiceRegistry serviceRegistry;
	
	public PreferencesService(){
		log.info("created");
	}

	@Start
	public void start() {
		long a = System.nanoTime();
		String persisterClassName = getSxUIService().getProperty(PersisterPropertyName);
		String persisterFactoryClassName = getSxUIService().getProperty(PersisterPropertyName);

		if (persisterClassName != null) {
			try {
				persister = (IPreferencesPersister) Class.forName(persisterClassName).newInstance();
			} catch (Exception e) {
				throw new RuntimeException(e);
			}
		} else if (persisterFactoryClassName != null) {
			try {
				IPreferencesPersisterFactory factory = (IPreferencesPersisterFactory) Class.forName(persisterClassName).newInstance();
				persister = factory.createPersister(serviceRegistry);
			} catch (Exception e) {
				throw new RuntimeException(e);
			}
		}

		if (persister == null) {
			persister = new XMLPreferencesPersister();
		}

		persister.init(serviceRegistry);
		log.debug((System.nanoTime() - a) / 1000000.0 + "ms");

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.simplextensions.ui.preferences.IPreferencesService#get(java.lang.
	 * String, java.lang.String)
	 */
	public IPreferences get(String id, String bundleId) {
		return this.persister.get(id, bundleId);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @seeorg.simplextensions.ui.preferences.IPreferencesService#save(org.
	 * simplextensions.ui.preferences.IPreferences)
	 */
	public void save(IPreferences preferences) {
		this.persister.save(preferences);
	}

	public ISimpleXtensionsUIService getSxUIService() {
		return sxUIService;
	}

}
