package org.simplextensions.registry.graph;

import static org.mockito.Matchers.argThat;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.reset;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;

import org.hamcrest.BaseMatcher;
import org.hamcrest.Description;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.runners.MockitoJUnitRunner;
import org.simplextensions.graph.Graph;
import org.simplextensions.graph.GraphEvent;
import org.simplextensions.graph.GraphEventListenerAdapter;
import org.simplextensions.graph.GraphException;
import org.simplextensions.graph.IGraphEventListener;
import org.simplextensions.graph.IGraphVisitor;

import static junit.framework.TestCase.*;
import static org.mockito.Mockito.*;

@RunWith(MockitoJUnitRunner.class)
public class GraphTest {

	private static final String id_1 = "1";
	private static final String id_2 = "2";
	private static final String id_3 = "3";

	private final class GraphEventNodeIdMatcher extends BaseMatcher<GraphEvent> {
		private final String nodeId;

		public GraphEventNodeIdMatcher(String id_2) {
			this.nodeId = id_2;
		}

		public boolean matches(Object item) {
			return ((GraphEvent) item).data.equals(this.nodeId);
		}

		public void describeTo(Description description) {
			description.appendText("event.data.equals(" + nodeId + ")");
		}
	}

	private Graph graph;
	private IGraphEventListener mockedListeners;

	@Before
	public void setUp() {
		graph = new Graph();
		mockedListeners = spy(new GraphEventListenerAdapter());
		graph.addGraphListener(mockedListeners);
	}

	@Test
	public void testAdd() throws GraphException {

		graph.addNode(id_1, new String[] { id_2, id_3 }, id_1);

		verify(mockedListeners).nodeAdded(eventDataEquals(id_1));
		verify(mockedListeners).nodeLostConnections(eventDataEquals(id_1));
		verifyNoMoreInteractions(mockedListeners);
		reset(mockedListeners);

		graph.addNode(id_2, new String[] { id_3 }, id_2);

		verify(mockedListeners).nodeAdded(eventDataEquals(id_2));
		verify(mockedListeners).nodeLostConnections(eventDataEquals(id_2));
		verifyNoMoreInteractions(mockedListeners);
		reset(mockedListeners);

		graph.addNode(id_3, new String[] {}, id_3);
		verify(mockedListeners).nodeAdded(eventDataEquals(id_3));
		verify(mockedListeners).nodeFullyConnected(eventDataEquals(id_3));
		verify(mockedListeners).nodeFullyConnected(eventDataEquals(id_2));
		verify(mockedListeners).nodeFullyConnected(eventDataEquals(id_1));

		verifyNoMoreInteractions(mockedListeners);

		assertEquals(0, graph.getIncomingNodes(id_1).size());
		assertEquals(1, graph.getIncomingNodes(id_2).size());
		assertEquals(2, graph.getIncomingNodes(id_3).size());

		assertEquals(2, graph.getOutgoingNodes(id_1).size());
		assertEquals(1, graph.getOutgoingNodes(id_2).size());
		assertEquals(0, graph.getOutgoingNodes(id_3).size());

		assertEquals(2, graph.getConnectedNodesIds(id_1).length);
	}

	@Test
	public void testRemove() throws GraphException {
		graph.addNode(id_1, new String[] { id_2, id_3 }, id_1);
		graph.addNode(id_2, new String[] { id_3 }, id_2);
		graph.addNode(id_3, new String[] {}, id_3);
		reset(mockedListeners);

		graph.removeNode(id_3);

		verify(mockedListeners).nodeLostConnections(eventDataEquals(id_2));// argThat(Matchers.anyOf(new
		// GraphEventNodeIdMatcher(id_2),
		// new
		// GraphEventNodeIdMatcher(id_1))));
		verify(mockedListeners).nodeLostConnections(eventDataEquals(id_1));// argThat(Matchers.anyOf(new
		// GraphEventNodeIdMatcher(id_2),
		// new
		// GraphEventNodeIdMatcher(id_1))));
		verify(mockedListeners).nodeRemoved(eventDataEquals(id_3));
		verifyNoMoreInteractions(mockedListeners);
	}

	@Test
	public void testCascadeAdd() throws GraphException {
		graph.addNode(id_1, new String[] { id_2 }, id_1);
		graph.addNode(id_2, new String[] { id_3 }, id_2);
		reset(mockedListeners);

		graph.addNode(id_3, new String[] {}, id_3);
		verify(mockedListeners).nodeAdded(eventDataEquals(id_3));
		verify(mockedListeners).nodeFullyConnected(eventDataEquals(id_3));
		verify(mockedListeners).nodeFullyConnected(eventDataEquals(id_1));
		verify(mockedListeners).nodeFullyConnected(eventDataEquals(id_2));

		verifyNoMoreInteractions(mockedListeners);
	}

	@Test
	public void testRemoveDependency() throws GraphException {
		graph.addNode(id_1, new String[] { id_2 }, id_1);
		graph.addNode(id_2, new String[] { id_3 }, id_2);
		graph.addNode(id_3, new String[] {}, id_3);
		reset(mockedListeners);

		graph.removeNode(id_3);
		verify(mockedListeners).nodeRemoved(eventDataEquals(id_3));
		verify(mockedListeners).nodeLostConnections(eventDataEquals(id_1));
		verify(mockedListeners).nodeLostConnections(eventDataEquals(id_2));

		verifyNoMoreInteractions(mockedListeners);
	}

	@Test
	public void testRemoveDependencyWithDependencies() throws GraphException {
		graph.addNode(id_1, new String[] { id_2, id_3 }, id_1);
		graph.addNode(id_2, new String[] { id_3 }, id_2);
		graph.addNode(id_3, new String[] {}, id_3);
		reset(mockedListeners);

		graph.removeNode(id_2);
		verify(mockedListeners).nodeRemoved(eventDataEquals(id_2));
		verify(mockedListeners).nodeLostConnections(eventDataEquals(id_1));
		verifyNoMoreInteractions(mockedListeners);
		reset(mockedListeners);

		graph.removeNode(id_3);
		verify(mockedListeners).nodeRemoved(eventDataEquals(id_3));
		verify(mockedListeners).nodeLostConnections(eventDataEquals(id_1));
		verify(mockedListeners).nodeLostConnections(eventDataEquals(id_2));

		// new MockitoDebuggerImpl().printInvocations(mockedListeners);

		verifyNoMoreInteractions(mockedListeners);

	}

	@Test
	public void testVisitors() throws GraphException {
		graph.addNode(id_1, new String[] { id_2 }, id_1);
		graph.addNode(id_2, new String[] { id_3 }, id_2);
		graph.addNode(id_3, new String[] {}, id_3);
		reset(mockedListeners);

		IGraphVisitor mock = mock(IGraphVisitor.class);
		graph.visitWithDependencies(id_1, mock);

		verify(mock).visit(eq(id_3));
		verify(mock).visit(eq(id_2));
		verify(mock).visit(eq(id_1));
		verifyNoMoreInteractions(mockedListeners);
	}

	public GraphEvent eventDataEquals(String data) {
		return argThat(new GraphEventNodeIdMatcher(data));
	}

}
