package org.simplextensions.ioc;

import org.simplextensions.annotations.AnnotationInfo;
import org.simplextensions.registry.IExtensionRegistry;
import org.simplextensions.registry.phaselisteners.IDependencyLocator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;

/**
 * @author Mirek Szajowski m.szajowski@gmail.com
 * 
 */
public class DependencyInjector {

	private static final Logger log = LoggerFactory.getLogger(DependencyInjector.class);

	private static class FieldInfo {

		public Field field;

		public List<Class<? extends Annotation>> foundAnnotations = new ArrayList<Class<? extends Annotation>>();
	}

	public static void resolveDependency(IExtensionRegistry registry, Object targetObject) {
		resolveDependency(new EnvironmentDependencyLocator(registry), targetObject);
	}

	public static void resolveDependency(IDependencyLocator locator, Object targetObject) {
		if (targetObject == null) {
			throw new NullPointerException("Object to injection is null");
		}
		Class<? extends Annotation>[] supportedAnnotations = locator.getSupportedAnnotations();
		List<FieldInfo> fieldsWithAnnotation = getFieldsWithAnnotation(targetObject.getClass(), supportedAnnotations);

		for (FieldInfo fieldInfo : fieldsWithAnnotation) {
			Field field = fieldInfo.field;
			for (Class<? extends Annotation> annotationClass : fieldInfo.foundAnnotations) {
				if (locator != null) {
					Object value = locator.getAnnotatedFieldValue(field,
							AnnotationInfo.generateAnnotationInfo(getAnnotation(field, annotationClass)));
					injectService(targetObject, field, value);

				}
			}
		}
	}

	private static Annotation getAnnotation(Field field, Class<? extends Annotation> annotationClass) {
		return field.getAnnotation(annotationClass);
	}

	/**
	 * Returns fields which contains searching annotations
	 * 
	 * @param clazz
	 * @param supportedAnnotations
	 * @return
	 */
	private static List<FieldInfo> getFieldsWithAnnotation(Class<?> clazz, Class<? extends Annotation>[] supportedAnnotations) {
		List<FieldInfo> fieldsWithAnnotations = new ArrayList<FieldInfo>();
		List<Field> allClassFields = ClassSearcherUtil.getAllClassFields(clazz);
		for (Field field : allClassFields) {
			FieldInfo fieldInfo = getFieldInfo(field, supportedAnnotations);
			if (fieldInfo.foundAnnotations.size() > 0) {
				fieldsWithAnnotations.add(fieldInfo);
			}
		}

		return fieldsWithAnnotations;
	}

	/**
	 * Return field informations
	 * 
	 * @param field
	 * @param supportedAnnotations
	 * @return
	 */
	private static FieldInfo getFieldInfo(Field field, Class<? extends Annotation>[] supportedAnnotations) {
		FieldInfo fieldInfo = new FieldInfo();

		for (Annotation annotation : field.getAnnotations()) {
			for (Class<? extends Annotation> searchingAnnotation : supportedAnnotations) {
				if (annotation.annotationType().equals(searchingAnnotation)) {
					fieldInfo.foundAnnotations.add(searchingAnnotation);
				}
			}
		}
		// if contains at least one of searching annotations
		if (fieldInfo.foundAnnotations.size() > 0) {
			fieldInfo.field = field;
		}
		return fieldInfo;
	}

	/**
	 * Inject value into field
	 * 
	 * @param targetObject
	 * @param field
	 * @param value
	 */
	private static void injectService(Object targetObject, Field field, Object value) {
		log.trace("--- injecting: " + value + " into: " + field.toGenericString());
		field.setAccessible(true);
		try {
			field.set(targetObject, value);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}
}
