/*
Copyright (c) 2012, Felipe Michels Fontoura, Lucas Longen Gioppo,
Luís Guilherme Bergamini Mendes e Renato Girardi Gasoto.
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of Universidade Tecnológica Federal do Paraná nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package utfpr.sistemaestagios.dao.text;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import jtextdatastore.TextDataCondition;
import jtextdatastore.TextDataTable;
import utfpr.sistemaestagios.dao.DAOEstagio;
import utfpr.sistemaestagios.types.Aluno;
import utfpr.sistemaestagios.types.Estagio;

/**
 * @author Felipe Michels Fontoura
 */
public class TextDAOEstagio implements DAOEstagio {
	private TextDataTable<Estagio> table;

	public TextDAOEstagio(File folder) {
		table = new TextDataTable<Estagio>(Estagio.class, new File(folder, "estagio.txt"));
	}

	public Estagio insert(Estagio estagio) {
		if (table.insert(estagio)) return estagio;
		else return null;
	}

	public boolean delete(long id) {
		Estagio estagio = new Estagio();
		estagio.id = id;
		if (table.delete(estagio)) return true;
		else return false;
	}

	public Estagio find(long id) {
		Estagio estagio = new Estagio();
		estagio.id = id;
		TextDataCondition<Estagio> condition = table.getMetadata().generatePrimaryKeyMatchesCondition(estagio);
		List<Estagio> estagios = table.select(condition);
		if (estagios == null) return null;
		if (estagios.size() == 0) return null;
		return estagios.get(0);
	}

	public Estagio update(Estagio empresa) {
		if (table.update(empresa)) return empresa;
		else return null;
	}

	public Collection<Estagio> select() {
		return table.select();
	}

	public Collection<Estagio> select_byAluno(String codigoAluno) {
		return table.select(new CodigoAlunoCondition(codigoAluno));
	}

	public Collection<Estagio> select_byProfessor(String codigoProfessor) {
		return table.select(new CodigoProfessorCondition(codigoProfessor));
	}

	public Collection<Estagio> select_byCurso(String codigoCurso) {
		Collection<Aluno> alunos = TextDAOFactory.getInstance().getDAOAluno().select_byCurso(codigoCurso);
		System.out.println(alunos.size());
		return table.select(new ListAlunoCondition(alunos));
	}

	private static class CodigoAlunoCondition implements TextDataCondition<Estagio> {
		String codigoAluno;
		public CodigoAlunoCondition(String codigoAluno) {
			this.codigoAluno = codigoAluno;
		}
		public boolean accept(Estagio item) {
			return codigoAluno.equals(item.codigoAluno);
		}
	}

	private static class CodigoProfessorCondition implements TextDataCondition<Estagio> {
		String codigoProfessor;
		public CodigoProfessorCondition(String codigoProfessor) {
			this.codigoProfessor = codigoProfessor;
		}
		public boolean accept(Estagio item) {
			return codigoProfessor.equals(item.codigoProfessor);
		}
	}

	private static class ListAlunoCondition implements TextDataCondition<Estagio> {
		List<String> codigoAluno;
		public ListAlunoCondition(Collection<Aluno> alunos) {
			if (alunos == null) {
				codigoAluno = Collections.emptyList();
			} else {
				this.codigoAluno = new ArrayList<String>(alunos.size());
				for (Aluno aluno : alunos) {
					codigoAluno.add(aluno.codigoMatricula);
				}
				Collections.sort(codigoAluno);
			}
		}
		public boolean accept(Estagio item) {
			if (item.codigoAluno != null)
				return Collections.binarySearch(codigoAluno, item.codigoAluno) >= 0;
			else 
				return false;
		}
	}
}
