/*
 * Copyright (c) 2004-2005, Volatile-Engine All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * Neither the name of the Volatile-Engine nor the
 * names of its contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package com.volatileengine.datatypes;

import java.nio.ByteBuffer;

import javax.vecmath.Color4f;
import javax.vecmath.Quat4f;
import javax.vecmath.Tuple2f;
import javax.vecmath.Tuple3f;

import org.lwjgl.BufferUtils;

/**
 * @author Ahmed
 * 
 */
public class ArrayByte extends Array<ByteBuffer> {

	public ArrayByte(int chunkSize) {
		super(chunkSize, 0);
	}

	public ArrayByte(int chunkSize, int length) {
		super(chunkSize, length);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.volatileEngine.datatypes.Array#getType()
	 */
	@Override
	public DataType getType() {
		return DataType.Byte;
	}

	@Override
	protected ByteBuffer createBuffer(int length, ByteBuffer oldBuffer) {
		ByteBuffer b = BufferUtils.createByteBuffer(length);

		if (oldBuffer != null) {
			b.put(oldBuffer);
		}
		return b;
	}

	public void updateBuffer(byte[] array, int bufferStart, int dataStart, int length) {
		buffer.rewind();
		buffer.position(bufferStart);
		for (int i = 0; i < length; i++) {
			buffer.put(array[dataStart++]);
		}
	}

	public void updateBuffer(byte[] array) {
		updateBuffer(array, 0, 0, array.length);
	}

	@Override
	public Color4f getColor4f(int index, int offset, Color4f target) {
		ByteBuffer buffer = getBuffer();
		buffer.position(chunkSize * index + offset);
		target.x = buffer.get();
		target.y = buffer.get();
		target.z = buffer.get();
		target.w = buffer.get();
		return target;
	}

	@Override
	public Quat4f getQuat4f(int index, int offset, Quat4f target) {
		ByteBuffer buffer = getBuffer();
		buffer.position(chunkSize * index + offset);
		target.x = buffer.get();
		target.y = buffer.get();
		target.z = buffer.get();
		target.w = buffer.get();
		return target;
	}

	@Override
	public Tuple2f getTuple2f(int index, int offset, Tuple2f target) {
		ByteBuffer buffer = getBuffer();
		buffer.position(chunkSize * index + offset);
		target.x = buffer.get();
		target.y = buffer.get();
		return target;
	}

	@Override
	public Tuple3f getTuple3f(int index, int offset, Tuple3f target) {
		ByteBuffer buffer = getBuffer();
		buffer.position(chunkSize * index + offset);
		target.x = buffer.get();
		target.y = buffer.get();
		target.z = buffer.get();
		return target;
	}

	@Override
	public void setColor4f(int index, int offset, Color4f source) {
		ByteBuffer buffer = getBuffer();
		buffer.position(chunkSize * index + offset);
		buffer.put((byte) source.x);
		buffer.put((byte) source.y);
		buffer.put((byte) source.z);
		buffer.put((byte) source.w);
	}

	@Override
	public void setQuat4f(int index, int offset, Quat4f source) {
		ByteBuffer buffer = getBuffer();
		buffer.position(chunkSize * index + offset);
		buffer.put((byte) source.x);
		buffer.put((byte) source.y);
		buffer.put((byte) source.z);
		buffer.put((byte) source.w);
	}

	@Override
	public void setTuple2f(int index, int offset, Tuple2f source) {
		ByteBuffer buffer = getBuffer();
		buffer.position(chunkSize * index + offset);
		buffer.put((byte) source.x);
		buffer.put((byte) source.y);
	}

	@Override
	public void setTuple3f(int index, int offset, Tuple3f source) {
		ByteBuffer buffer = getBuffer();
		buffer.position(chunkSize * index + offset);
		buffer.put((byte) source.x);
		buffer.put((byte) source.y);
		buffer.put((byte) source.z);
	}

	@Override
	public byte getByte(int index, int offset) {
		return getBuffer().get(index * chunkSize + offset);
	}

	@Override
	public float getFloat(int index, int offset) {
		return getBuffer().getFloat(index * chunkSize + offset);
	}

	@Override
	public int getInt(int index, int offset) {
		return getBuffer().getInt(index * chunkSize + offset);
	}

	@Override
	public void setByte(int index, int offset, byte value) {
		getBuffer().put(index * chunkSize + offset, value);
	}

	@Override
	public void setFloat(int index, int offset, float value) {
		getBuffer().putFloat(index * chunkSize + offset, (byte) value);
	}

	@Override
	public void setInt(int index, int offset, int value) {
		getBuffer().putInt(index * chunkSize + offset, (byte) value);
	}

	@Override
	public void copyFrom(ByteBuffer buffer) {
		if (buffer.remaining() != this.buffer.limit()) {
			throw new IllegalArgumentException("Mismatched sizes");
		}
		this.buffer.rewind();
		this.buffer.put(buffer);
	}

	@Override
	public void copyInto(ByteBuffer buffer) {
		if (buffer.remaining() != this.buffer.limit()) {
			throw new IllegalArgumentException("Mismatched sizes");
		}
		this.buffer.rewind();
		buffer.put(this.buffer);
	}
}
