/*
 * Copyright (c) 2004-2006, Volatile-Engine All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * Neither the name of the Volatile-Engine nor the
 * names of its contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */
/**
 * 
 */
package com.volatileengine.scene;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;

/**
 * @author Administrator
 * 
 */
public class SceneNode extends SceneObject implements List<SceneObject> {

	private List<SceneObject> children;

	@Override
	public int draw(List<SceneMesh> list, Frustum frustum, SceneNode parent, int parentCheck) {
		// store result of culling
		int result = 1;
		if (parentCheck == 3) {
			result = super.draw(list, frustum, parent, parentCheck);
		}

		// feed culling result to children
		if (result == 1 || result == 3) {
			for (int i = children.size(); --i >= 0;) {
				children.get(i).draw(list, frustum, this, result);
			}
		}

		// return result
		return result;
	}

	public boolean add(SceneObject e) {
		if (children == null) {
			children = new ArrayList<SceneObject>();
		}

		e.onAttach(this);
		return children.add(e);
	}

	public void add(int index, SceneObject element) {
		if (children == null) {
			children = new ArrayList<SceneObject>();
		}

		element.onAttach(this);
		children.add(index, element);
	}

	public boolean addAll(Collection<? extends SceneObject> c) {
		if (children == null) {
			children = new ArrayList<SceneObject>();
		}

		for (SceneObject obj : c) {
			obj.onAttach(this);
		}

		return children.addAll(c);
	}

	public boolean addAll(int index, Collection<? extends SceneObject> c) {
		if (children == null) {
			children = new ArrayList<SceneObject>();
		}

		for (SceneObject obj : c) {
			obj.onAttach(this);
		}

		return children.addAll(index, c);
	}

	public void clear() {
		children.clear();
	}

	public boolean contains(Object o) {
		if (children != null) {
			return children.contains(o);
		}
		return false;
	}

	public boolean containsAll(Collection<?> c) {
		if (children != null) {
			return children.containsAll(c);
		}
		return false;
	}

	public SceneObject get(int index) {
		if (children != null) {
			return children.get(index);
		}

		return null;
	}

	public int indexOf(Object o) {
		if (children != null) {
			return children.indexOf(o);
		}
		return -1;
	}

	public boolean isEmpty() {
		if (children == null) {
			return true;
		}

		return children.isEmpty();
	}

	public Iterator<SceneObject> iterator() {
		if (children == null) {
			children = new ArrayList<SceneObject>();
		}
		return children.iterator();
	}

	public int lastIndexOf(Object o) {
		if (children != null) {
			return children.lastIndexOf(o);
		}
		return -1;
	}

	public ListIterator<SceneObject> listIterator() {
		if (children == null) {
			children = new ArrayList<SceneObject>();
		}
		return children.listIterator();
	}

	public ListIterator<SceneObject> listIterator(int index) {
		if (children == null) {
			children = new ArrayList<SceneObject>();
		}
		return children.listIterator(index);
	}

	public boolean remove(Object o) {
		if (children != null) {
			boolean removed = children.remove(o);

			if (removed) {
				// must be a SceneObject because we can only add
				// SceneObjects
				// and
				// removed would be false if it was anything else
				((SceneObject) o).onDetach(this);
			}
			return removed;
		}

		return false;
	}

	public SceneObject remove(int index) {
		if (children != null) {
			SceneObject obj = children.remove(index);

			if (obj != null) {
				obj.onDetach(this);
			}
			return obj;
		}
		throw new IndexOutOfBoundsException("Index: " + index + ", Size: 0");
	}

	public boolean removeAll(Collection<?> c) {
		boolean modified = false;
		Iterator<SceneObject> e = iterator();
		while (e.hasNext()) {
			SceneObject obj = e.next();
			if (c.contains(obj)) {
				e.remove();
				obj.onDetach(this);
				modified = true;
			}
		}
		return modified;
	}

	public boolean retainAll(Collection<?> c) {
		boolean modified = false;
		Iterator<SceneObject> e = iterator();
		while (e.hasNext()) {
			SceneObject obj = e.next();
			if (!c.contains(obj)) {
				e.remove();
				obj.onDetach(this);
				modified = true;
			}
		}
		return modified;
	}

	public SceneObject set(int index, SceneObject element) {
		if (children == null) {
			children = new ArrayList<SceneObject>();
		}
		return children.set(index, element);
	}

	public int size() {
		if (children != null) {
			return children.size();
		}

		return 0;
	}

	public List<SceneObject> subList(int fromIndex, int toIndex) {
		if (children != null) {
			return children.subList(fromIndex, toIndex);
		}

		return null;
	}

	public Object[] toArray() {
		if (children != null) {
			return children.toArray();
		}

		return null;
	}

	public <T> T[] toArray(T[] a) {
		if (children != null) {
			return children.toArray(a);
		}

		return null;
	}
}
