from argparse import ArgumentParser, _SubParsersAction
from dataclasses import MISSING, Field, is_dataclass
from types import UnionType
from typing import (
    Dict,
    Literal,
    Optional,
    Type,
    TypeVar,
    Union,
    get_args,
    get_origin,
    overload,
)

T = TypeVar("T")
T2 = TypeVar("T2")
T3 = TypeVar("T3")
T4 = TypeVar("T4")
T5 = TypeVar("T5")
T6 = TypeVar("T6")
T7 = TypeVar("T7")
T8 = TypeVar("T8")


def _add_dataclass_arguments(parser: Union[ArgumentParser, _SubParsersAction], class_type: Type[T]) -> None:
    """
    Add arguments from a dataclass to an argument parser or subparser.

    Args:
        parser: The parser to add arguments to
        class_type: The dataclass type containing the arguments
    """

    if not is_dataclass(class_type):
        raise TypeError(f"{class_type.__name__} must be a dataclass")

    for field_name, field in class_type.__dataclass_fields__.items():

        field_type = field.type
        origin = get_origin(field_type)
        if origin is not None:
            if origin is Union or origin is UnionType:  # Handle Union[T, None]
                args = get_args(field_type)
                if len(args) == 2 and args[1] is type(None):
                    field_type = args[0]
            elif origin is Literal:
                args = get_args(field_type)
                literal_type = type(args[0])
                if all(isinstance(arg, literal_type) for arg in args):
                    field_type = literal_type
                else:
                    raise TypeError("Cannot parse Literal with differing argument types")
            else:
                field_type = origin

        default = None if field.default is MISSING else field.default
        if field.default_factory is not MISSING:
            default = field.default_factory()

        required = default is None and field.default is MISSING and field.default_factory is MISSING

        metadata = field.metadata.copy()
        help_text = metadata.pop("help", field.metadata.get("description", ""))

        if field_type is bool and default is not None:
            action = "store_false" if default else "store_true"
            parser.add_argument(f"--{field_name}", action=action, help=help_text, **metadata)  # type: ignore[arg-type]
        else:
            parser.add_argument(f"--{field_name}", type=field_type, default=default, required=required, help=help_text, **metadata)  # type: ignore[arg-type]


DataclassType = Union[Type[T], tuple[Type[T], ...]]
ParseResult = Union[T, tuple[T, ...]]


@overload
def parse_args_to_dataclass(dataclasses: Type[T], description: Optional[str] = None) -> T: ...
@overload
def parse_args_to_dataclass(dataclasses: tuple[Type[T], Type[T2]], description: Optional[str] = None) -> tuple[T, T2]: ...
@overload
def parse_args_to_dataclass(dataclasses: tuple[Type[T], Type[T2], Type[T3]], description: Optional[str] = None) -> tuple[T, T2, T3]: ...
@overload
def parse_args_to_dataclass(dataclasses: tuple[Type[T], Type[T2], Type[T3], Type[T4]], description: Optional[str] = None) -> tuple[T, T2, T3, T4]: ...
@overload
def parse_args_to_dataclass(dataclasses: tuple[Type[T], Type[T2], Type[T3], Type[T4], Type[T5]], description: Optional[str] = None) -> tuple[T, T2, T3, T4, T5]: ...
@overload
def parse_args_to_dataclass(dataclasses: tuple[Type[T], Type[T2], Type[T3], Type[T4], Type[T5], Type[T6]], description: Optional[str] = None) -> tuple[T, T2, T3, T4, T5, T6]: ...
@overload
def parse_args_to_dataclass(
    dataclasses: tuple[Type[T], Type[T2], Type[T3], Type[T4], Type[T5], Type[T6], Type[T7]], description: Optional[str] = None
) -> tuple[T, T2, T3, T4, T5, T6, T7]: ...


def parse_args_to_dataclass(dataclasses: DataclassType, description: Optional[str] = None) -> ParseResult:
    """
    Parse command line arguments into one or more dataclass instances.

    Args:
        dataclasses: Either:
            - A single dataclass type
            - A tuple of dataclass types
        description: Optional description for the argument parser

    Returns:
        out: `T | tuple[T, ...]`
        - For a single dataclass: The dataclass instance
        - For multiple dataclasses (tuple): Tuple with dataclass instances

    Raises:
        TypeError: If any provided type is not a dataclass
        ValueError: If required fields are missing or invalid format
    """

    if isinstance(dataclasses, type):
        parser = ArgumentParser(description=description or dataclasses.__doc__)
        _add_dataclass_arguments(parser, dataclasses)

        args = parser.parse_args()
        args_dict = {k: v for k, v in vars(args).items() if k in dataclasses.__dataclass_fields__}

        return dataclasses(**args_dict)

    elif isinstance(dataclasses, tuple):
        parser = ArgumentParser(description=description)

        for i, class_type in enumerate(dataclasses):
            group = parser.add_argument_group(f"group_{i}", class_type.__doc__)
            _add_dataclass_arguments(group, class_type)  # type: ignore

        args = parser.parse_args()

        result = []
        for class_type in dataclasses:
            args_dict = {k: v for k, v in vars(args).items() if k in class_type.__dataclass_fields__}
            result.append(class_type(**args_dict))

        return tuple(result)

    else:
        raise ValueError("dataclasses argument must be a dataclass type, or tuple of dataclass types")
