# coding=utf-8
# Copyright 2022 EleutherAI and the HuggingFace Inc. team. All rights reserved.
#
# This code is based on EleutherAI's GPT-NeoX library and the GPT-NeoX
# and OPT implementations in this library. It has been modified from its
# original forms to accommodate minor architectural differences compared
# to GPT-NeoX and OPT used by the Meta AI team that trained the model.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
""" PyTorch LLaMA model."""
import math
import warnings
from typing import List, Optional, Tuple, Union

import torch
import torch.nn.functional as F
import torch.utils.checkpoint
from torch import nn
from torch.nn import BCEWithLogitsLoss, CrossEntropyLoss, MSELoss

from transformers.activations import ACT2FN
from transformers.modeling_attn_mask_utils import AttentionMaskConverter, _prepare_4d_causal_attention_mask
from transformers.modeling_outputs import BaseModelOutputWithPast, CausalLMOutputWithPast, SequenceClassifierOutputWithPast
from transformers.modeling_utils import PreTrainedModel
from transformers.pytorch_utils import ALL_LAYERNORM_LAYERS
from transformers.utils import (
    add_start_docstrings,
    add_start_docstrings_to_model_forward,
    is_flash_attn_2_available,
    logging,
    replace_return_docstrings,
)
from transformers.utils.import_utils import is_torch_fx_available
from .configuration_llama import LlamaConfig


if is_flash_attn_2_available():
    from flash_attn import flash_attn_func, flash_attn_varlen_func
    from flash_attn.bert_padding import index_first_axis, pad_input, unpad_input  # noqa

import os
import pickle
import numpy as np
from collections import defaultdict
from scipy.special import softmax
import re
from einops import rearrange
import sys
sys.path.append('../')
from dx_privacy.dx_utils import DX_privacy
from functools import partial
from baukit import Trace, TraceDict
from common_utils import inject_prompt, format_questions_mc, format_ans, \
    lt_modulated_vector_add_new_direction, lt_modulated_vector_add_new_direction_client, client_direction, lt_modulated_vector_add_new_direction_speed, \
    lt_modulated_vector_add_new_direction_noWeight
import random


# This makes `_prepare_4d_causal_attention_mask` a leaf function in the FX graph.
# It means that the function will not be traced through and simply appear as a node in the graph.
if is_torch_fx_available():
    _prepare_4d_causal_attention_mask = torch.fx.wrap(_prepare_4d_causal_attention_mask)


logger = logging.get_logger(__name__)

_CONFIG_FOR_DOC = "LlamaConfig"


def _get_unpad_data(attention_mask):
    seqlens_in_batch = attention_mask.sum(dim=-1, dtype=torch.int32)
    indices = torch.nonzero(attention_mask.flatten(), as_tuple=False).flatten()
    max_seqlen_in_batch = seqlens_in_batch.max().item()
    cu_seqlens = F.pad(torch.cumsum(seqlens_in_batch, dim=0, dtype=torch.torch.int32), (1, 0))
    return (
        indices,
        cu_seqlens,
        max_seqlen_in_batch,
    )


def _expand_mask(mask: torch.Tensor, dtype: torch.dtype, tgt_len: Optional[int] = None):
    warnings.warn(
        "Calling `transformers.models.llama.modeling_llama._prepare_4d_attention_mask` is deprecated and will be removed in v4.37. Use `transformers.modeling_attn_mask_utils.AttentionMaskConverter._prepare_4d_attention_mask"
    )
    return AttentionMaskConverter._prepare_4d_attention_mask(mask=mask, dtype=dtype, tgt_len=tgt_len)


def _make_causal_mask(
    input_ids_shape: torch.Size, dtype: torch.dtype, device: torch.device, past_key_values_length: int = 0
):
    warnings.warn(
        "Calling `transformers.models.llama.modeling_llama._make_causal_mask` is deprecated and will be removed in v4.37. Use `transformers.models.llama.modeling_llama.AttentionMaskConverter._make_causal_mask"
    )
    return AttentionMaskConverter._make_causal_mask(
        input_ids_shape=input_ids_shape, dtype=dtype, device=device, past_key_values_length=past_key_values_length
    )


class LlamaRMSNorm(nn.Module):
    def __init__(self, hidden_size, eps=1e-6):
        """
        LlamaRMSNorm is equivalent to T5LayerNorm
        """
        super().__init__()
        self.weight = nn.Parameter(torch.ones(hidden_size))
        self.variance_epsilon = eps

    def forward(self, hidden_states):
        input_dtype = hidden_states.dtype
        hidden_states = hidden_states.to(torch.float32)
        variance = hidden_states.pow(2).mean(-1, keepdim=True)
        hidden_states = hidden_states * torch.rsqrt(variance + self.variance_epsilon)
        return self.weight * hidden_states.to(input_dtype)


ALL_LAYERNORM_LAYERS.append(LlamaRMSNorm)


class LlamaRotaryEmbedding(nn.Module):
    def __init__(self, dim, max_position_embeddings=2048, base=10000, device=None):
        super().__init__()

        self.dim = dim
        self.max_position_embeddings = max_position_embeddings
        self.base = base
        inv_freq = 1.0 / (self.base ** (torch.arange(0, self.dim, 2).float().to(device) / self.dim))
        self.register_buffer("inv_freq", inv_freq, persistent=False)

        # Build here to make `torch.jit.trace` work.
        self._set_cos_sin_cache(
            seq_len=max_position_embeddings, device=self.inv_freq.device, dtype=torch.get_default_dtype()
        )

    def _set_cos_sin_cache(self, seq_len, device, dtype):
        self.max_seq_len_cached = seq_len
        t = torch.arange(self.max_seq_len_cached, device=device, dtype=self.inv_freq.dtype)

        freqs = torch.einsum("i,j->ij", t, self.inv_freq)
        # Different from paper, but it uses a different permutation in order to obtain the same calculation
        emb = torch.cat((freqs, freqs), dim=-1)
        self.register_buffer("cos_cached", emb.cos().to(dtype), persistent=False)
        self.register_buffer("sin_cached", emb.sin().to(dtype), persistent=False)

    def forward(self, x, seq_len=None):
        # x: [bs, num_attention_heads, seq_len, head_size]
        if seq_len > self.max_seq_len_cached:
            self._set_cos_sin_cache(seq_len=seq_len, device=x.device, dtype=x.dtype)

        return (
            self.cos_cached[:seq_len].to(dtype=x.dtype),
            self.sin_cached[:seq_len].to(dtype=x.dtype),
        )


class LlamaLinearScalingRotaryEmbedding(LlamaRotaryEmbedding):
    """LlamaRotaryEmbedding extended with linear scaling. Credits to the Reddit user /u/kaiokendev"""

    def __init__(self, dim, max_position_embeddings=2048, base=10000, device=None, scaling_factor=1.0):
        self.scaling_factor = scaling_factor
        super().__init__(dim, max_position_embeddings, base, device)

    def _set_cos_sin_cache(self, seq_len, device, dtype):
        self.max_seq_len_cached = seq_len
        t = torch.arange(self.max_seq_len_cached, device=device, dtype=self.inv_freq.dtype)
        t = t / self.scaling_factor

        freqs = torch.einsum("i,j->ij", t, self.inv_freq)
        # Different from paper, but it uses a different permutation in order to obtain the same calculation
        emb = torch.cat((freqs, freqs), dim=-1)
        self.register_buffer("cos_cached", emb.cos().to(dtype), persistent=False)
        self.register_buffer("sin_cached", emb.sin().to(dtype), persistent=False)


class LlamaDynamicNTKScalingRotaryEmbedding(LlamaRotaryEmbedding):
    """LlamaRotaryEmbedding extended with Dynamic NTK scaling. Credits to the Reddit users /u/bloc97 and /u/emozilla"""

    def __init__(self, dim, max_position_embeddings=2048, base=10000, device=None, scaling_factor=1.0):
        self.scaling_factor = scaling_factor
        super().__init__(dim, max_position_embeddings, base, device)

    def _set_cos_sin_cache(self, seq_len, device, dtype):
        self.max_seq_len_cached = seq_len

        if seq_len > self.max_position_embeddings:
            base = self.base * (
                (self.scaling_factor * seq_len / self.max_position_embeddings) - (self.scaling_factor - 1)
            ) ** (self.dim / (self.dim - 2))
            inv_freq = 1.0 / (base ** (torch.arange(0, self.dim, 2).float().to(device) / self.dim))
            self.register_buffer("inv_freq", inv_freq, persistent=False)

        t = torch.arange(self.max_seq_len_cached, device=device, dtype=self.inv_freq.dtype)

        freqs = torch.einsum("i,j->ij", t, self.inv_freq)
        # Different from paper, but it uses a different permutation in order to obtain the same calculation
        emb = torch.cat((freqs, freqs), dim=-1)
        self.register_buffer("cos_cached", emb.cos().to(dtype), persistent=False)
        self.register_buffer("sin_cached", emb.sin().to(dtype), persistent=False)


def rotate_half(x):
    """Rotates half the hidden dims of the input."""
    x1 = x[..., : x.shape[-1] // 2]
    x2 = x[..., x.shape[-1] // 2 :]
    return torch.cat((-x2, x1), dim=-1)


def apply_rotary_pos_emb(q, k, cos, sin, position_ids, unsqueeze_dim=1):
    """Applies Rotary Position Embedding to the query and key tensors.

    Args:
        q (`torch.Tensor`): The query tensor.
        k (`torch.Tensor`): The key tensor.
        cos (`torch.Tensor`): The cosine part of the rotary embedding.
        sin (`torch.Tensor`): The sine part of the rotary embedding.
        position_ids (`torch.Tensor`):
            The position indices of the tokens corresponding to the query and key tensors. For example, this can be
            used to pass offsetted position ids when working with a KV-cache.
        unsqueeze_dim (`int`, *optional*, defaults to 1):
            The 'unsqueeze_dim' argument specifies the dimension along which to unsqueeze cos[position_ids] and
            sin[position_ids] so that they can be properly broadcasted to the dimensions of q and k. For example, note
            that cos[position_ids] and sin[position_ids] have the shape [batch_size, seq_len, head_dim]. Then, if q and
            k have the shape [batch_size, heads, seq_len, head_dim], then setting unsqueeze_dim=1 makes
            cos[position_ids] and sin[position_ids] broadcastable to the shapes of q and k. Similarly, if q and k have
            the shape [batch_size, seq_len, heads, head_dim], then set unsqueeze_dim=2.
    Returns:
        `tuple(torch.Tensor)` comprising of the query and key tensors rotated using the Rotary Position Embedding.
    """
    cos = cos[position_ids].unsqueeze(unsqueeze_dim)
    sin = sin[position_ids].unsqueeze(unsqueeze_dim)
    q_embed = (q * cos) + (rotate_half(q) * sin)
    k_embed = (k * cos) + (rotate_half(k) * sin)
    return q_embed, k_embed


class LlamaMLP(nn.Module):
    def __init__(self, config):
        super().__init__()
        self.config = config
        self.hidden_size = config.hidden_size
        self.intermediate_size = config.intermediate_size
        self.gate_proj = nn.Linear(self.hidden_size, self.intermediate_size, bias=False)
        self.up_proj = nn.Linear(self.hidden_size, self.intermediate_size, bias=False)
        self.down_proj = nn.Linear(self.intermediate_size, self.hidden_size, bias=False)
        self.act_fn = ACT2FN[config.hidden_act]

    def forward(self, x):
        if self.config.pretraining_tp > 1:
            slice = self.intermediate_size // self.config.pretraining_tp
            gate_proj_slices = self.gate_proj.weight.split(slice, dim=0)
            up_proj_slices = self.up_proj.weight.split(slice, dim=0)
            down_proj_slices = self.down_proj.weight.split(slice, dim=1)

            gate_proj = torch.cat(
                [F.linear(x, gate_proj_slices[i]) for i in range(self.config.pretraining_tp)], dim=-1
            )
            up_proj = torch.cat([F.linear(x, up_proj_slices[i]) for i in range(self.config.pretraining_tp)], dim=-1)

            intermediate_states = (self.act_fn(gate_proj) * up_proj).split(slice, dim=2)
            down_proj = [
                F.linear(intermediate_states[i], down_proj_slices[i]) for i in range(self.config.pretraining_tp)
            ]
            down_proj = sum(down_proj)
        else:
            down_proj = self.down_proj(self.act_fn(self.gate_proj(x)) * self.up_proj(x))

        return down_proj


def repeat_kv(hidden_states: torch.Tensor, n_rep: int) -> torch.Tensor:
    """
    This is the equivalent of torch.repeat_interleave(x, dim=1, repeats=n_rep). The hidden states go from (batch,
    num_key_value_heads, seqlen, head_dim) to (batch, num_attention_heads, seqlen, head_dim)
    """
    batch, num_key_value_heads, slen, head_dim = hidden_states.shape
    if n_rep == 1:
        return hidden_states
    hidden_states = hidden_states[:, :, None, :, :].expand(batch, num_key_value_heads, n_rep, slen, head_dim)
    return hidden_states.reshape(batch, num_key_value_heads * n_rep, slen, head_dim)


class LlamaAttention(nn.Module):
    """Multi-headed attention from 'Attention Is All You Need' paper"""

    def __init__(self, config: LlamaConfig):
        super().__init__()
        self.config = config
        self.hidden_size = config.hidden_size
        self.num_heads = config.num_attention_heads
        self.head_dim = self.hidden_size // self.num_heads
        self.num_key_value_heads = config.num_key_value_heads
        self.num_key_value_groups = self.num_heads // self.num_key_value_heads
        self.max_position_embeddings = config.max_position_embeddings
        self.rope_theta = config.rope_theta
        self.is_causal = True

        if (self.head_dim * self.num_heads) != self.hidden_size:
            raise ValueError(
                f"hidden_size must be divisible by num_heads (got `hidden_size`: {self.hidden_size}"
                f" and `num_heads`: {self.num_heads})."
            )
        self.q_proj = nn.Linear(self.hidden_size, self.num_heads * self.head_dim, bias=config.attention_bias)
        self.k_proj = nn.Linear(self.hidden_size, self.num_key_value_heads * self.head_dim, bias=config.attention_bias)
        self.v_proj = nn.Linear(self.hidden_size, self.num_key_value_heads * self.head_dim, bias=config.attention_bias)
        self.o_proj = nn.Linear(self.num_heads * self.head_dim, self.hidden_size, bias=config.attention_bias)
        self._init_rope()
        self.att_out = nn.Identity()
        self.value_out = nn.Identity()
        self.head_out = nn.Identity()




    def _init_rope(self):
        if self.config.rope_scaling is None:
            self.rotary_emb = LlamaRotaryEmbedding(
                self.head_dim,
                max_position_embeddings=self.max_position_embeddings,
                base=self.rope_theta,
            )
        else:
            scaling_type = self.config.rope_scaling["type"]
            scaling_factor = self.config.rope_scaling["factor"]
            if scaling_type == "linear":
                self.rotary_emb = LlamaLinearScalingRotaryEmbedding(
                    self.head_dim,
                    max_position_embeddings=self.max_position_embeddings,
                    scaling_factor=scaling_factor,
                    base=self.rope_theta,
                )
            elif scaling_type == "dynamic":
                self.rotary_emb = LlamaDynamicNTKScalingRotaryEmbedding(
                    self.head_dim,
                    max_position_embeddings=self.max_position_embeddings,
                    scaling_factor=scaling_factor,
                    base=self.rope_theta,
                )
            else:
                raise ValueError(f"Unknown RoPE scaling type {scaling_type}")

    def _shape(self, tensor: torch.Tensor, seq_len: int, bsz: int):
        return tensor.view(bsz, seq_len, self.num_heads, self.head_dim).transpose(1, 2).contiguous()

    def forward(
        self,
        hidden_states: torch.Tensor,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_value: Optional[Tuple[torch.Tensor]] = None,
        output_attentions: bool = False,
        use_cache: bool = False,
        **kwargs,
    ) -> Tuple[torch.Tensor, Optional[torch.Tensor], Optional[Tuple[torch.Tensor]]]:
        if "padding_mask" in kwargs:
            warnings.warn(
                "Passing `padding_mask` is deprecated and will be removed in v4.37. Please make sure use `attention_mask` instead.`"
            )

        bsz, q_len, _ = hidden_states.size()

        if self.config.pretraining_tp > 1:
            key_value_slicing = (self.num_key_value_heads * self.head_dim) // self.config.pretraining_tp
            query_slices = self.q_proj.weight.split(
                (self.num_heads * self.head_dim) // self.config.pretraining_tp, dim=0
            )
            key_slices = self.k_proj.weight.split(key_value_slicing, dim=0)
            value_slices = self.v_proj.weight.split(key_value_slicing, dim=0)

            query_states = [F.linear(hidden_states, query_slices[i]) for i in range(self.config.pretraining_tp)]
            query_states = torch.cat(query_states, dim=-1)

            key_states = [F.linear(hidden_states, key_slices[i]) for i in range(self.config.pretraining_tp)]
            key_states = torch.cat(key_states, dim=-1)

            value_states = [F.linear(hidden_states, value_slices[i]) for i in range(self.config.pretraining_tp)]
            value_states = torch.cat(value_states, dim=-1)

        else:
            query_states = self.q_proj(hidden_states)
            key_states = self.k_proj(hidden_states)
            value_states = self.v_proj(hidden_states)

        query_states = query_states.view(bsz, q_len, self.num_heads, self.head_dim).transpose(1, 2)
        key_states = key_states.view(bsz, q_len, self.num_key_value_heads, self.head_dim).transpose(1, 2)
        value_states = value_states.view(bsz, q_len, self.num_key_value_heads, self.head_dim).transpose(1, 2)

        kv_seq_len = key_states.shape[-2]
        if past_key_value is not None:
            kv_seq_len += past_key_value[0].shape[-2]
        cos, sin = self.rotary_emb(value_states, seq_len=kv_seq_len)
        query_states, key_states = apply_rotary_pos_emb(query_states, key_states, cos, sin, position_ids)

        if past_key_value is not None:
            # reuse k, v, self_attention
            key_states = torch.cat([past_key_value[0], key_states], dim=2)
            value_states = torch.cat([past_key_value[1], value_states], dim=2)

        past_key_value = (key_states, value_states) if use_cache else None

        key_states = repeat_kv(key_states, self.num_key_value_groups)
        value_states = repeat_kv(value_states, self.num_key_value_groups)

        attn_weights = torch.matmul(query_states, key_states.transpose(2, 3)) / math.sqrt(self.head_dim)

        if attn_weights.size() != (bsz, self.num_heads, q_len, kv_seq_len):
            raise ValueError(
                f"Attention weights should be of size {(bsz, self.num_heads, q_len, kv_seq_len)}, but is"
                f" {attn_weights.size()}"
            )

        if attention_mask is not None:
            if attention_mask.size() != (bsz, 1, q_len, kv_seq_len):
                raise ValueError(
                    f"Attention mask should be of size {(bsz, 1, q_len, kv_seq_len)}, but is {attention_mask.size()}"
                )
            attn_weights = attn_weights + attention_mask

        # upcast attention to fp32
        attn_weights = nn.functional.softmax(attn_weights, dim=-1, dtype=torch.float32).to(query_states.dtype)
        attn_weights = self.att_out(attn_weights)
        value_states = self.value_out(value_states)
        attn_output = torch.matmul(attn_weights, value_states)

        if attn_output.size() != (bsz, self.num_heads, q_len, self.head_dim):
            raise ValueError(
                f"`attn_output` should be of size {(bsz, self.num_heads, q_len, self.head_dim)}, but is"
                f" {attn_output.size()}"
            )

        attn_output = attn_output.transpose(1, 2).contiguous()

        attn_output = attn_output.reshape(bsz, q_len, self.hidden_size)

        if self.config.pretraining_tp > 1:
            attn_output = attn_output.split(self.hidden_size // self.config.pretraining_tp, dim=2)
            o_proj_slices = self.o_proj.weight.split(self.hidden_size // self.config.pretraining_tp, dim=1)
            attn_output = sum([F.linear(attn_output[i], o_proj_slices[i]) for i in range(self.config.pretraining_tp)])
            attn_output = self.head_out(attn_output)
        else:
            attn_output = self.head_out(attn_output)
            attn_output = self.o_proj(attn_output)

        if not output_attentions:
            attn_weights = None

        return attn_output, attn_weights, past_key_value


class LlamaFlashAttention2(LlamaAttention):
    """
    Llama flash attention module. This module inherits from `LlamaAttention` as the weights of the module stays
    untouched. The only required change would be on the forward pass where it needs to correctly call the public API of
    flash attention and deal with padding tokens in case the input contains any of them.
    """

    def forward(
        self,
        hidden_states: torch.Tensor,
        attention_mask: Optional[torch.LongTensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_value: Optional[Tuple[torch.Tensor]] = None,
        output_attentions: bool = False,
        use_cache: bool = False,
        **kwargs,
    ) -> Tuple[torch.Tensor, Optional[torch.Tensor], Optional[Tuple[torch.Tensor]]]:
        # LlamaFlashAttention2 attention does not support output_attentions
        if "padding_mask" in kwargs:
            warnings.warn(
                "Passing `padding_mask` is deprecated and will be removed in v4.37. Please make sure use `attention_mask` instead.`"
            )

            # overwrite attention_mask with padding_mask
            attention_mask = kwargs.pop("padding_mask")

        output_attentions = False

        bsz, q_len, _ = hidden_states.size()

        query_states = self.q_proj(hidden_states)
        key_states = self.k_proj(hidden_states)
        value_states = self.v_proj(hidden_states)

        # Flash attention requires the input to have the shape
        # batch_size x seq_length x head_dim x hidden_dim
        # therefore we just need to keep the original shape
        query_states = query_states.view(bsz, q_len, self.num_heads, self.head_dim).transpose(1, 2)
        key_states = key_states.view(bsz, q_len, self.num_key_value_heads, self.head_dim).transpose(1, 2)
        value_states = value_states.view(bsz, q_len, self.num_key_value_heads, self.head_dim).transpose(1, 2)

        kv_seq_len = key_states.shape[-2]
        if past_key_value is not None:
            kv_seq_len += past_key_value[0].shape[-2]

        cos, sin = self.rotary_emb(value_states, seq_len=kv_seq_len)

        query_states, key_states = apply_rotary_pos_emb(query_states, key_states, cos, sin, position_ids)

        if past_key_value is not None:
            # reuse k, v, self_attention
            key_states = torch.cat([past_key_value[0], key_states], dim=2)
            value_states = torch.cat([past_key_value[1], value_states], dim=2)

        past_key_value = (key_states, value_states) if use_cache else None

        query_states = query_states.transpose(1, 2)
        key_states = key_states.transpose(1, 2)
        value_states = value_states.transpose(1, 2)

        # TODO: llama does not have dropout in the config??
        # It is recommended to use dropout with FA according to the docs
        # when training.
        dropout_rate = 0.0  # if not self.training else self.attn_dropout

        # In PEFT, usually we cast the layer norms in float32 for training stability reasons
        # therefore the input hidden states gets silently casted in float32. Hence, we need
        # cast them back in the correct dtype just to be sure everything works as expected.
        # This might slowdown training & inference so it is recommended to not cast the LayerNorms
        # in fp32. (LlamaRMSNorm handles it correctly)

        input_dtype = query_states.dtype
        if input_dtype == torch.float32:
            # Handle the case where the model is quantized
            if hasattr(self.config, "_pre_quantization_dtype"):
                target_dtype = self.config._pre_quantization_dtype
            else:
                target_dtype = self.q_proj.weight.dtype

            logger.warning_once(
                f"The input hidden states seems to be silently casted in float32, this might be related to"
                f" the fact you have upcasted embedding or layer norm layers in float32. We will cast back the input in"
                f" {target_dtype}."
            )

            query_states = query_states.to(target_dtype)
            key_states = key_states.to(target_dtype)
            value_states = value_states.to(target_dtype)

        attn_output = self._flash_attention_forward(
            query_states, key_states, value_states, attention_mask, q_len, dropout=dropout_rate
        )

        attn_output = attn_output.reshape(bsz, q_len, self.hidden_size).contiguous()
        attn_output = self.o_proj(attn_output)

        if not output_attentions:
            attn_weights = None

        return attn_output, attn_weights, past_key_value

    def _flash_attention_forward(
        self, query_states, key_states, value_states, attention_mask, query_length, dropout=0.0, softmax_scale=None
    ):
        """
        Calls the forward method of Flash Attention - if the input hidden states contain at least one padding token
        first unpad the input, then computes the attention scores and pad the final attention scores.

        Args:
            query_states (`torch.Tensor`):
                Input query states to be passed to Flash Attention API
            key_states (`torch.Tensor`):
                Input key states to be passed to Flash Attention API
            value_states (`torch.Tensor`):
                Input value states to be passed to Flash Attention API
            attention_mask (`torch.Tensor`):
                The padding mask - corresponds to a tensor of size `(batch_size, seq_len)` where 0 stands for the
                position of padding tokens and 1 for the position of non-padding tokens.
            dropout (`int`, *optional*):
                Attention dropout
            softmax_scale (`float`, *optional*):
                The scaling of QK^T before applying softmax. Default to 1 / sqrt(head_dim)
        """
        # Contains at least one padding token in the sequence
        if attention_mask is not None:
            batch_size = query_states.shape[0]
            query_states, key_states, value_states, indices_q, cu_seq_lens, max_seq_lens = self._upad_input(
                query_states, key_states, value_states, attention_mask, query_length
            )

            cu_seqlens_q, cu_seqlens_k = cu_seq_lens
            max_seqlen_in_batch_q, max_seqlen_in_batch_k = max_seq_lens

            attn_output_unpad = flash_attn_varlen_func(
                query_states,
                key_states,
                value_states,
                cu_seqlens_q=cu_seqlens_q,
                cu_seqlens_k=cu_seqlens_k,
                max_seqlen_q=max_seqlen_in_batch_q,
                max_seqlen_k=max_seqlen_in_batch_k,
                dropout_p=dropout,
                softmax_scale=softmax_scale,
                causal=self.is_causal,
            )

            attn_output = pad_input(attn_output_unpad, indices_q, batch_size, query_length)
        else:
            attn_output = flash_attn_func(
                query_states, key_states, value_states, dropout, softmax_scale=softmax_scale, causal=self.is_causal
            )

        return attn_output

    def _upad_input(self, query_layer, key_layer, value_layer, attention_mask, query_length):
        indices_k, cu_seqlens_k, max_seqlen_in_batch_k = _get_unpad_data(attention_mask)
        batch_size, kv_seq_len, num_key_value_heads, head_dim = key_layer.shape

        key_layer = index_first_axis(
            key_layer.reshape(batch_size * kv_seq_len, num_key_value_heads, head_dim), indices_k
        )
        value_layer = index_first_axis(
            value_layer.reshape(batch_size * kv_seq_len, num_key_value_heads, head_dim), indices_k
        )
        if query_length == kv_seq_len:
            query_layer = index_first_axis(
                query_layer.reshape(batch_size * kv_seq_len, self.num_heads, head_dim), indices_k
            )
            cu_seqlens_q = cu_seqlens_k
            max_seqlen_in_batch_q = max_seqlen_in_batch_k
            indices_q = indices_k
        elif query_length == 1:
            max_seqlen_in_batch_q = 1
            cu_seqlens_q = torch.arange(
                batch_size + 1, dtype=torch.int32, device=query_layer.device
            )  # There is a memcpy here, that is very bad.
            indices_q = cu_seqlens_q[:-1]
            query_layer = query_layer.squeeze(1)
        else:
            # The -q_len: slice assumes left padding.
            attention_mask = attention_mask[:, -query_length:]
            query_layer, indices_q, cu_seqlens_q, max_seqlen_in_batch_q = unpad_input(query_layer, attention_mask)

        return (
            query_layer,
            key_layer,
            value_layer,
            indices_q,
            (cu_seqlens_q, cu_seqlens_k),
            (max_seqlen_in_batch_q, max_seqlen_in_batch_k),
        )


class LlamaDecoderLayer(nn.Module):
    def __init__(self, config: LlamaConfig):
        super().__init__()
        self.hidden_size = config.hidden_size
        self.self_attn = (
            LlamaAttention(config=config)
            if not getattr(config, "_flash_attn_2_enabled", False)
            else LlamaFlashAttention2(config=config)
        )
        self.mlp = LlamaMLP(config)
        self.input_layernorm = LlamaRMSNorm(config.hidden_size, eps=config.rms_norm_eps)
        self.post_attention_layernorm = LlamaRMSNorm(config.hidden_size, eps=config.rms_norm_eps)

    def forward(
        self,
        hidden_states: torch.Tensor,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_value: Optional[Tuple[torch.Tensor]] = None,
        output_attentions: Optional[bool] = False,
        use_cache: Optional[bool] = False,
        **kwargs,
    ) -> Tuple[torch.FloatTensor, Optional[Tuple[torch.FloatTensor, torch.FloatTensor]]]:
        """
        Args:
            hidden_states (`torch.FloatTensor`): input to the layer of shape `(batch, seq_len, embed_dim)`
            attention_mask (`torch.FloatTensor`, *optional*):
                attention mask of size `(batch_size, sequence_length)` if flash attention is used or `(batch_size, 1,
                query_sequence_length, key_sequence_length)` if default attention is used.
            output_attentions (`bool`, *optional*):
                Whether or not to return the attentions tensors of all attention layers. See `attentions` under
                returned tensors for more detail.
            use_cache (`bool`, *optional*):
                If set to `True`, `past_key_values` key value states are returned and can be used to speed up decoding
                (see `past_key_values`).
            past_key_value (`Tuple(torch.FloatTensor)`, *optional*): cached past key and value projection states
        """
        if "padding_mask" in kwargs:
            warnings.warn(
                "Passing `padding_mask` is deprecated and will be removed in v4.37. Please make sure use `attention_mask` instead.`"
            )

        residual = hidden_states

        hidden_states = self.input_layernorm(hidden_states)

        # Self Attention
        hidden_states, self_attn_weights, present_key_value = self.self_attn(
            hidden_states=hidden_states,
            attention_mask=attention_mask,
            position_ids=position_ids,
            past_key_value=past_key_value,
            output_attentions=output_attentions,
            use_cache=use_cache,
            **kwargs,
        )
        hidden_states = residual + hidden_states

        # Fully Connected
        residual = hidden_states
        hidden_states = self.post_attention_layernorm(hidden_states)
        hidden_states = self.mlp(hidden_states)
        hidden_states = residual + hidden_states

        outputs = (hidden_states,)

        if output_attentions:
            outputs += (self_attn_weights,)

        if use_cache:
            outputs += (present_key_value,)

        return outputs


LLAMA_START_DOCSTRING = r"""
    This model inherits from [`PreTrainedModel`]. Check the superclass documentation for the generic methods the
    library implements for all its model (such as downloading or saving, resizing the input embeddings, pruning heads
    etc.)

    This model is also a PyTorch [torch.nn.Module](https://pytorch.org/docs/stable/nn.html#torch.nn.Module) subclass.
    Use it as a regular PyTorch Module and refer to the PyTorch documentation for all matter related to general usage
    and behavior.

    Parameters:
        config ([`LlamaConfig`]):
            Model configuration class with all the parameters of the model. Initializing with a config file does not
            load the weights associated with the model, only the configuration. Check out the
            [`~PreTrainedModel.from_pretrained`] method to load the model weights.
"""


@add_start_docstrings(
    "The bare LLaMA Model outputting raw hidden-states without any specific head on top.",
    LLAMA_START_DOCSTRING,
)
class LlamaPreTrainedModel(PreTrainedModel):
    config_class = LlamaConfig
    base_model_prefix = "model"
    supports_gradient_checkpointing = True
    _no_split_modules = ["LlamaDecoderLayer"]
    _skip_keys_device_placement = "past_key_values"
    _supports_flash_attn_2 = True

    def _init_weights(self, module):
        std = self.config.initializer_range
        if isinstance(module, nn.Linear):
            module.weight.data.normal_(mean=0.0, std=std)
            if module.bias is not None:
                module.bias.data.zero_()
        elif isinstance(module, nn.Embedding):
            module.weight.data.normal_(mean=0.0, std=std)
            if module.padding_idx is not None:
                module.weight.data[module.padding_idx].zero_()


LLAMA_INPUTS_DOCSTRING = r"""
    Args:
        input_ids (`torch.LongTensor` of shape `(batch_size, sequence_length)`):
            Indices of input sequence tokens in the vocabulary. Padding will be ignored by default should you provide
            it.

            Indices can be obtained using [`AutoTokenizer`]. See [`PreTrainedTokenizer.encode`] and
            [`PreTrainedTokenizer.__call__`] for details.

            [What are input IDs?](../glossary#input-ids)
        attention_mask (`torch.Tensor` of shape `(batch_size, sequence_length)`, *optional*):
            Mask to avoid performing attention on padding token indices. Mask values selected in `[0, 1]`:

            - 1 for tokens that are **not masked**,
            - 0 for tokens that are **masked**.

            [What are attention masks?](../glossary#attention-mask)

            Indices can be obtained using [`AutoTokenizer`]. See [`PreTrainedTokenizer.encode`] and
            [`PreTrainedTokenizer.__call__`] for details.

            If `past_key_values` is used, optionally only the last `input_ids` have to be input (see
            `past_key_values`).

            If you want to change padding behavior, you should read [`modeling_opt._prepare_decoder_attention_mask`]
            and modify to your needs. See diagram 1 in [the paper](https://arxiv.org/abs/1910.13461) for more
            information on the default strategy.

            - 1 indicates the head is **not masked**,
            - 0 indicates the head is **masked**.
        position_ids (`torch.LongTensor` of shape `(batch_size, sequence_length)`, *optional*):
            Indices of positions of each input sequence tokens in the position embeddings. Selected in the range `[0,
            config.n_positions - 1]`.

            [What are position IDs?](../glossary#position-ids)
        past_key_values (`tuple(tuple(torch.FloatTensor))`, *optional*, returned when `use_cache=True` is passed or when `config.use_cache=True`):
            Tuple of `tuple(torch.FloatTensor)` of length `config.n_layers`, with each tuple having 2 tensors of shape
            `(batch_size, num_heads, sequence_length, embed_size_per_head)`) and 2 additional tensors of shape
            `(batch_size, num_heads, decoder_sequence_length, embed_size_per_head)`.

            Contains pre-computed hidden-states (key and values in the self-attention blocks and in the cross-attention
            blocks) that can be used (see `past_key_values` input) to speed up sequential decoding.

            If `past_key_values` are used, the user can optionally input only the last `input_ids` (those that don't
            have their past key value states given to this model) of shape `(batch_size, 1)` instead of all `input_ids`
            of shape `(batch_size, sequence_length)`.
        inputs_embeds (`torch.FloatTensor` of shape `(batch_size, sequence_length, hidden_size)`, *optional*):
            Optionally, instead of passing `input_ids` you can choose to directly pass an embedded representation. This
            is useful if you want more control over how to convert `input_ids` indices into associated vectors than the
            model's internal embedding lookup matrix.
        use_cache (`bool`, *optional*):
            If set to `True`, `past_key_values` key value states are returned and can be used to speed up decoding (see
            `past_key_values`).
        output_attentions (`bool`, *optional*):
            Whether or not to return the attentions tensors of all attention layers. See `attentions` under returned
            tensors for more detail.
        output_hidden_states (`bool`, *optional*):
            Whether or not to return the hidden states of all layers. See `hidden_states` under returned tensors for
            more detail.
        return_dict (`bool`, *optional*):
            Whether or not to return a [`~utils.ModelOutput`] instead of a plain tuple.
"""


@add_start_docstrings(
    "The bare LLaMA Model outputting raw hidden-states without any specific head on top.",
    LLAMA_START_DOCSTRING,
)
class LlamaModel(LlamaPreTrainedModel):
    """
    Transformer decoder consisting of *config.num_hidden_layers* layers. Each layer is a [`LlamaDecoderLayer`]

    Args:
        config: LlamaConfig
    """

    def __init__(self, config: LlamaConfig):
        super().__init__(config)
        self.config = config
        self.padding_idx = config.pad_token_id
        self.vocab_size = config.vocab_size

        self.embed_tokens = nn.Embedding(config.vocab_size, config.hidden_size, self.padding_idx)
        self.layers = nn.ModuleList([LlamaDecoderLayer(config) for _ in range(config.num_hidden_layers)])
        self.norm = LlamaRMSNorm(config.hidden_size, eps=config.rms_norm_eps)

        self.use_agg = config.use_agg
        
        if self.config.bank_intervention_dir != "":
            # construct intervention parameters
            with open(config.evidences_path, 'rb') as file:
                self.bank_evidences = pickle.load(file)
                self.evidences_len = len(self.bank_evidences)
                self.evidences2id = defaultdict(int)
                for idx, evidence in enumerate(self.bank_evidences):
                    self.evidences2id[evidence] = idx


            with open(config.inter_heads_path, 'rb') as file:
                self.intervention_heads = pickle.load(file)

            if not self.use_agg:
                candidateintervention_heads = range(config.num_hidden_layers * config.num_attention_heads)
                self.valid_intervention_heads = self.intervention_heads
                self.intervention_heads = random.sample(candidateintervention_heads, len(self.valid_intervention_heads))

            self.interHead_len = len(self.intervention_heads)
            self.head_out_intervention = nn.Linear(self.evidences_len * self.interHead_len, 
                                                (config.hidden_size // config.num_attention_heads), bias=False)

            print(f"evi_len={self.evidences_len}, interhead_len={self.interHead_len}")

        self.gradient_checkpointing = False
        # Initialize weights and apply final processing
        self.post_init()

    @staticmethod
    def flattened_idx_to_layer_head(flattened_idx, num_heads=32):
        return flattened_idx // num_heads, flattened_idx % num_heads
    @staticmethod
    def layer_head_to_flattened_idx(layer, head, num_heads=32):
        return layer * num_heads + head


    def get_input_embeddings(self):
        return self.embed_tokens

    def set_input_embeddings(self, value):
        self.embed_tokens = value

    @add_start_docstrings_to_model_forward(LLAMA_INPUTS_DOCSTRING)
    def forward(
        self,
        input_ids: torch.LongTensor = None,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_values: Optional[List[torch.FloatTensor]] = None,
        inputs_embeds: Optional[torch.FloatTensor] = None,
        use_cache: Optional[bool] = None,
        output_attentions: Optional[bool] = None,
        output_hidden_states: Optional[bool] = None,
        return_dict: Optional[bool] = None,
    ) -> Union[Tuple, BaseModelOutputWithPast]:
        output_attentions = output_attentions if output_attentions is not None else self.config.output_attentions
        output_hidden_states = (
            output_hidden_states if output_hidden_states is not None else self.config.output_hidden_states
        )
        use_cache = use_cache if use_cache is not None else self.config.use_cache

        return_dict = return_dict if return_dict is not None else self.config.use_return_dict

        # retrieve input_ids and inputs_embeds
        if input_ids is not None and inputs_embeds is not None:
            raise ValueError("You cannot specify both input_ids and inputs_embeds at the same time")
        elif input_ids is not None:
            batch_size, seq_length = input_ids.shape[:2]
        elif inputs_embeds is not None:
            batch_size, seq_length = inputs_embeds.shape[:2]
        else:
            raise ValueError("You have to specify either input_ids or inputs_embeds")

        past_key_values_length = 0
        if past_key_values is not None:
            past_key_values_length = past_key_values[0][0].shape[2]

        if position_ids is None:
            device = input_ids.device if input_ids is not None else inputs_embeds.device
            position_ids = torch.arange(
                past_key_values_length, seq_length + past_key_values_length, dtype=torch.long, device=device
            )
            position_ids = position_ids.unsqueeze(0)

        if inputs_embeds is None:
            inputs_embeds = self.embed_tokens(input_ids)

        if getattr(self.config, "_flash_attn_2_enabled", False):
            # 2d mask is passed through the layers
            attention_mask = attention_mask if (attention_mask is not None and 0 in attention_mask) else None
        else:
            # 4d mask is passed through the layers
            attention_mask = _prepare_4d_causal_attention_mask(
                attention_mask, (batch_size, seq_length), inputs_embeds, past_key_values_length
            )

        # embed positions
        hidden_states = inputs_embeds

        if self.gradient_checkpointing and self.training:
            if use_cache:
                logger.warning_once(
                    "`use_cache=True` is incompatible with gradient checkpointing. Setting `use_cache=False`..."
                )
                use_cache = False

        # decoder layers
        all_hidden_states = () if output_hidden_states else None
        all_self_attns = () if output_attentions else None
        next_decoder_cache = () if use_cache else None

        for idx, decoder_layer in enumerate(self.layers):
            if output_hidden_states:
                all_hidden_states += (hidden_states,)

            past_key_value = past_key_values[idx] if past_key_values is not None else None

            if self.gradient_checkpointing and self.training:
                layer_outputs = self._gradient_checkpointing_func(
                    decoder_layer.__call__,
                    hidden_states,
                    attention_mask,
                    position_ids,
                    past_key_value,
                    output_attentions,
                    use_cache,
                )
            else:
                layer_outputs = decoder_layer(
                    hidden_states,
                    attention_mask=attention_mask,
                    position_ids=position_ids,
                    past_key_value=past_key_value,
                    output_attentions=output_attentions,
                    use_cache=use_cache,
                )

            hidden_states = layer_outputs[0]

            if use_cache:
                next_decoder_cache += (layer_outputs[2 if output_attentions else 1],)

            if output_attentions:
                all_self_attns += (layer_outputs[1],)

        hidden_states = self.norm(hidden_states)

        # add hidden states from the last decoder layer
        if output_hidden_states:
            all_hidden_states += (hidden_states,)

        next_cache = next_decoder_cache if use_cache else None
        if not return_dict:
            return tuple(v for v in [hidden_states, next_cache, all_hidden_states, all_self_attns] if v is not None)
        return BaseModelOutputWithPast(
            last_hidden_state=hidden_states,
            past_key_values=next_cache,
            hidden_states=all_hidden_states,
            attentions=all_self_attns,
        )


class LlamaForCausalLM(LlamaPreTrainedModel):
    _tied_weights_keys = ["lm_head.weight"]

    def __init__(self, config):
        super().__init__(config)
        self.model = LlamaModel(config)
        self.vocab_size = config.vocab_size
        self.lm_head = nn.Linear(config.hidden_size, config.vocab_size, bias=False)
        self.config = config

        # Initialize weights and apply final processing
        self.post_init()
    
    @staticmethod
    def flattened_idx_to_layer_head(flattened_idx, num_heads=32):
        return flattened_idx // num_heads, flattened_idx % num_heads
    @staticmethod
    def layer_head_to_flattened_idx(layer, head, num_heads=32):
        return layer * num_heads + head
    
    def _init_weights(self, module):

        def get_intervention(top_heads, com_directions, num_heads): 
            interventions =[]
            for head in top_heads:
                direction = com_directions[head]
                direction = direction / np.linalg.norm(direction)
                # print(direction.shape)
                interventions.append(torch.tensor(direction, dtype=torch.float16))
            
            return torch.stack(interventions, dim = 0)

        if self.config.bank_intervention_dir != "":
            interventions_list = [None for i in range(self.model.evidences_len)]
            for idx, evidence in enumerate(self.model.bank_evidences):
                symptom_dir = os.path.join(self.config.bank_intervention_dir, f"{evidence.lower()}/")
                directions_path = os.path.join(symptom_dir, 'directions.npy')
                symptom_com_directions = np.load(directions_path)
                evidence_intervention = get_intervention(self.model.intervention_heads, symptom_com_directions, \
                                                        self.config.hidden_size // self.config.num_attention_heads)
                interventions_list[self.model.evidences2id[evidence]] = evidence_intervention
            interventions_tensor = torch.cat(interventions_list, dim=0).transpose(1, 0)
            self.model.head_out_intervention.load_state_dict({
                "weight": interventions_tensor
            })
    
    def get_input_embeddings(self):
        return self.model.embed_tokens

    def set_input_embeddings(self, value):
        self.model.embed_tokens = value

    def get_output_embeddings(self):
        return self.lm_head

    def set_output_embeddings(self, new_embeddings):
        self.lm_head = new_embeddings

    def set_decoder(self, decoder):
        self.model = decoder

    def get_decoder(self):
        return self.model

    @add_start_docstrings_to_model_forward(LLAMA_INPUTS_DOCSTRING)
    @replace_return_docstrings(output_type=CausalLMOutputWithPast, config_class=_CONFIG_FOR_DOC)
    def forward(
        self,
        input_ids: torch.LongTensor = None,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_values: Optional[List[torch.FloatTensor]] = None,
        inputs_embeds: Optional[torch.FloatTensor] = None,
        labels: Optional[torch.LongTensor] = None,
        use_cache: Optional[bool] = None,
        output_attentions: Optional[bool] = None,
        output_hidden_states: Optional[bool] = None,
        return_dict: Optional[bool] = None,
    ) -> Union[Tuple, CausalLMOutputWithPast]:
        r"""
        Args:
            labels (`torch.LongTensor` of shape `(batch_size, sequence_length)`, *optional*):
                Labels for computing the masked language modeling loss. Indices should either be in `[0, ...,
                config.vocab_size]` or -100 (see `input_ids` docstring). Tokens with indices set to `-100` are ignored
                (masked), the loss is only computed for the tokens with labels in `[0, ..., config.vocab_size]`.

        Returns:

        Example:

        ```python
        >>> from transformers import AutoTokenizer, LlamaForCausalLM

        >>> model = LlamaForCausalLM.from_pretrained(PATH_TO_CONVERTED_WEIGHTS)
        >>> tokenizer = AutoTokenizer.from_pretrained(PATH_TO_CONVERTED_TOKENIZER)

        >>> prompt = "Hey, are you conscious? Can you talk to me?"
        >>> inputs = tokenizer(prompt, return_tensors="pt")

        >>> # Generate
        >>> generate_ids = model.generate(inputs.input_ids, max_length=30)
        >>> tokenizer.batch_decode(generate_ids, skip_special_tokens=True, clean_up_tokenization_spaces=False)[0]
        "Hey, are you conscious? Can you talk to me?\nI'm not conscious, but I can talk to you."
        ```"""

        output_attentions = output_attentions if output_attentions is not None else self.config.output_attentions
        output_hidden_states = (
            output_hidden_states if output_hidden_states is not None else self.config.output_hidden_states
        )
        return_dict = return_dict if return_dict is not None else self.config.use_return_dict

        # decoder outputs consists of (dec_features, layer_state, dec_hidden, dec_attn)
        outputs = self.model(
            input_ids=input_ids,
            attention_mask=attention_mask,
            position_ids=position_ids,
            past_key_values=past_key_values,
            inputs_embeds=inputs_embeds,
            use_cache=use_cache,
            output_attentions=output_attentions,
            output_hidden_states=output_hidden_states,
            return_dict=return_dict,
        )

        hidden_states = outputs[0]
        if self.config.pretraining_tp > 1:
            lm_head_slices = self.lm_head.weight.split(self.vocab_size // self.config.pretraining_tp, dim=0)
            logits = [F.linear(hidden_states, lm_head_slices[i]) for i in range(self.config.pretraining_tp)]
            logits = torch.cat(logits, dim=-1)
        else:
            logits = self.lm_head(hidden_states)
        logits = logits.float()

        loss = None
        if labels is not None:
            # Shift so that tokens < n predict n
            shift_logits = logits[..., :-1, :].contiguous()
            shift_labels = labels[..., 1:].contiguous()
            # Flatten the tokens
            loss_fct = CrossEntropyLoss()
            shift_logits = shift_logits.view(-1, self.config.vocab_size)
            shift_labels = shift_labels.view(-1)
            # Enable model parallelism
            shift_labels = shift_labels.to(shift_logits.device)
            loss = loss_fct(shift_logits, shift_labels)

        if not return_dict:
            output = (logits,) + outputs[1:]
            return (loss,) + output if loss is not None else output

        return CausalLMOutputWithPast(
            loss=loss,
            logits=logits,
            past_key_values=outputs.past_key_values,
            hidden_states=outputs.hidden_states,
            attentions=outputs.attentions,
        )

    def prepare_inputs_for_generation(
        self, input_ids, past_key_values=None, attention_mask=None, inputs_embeds=None, **kwargs
    ):
        if past_key_values is not None:
            past_length = past_key_values[0][0].shape[2]

            # Some generation methods already pass only the last input ID
            if input_ids.shape[1] > past_length:
                remove_prefix_length = past_length
            else:
                # Default to old behavior: keep only final ID
                remove_prefix_length = input_ids.shape[1] - 1

            input_ids = input_ids[:, remove_prefix_length:]

        position_ids = kwargs.get("position_ids", None)
        if attention_mask is not None and position_ids is None:
            # create position_ids on the fly for batch generation
            position_ids = attention_mask.long().cumsum(-1) - 1
            position_ids.masked_fill_(attention_mask == 0, 1)
            if past_key_values:
                position_ids = position_ids[:, -input_ids.shape[1] :]

        # if `inputs_embeds` are passed, we only want to use them in the 1st generation step
        if inputs_embeds is not None and past_key_values is None:
            model_inputs = {"inputs_embeds": inputs_embeds}
        else:
            model_inputs = {"input_ids": input_ids}

        model_inputs.update(
            {
                "position_ids": position_ids,
                "past_key_values": past_key_values,
                "use_cache": kwargs.get("use_cache"),
                "attention_mask": attention_mask,
            }
        )
        return model_inputs

    @staticmethod
    def _reorder_cache(past_key_values, beam_idx):
        reordered_past = ()
        for layer_past in past_key_values:
            reordered_past += (
                tuple(past_state.index_select(0, beam_idx.to(past_state.device)) for past_state in layer_past),
            )
        return reordered_past


    def get_attention_diff(self, auxiliary_model, tokenizer, restore_symptoms, keep_symptoms, question_init, device):


        # restore symptoms activations 
        restore_begin_index = []
        restore_end_index = []
        re_evi_attention = [0 for i in range(len(restore_symptoms))]

        for symptom in restore_symptoms:
            restore_begin_index.append(len(tokenizer.tokenize(
                question_init.split(symptom)[0])))
            restore_end_index.append(restore_begin_index[-1] + len(tokenizer.tokenize(symptom)))
        
        input_ids = tokenizer(question_init, return_tensors="pt").input_ids.to(device)
        output = auxiliary_model.forward(input_ids, output_attentions=True)
        attentions = output['attentions']

        for idx, (b_id, e_id) in enumerate(zip(restore_begin_index, restore_end_index)):
            all_symptom_re_attention = []
            for layer_index, layer_attention in enumerate(attentions):
                layer_symptom_re_attention = torch.sum(layer_attention[:, :, :, b_id:e_id], dim=-1)
                layer_symptom_re_attention = torch.sum(layer_symptom_re_attention, dim=-1) # batch_size, head_count
                layer_symptom_re_attention = torch.mean(layer_symptom_re_attention, dim = -1) # batch_size
                all_symptom_re_attention.append(layer_symptom_re_attention)

            all_symptom_re_attention = torch.stack(all_symptom_re_attention, dim=-1)
            re_evi_attention[idx] = torch.mean(all_symptom_re_attention, dim=-1)[0].detach().cpu().numpy()

        softmax_re_evi_attention = softmax(re_evi_attention)

        # keep symptoms activations
        if len(keep_symptoms) == 0:
            return softmax_re_evi_attention, 1
        keep_begin_index = []
        keep_end_index = []
        kp_evi_attention = [0 for i in range(len(keep_symptoms))]
        for symptom in keep_symptoms:
            keep_begin_index.append(len(tokenizer.tokenize(
                question_init.split(symptom)[0])))
            keep_end_index.append(keep_begin_index[-1] + len(tokenizer.tokenize(symptom)))

        for idx, (b_id, e_id) in enumerate(zip(keep_begin_index, keep_end_index)):
            all_symptom_kp_attention = []
            for layer_index, layer_attention in enumerate(attentions):
                layer_symptom_kp_attention = torch.sum(layer_attention[:, :, :, b_id:e_id], dim=-1)
                layer_symptom_kp_attention = torch.sum(layer_symptom_kp_attention, dim=-1) # batch_size, head_count
                layer_symptom_kp_attention = torch.mean(layer_symptom_kp_attention, dim = -1) # batch_size
                all_symptom_kp_attention.append(layer_symptom_kp_attention)

            all_symptom_kp_attention = torch.stack(all_symptom_kp_attention, dim=-1)
            kp_evi_attention[idx] = torch.mean(all_symptom_kp_attention, dim=-1)[0].detach().cpu().numpy()

        mutiplier = np.mean(re_evi_attention) / np.mean(kp_evi_attention)
        
        return softmax_re_evi_attention, mutiplier


    #====================
    # 训练forward
    # v1: 因为每个样本涉及的intervention不同，所以训练的时候只能设置batch=1
    #====================
    def compute_logps(self, input_ids, output_logits, labels, qlen):
        per_token_logps = torch.gather(output_logits[:, qlen-1:-1, :].log_softmax(-1), dim=2, 
                                       index=(labels[:, qlen:]).unsqueeze(2)).squeeze(2)
        return torch.sum(per_token_logps, dim=1).to(dtype=torch.float64) / per_token_logps.shape[1]

    def forward_intervention_orpo(
            self,
            input = None,
            tokenizer = None,
            device = None,
            auxiliary_model = None,
            auxiliary_tokenizer=None
    ):
        
        re_evi_attention, multiplier \
            = self.get_attention_diff(auxiliary_model, auxiliary_tokenizer, \
                                 input['restore_evidences'], input['keep_evidences'], \
                                 input['question_init'], device)
        
        layers_to_intervene = []
        for flattened_idx in self.model.intervention_heads:
            layer, head = LlamaForCausalLM.flattened_idx_to_layer_head(flattened_idx, self.config.num_attention_heads)
            layers_to_intervene.append(f"model.layers.{layer}.self_attn.head_out")
        layers_to_intervene = list(set(layers_to_intervene))

        intervene = partial(lt_modulated_vector_add_new_direction, llamaModel=self, restore_evidences=input['restore_evidences'], \
                                re_evi_attention= re_evi_attention, multiplier = multiplier)

        # Orpo based on pos and neg
        user_text = format_questions_mc(input['question_mask'])
        output_init_text = input['output_init']
        output_mask_text = input['output_mask']

        user_ids = tokenizer.encode(user_text)
        output_init_ids = tokenizer.encode(output_init_text)
        output_mask_ids = tokenizer.encode(output_mask_text)

        pos_input_ids = user_ids + output_init_ids
        pos_labels = [-100] * len(user_ids) + output_init_ids

        neg_input_ids = user_ids + output_mask_ids
        neg_labels = [-100] * len(user_ids) + output_mask_ids

        with TraceDict(self, layers_to_intervene, edit_output=intervene) as ret: 
            outputs_pos = self(
                input_ids = torch.tensor(pos_input_ids).to(device).unsqueeze(0),
                labels = torch.tensor(pos_labels).to(device).unsqueeze(0),
                output_hidden_states = True
            )

            output_neg = self(
                input_ids = torch.tensor(neg_input_ids).to(device).unsqueeze(0),
                labels = torch.tensor(neg_labels).to(device).unsqueeze(0),
                output_hidden_states = True
            )
        pos_loss = outputs_pos.loss
        # calculate Log Probability
        pos_prob = self.compute_logps(
            input_ids=torch.tensor(pos_input_ids).to(device).unsqueeze(0),
            labels=torch.tensor(pos_labels).to(device).unsqueeze(0),
            output_logits=outputs_pos['logits'],
            qlen = len(user_ids)
        )
        neg_prob = self.compute_logps(
            input_ids=torch.tensor(neg_input_ids).to(device).unsqueeze(0),
            labels=torch.tensor(neg_labels).to(device).unsqueeze(0),
            output_logits=output_neg['logits'],
            qlen = len(user_ids)
        )
        # Calculate log odds
        log_odds = (pos_prob - neg_prob) - (torch.log(1 - torch.exp(pos_prob)) - torch.log(1 - torch.exp(neg_prob)))
        sig_ratio = torch.nn.functional.sigmoid(log_odds)
        ratio = torch.log(sig_ratio)
        self.alpha = 1.0
        # Calculate the Final Loss
        orpo_loss = torch.mean(pos_loss - self.alpha * ratio).to(dtype=torch.float32)



        # LM loss based on correct answer
        output_correct_ans_ids = tokenizer.encode(input['correct_answer'])
        LM_input_ids = user_ids + output_correct_ans_ids
        LM_labels = [-100] * len(user_ids) + output_correct_ans_ids
        with TraceDict(self, layers_to_intervene, edit_output=intervene) as ret: 
            output_LM = self(
                input_ids = torch.tensor(LM_input_ids).to(device).unsqueeze(0),
                labels = torch.tensor(LM_labels).to(device).unsqueeze(0)
            )
        LM_loss = output_LM.loss

        # output_seed_ans_ids = tokenizer.encode(input['seed_evidence'])
        # LM_input_ids_extra = user_ids + output_seed_ans_ids
        # LM_labels_extra = [-100] * len(user_ids) + output_seed_ans_ids
        # with TraceDict(self, layers_to_intervene, edit_output=intervene) as ret: 
        #     output_LM_extra = self(
        #         input_ids = torch.tensor(LM_input_ids_extra).to(device).unsqueeze(0),
        #         labels = torch.tensor(LM_labels_extra).to(device).unsqueeze(0)
        #     )
        # LM_loss_extra = output_LM_extra.loss
        
        return orpo_loss, LM_loss


    def forward_intervention_orpo_client(
            self,
            input = None,
            tokenizer = None,
            device = None,
            auxiliary_model = None,
            auxiliary_tokenizer=None
    ):
        
        re_evi_attention, multiplier \
            = self.get_attention_diff(auxiliary_model, auxiliary_tokenizer, \
                                 input['restore_evidences'], input['keep_evidences'], \
                                 input['question_init'], device)
        layers_to_intervene = []
        for flattened_idx in self.model.intervention_heads:
            layer, head = LlamaForCausalLM.flattened_idx_to_layer_head(flattened_idx, self.config.num_attention_heads)
            layers_to_intervene.append(f"model.layers.{layer}.self_attn.head_out")
        layers_to_intervene = list(set(layers_to_intervene))
        # client
        client_directions = client_direction(self, restore_evidences=input['restore_evidences'], re_evi_attention=re_evi_attention,
                                             device = device)
        intervene = partial(lt_modulated_vector_add_new_direction_client, llamaModel=self, restore_evidences=input['restore_evidences'], \
                                re_evi_attention = re_evi_attention, multiplier = multiplier, client_vectors=client_directions)
        # Orpo based on pos and neg
        user_text = format_questions_mc(input['question_mask'])
        output_init_text = input['output_init']
        output_mask_text = input['output_mask']
        user_ids = tokenizer.encode(user_text)
        output_init_ids = tokenizer.encode(output_init_text)
        output_mask_ids = tokenizer.encode(output_mask_text)
        pos_input_ids = user_ids + output_init_ids
        pos_labels = [-100] * len(user_ids) + output_init_ids
        neg_input_ids = user_ids + output_mask_ids
        neg_labels = [-100] * len(user_ids) + output_mask_ids

        with TraceDict(self, layers_to_intervene, edit_output=intervene) as ret: 
            outputs_pos = self(
                input_ids = torch.tensor(pos_input_ids).to(device).unsqueeze(0),
                labels = torch.tensor(pos_labels).to(device).unsqueeze(0),
                output_hidden_states = True
            )

            output_neg = self(
                input_ids = torch.tensor(neg_input_ids).to(device).unsqueeze(0),
                labels = torch.tensor(neg_labels).to(device).unsqueeze(0),
                output_hidden_states = True
            )
        pos_loss = outputs_pos.loss
        # calculate Log Probability
        pos_prob = self.compute_logps(
            input_ids=torch.tensor(pos_input_ids).to(device).unsqueeze(0),
            labels=torch.tensor(pos_labels).to(device).unsqueeze(0),
            output_logits=outputs_pos['logits'],
            qlen = len(user_ids)
        )
        neg_prob = self.compute_logps(
            input_ids=torch.tensor(neg_input_ids).to(device).unsqueeze(0),
            labels=torch.tensor(neg_labels).to(device).unsqueeze(0),
            output_logits=output_neg['logits'],
            qlen = len(user_ids)
        )
        # Calculate log odds
        log_odds = (pos_prob - neg_prob) - (torch.log(1 - torch.exp(pos_prob)) - torch.log(1 - torch.exp(neg_prob)))
        sig_ratio = torch.nn.functional.sigmoid(log_odds)
        ratio = torch.log(sig_ratio)
        self.alpha = 1.0
        # Calculate the Final Loss
        orpo_loss = torch.mean(pos_loss - self.alpha * ratio).to(dtype=torch.float32)
        # LM loss based on correct answer
        output_correct_ans_ids = tokenizer.encode(input['correct_answer'])
        LM_input_ids = user_ids + output_correct_ans_ids
        LM_labels = [-100] * len(user_ids) + output_correct_ans_ids
        with TraceDict(self, layers_to_intervene, edit_output=intervene) as ret: 
            output_LM = self(
                input_ids = torch.tensor(LM_input_ids).to(device).unsqueeze(0),
                labels = torch.tensor(LM_labels).to(device).unsqueeze(0)
            )
        LM_loss = output_LM.loss
        return orpo_loss, LM_loss


    #====================
    # 计算选择题的指标
    # input: dict
    #   - restore_evidences
    #   - keep_evidences
    #   - question_init
    #   - question_mask
    #   - correct_answer
    #   - options
    #====================
    def MC_calcs(self, scores_true, scores_false):
        """Given model scores for true / false reference answers, calculates MC scores"""
        MC1=np.nan
        MC2=np.nan
        # compute MC1: 1vFalse -- best correct answer vs all false answers
        max_false = max(scores_false) if len(scores_false) > 0 else 0 
        if scores_true[0] > max_false:
            MC1 = 1.0
        else:
            MC1 = 0.0
        # compute MC2: normalized probability mass for correct answers
        probs_true = np.exp(scores_true)
        probs_false = np.exp(scores_false)
        probs_true = probs_true / (sum(probs_true) + sum(probs_false))
        MC2 = sum(probs_true)
        return MC1, MC2
    
    
    def base_infer_mc_score(
        self,
        question = None,
        correct_answer = None,
        false_answer = None,
        tokenizer = None,
        device = None
    ):
        input_prompt = format_questions_mc(question)

        ref_true = format_ans(correct_answer)
        ref_false_list = [format_ans(item) for item in false_answer]

        # true answer
        scores_true=[]
        prompt = input_prompt + ref_true
        input_ids = tokenizer(input_prompt, return_tensors="pt").input_ids.to(device)
        prompt_ids = tokenizer(prompt, return_tensors="pt").input_ids.to(device)
        outputs = self(prompt_ids)[0].squeeze(0)
        outputs = outputs.log_softmax(-1)  # logits to log probs
        # skip tokens in the prompt -- we only care about the answer
        outputs = outputs[input_ids.shape[-1] - 1: -1, :]
        prompt_ids = prompt_ids[0, input_ids.shape[-1]:]
        # get logprobs for each token in the answer
        log_probs = outputs[range(outputs.shape[0]), prompt_ids.squeeze(0)]
        scores_true.append(log_probs.sum().item())

        # false answer
        scores_false=[]
        for temp_ans in ref_false_list:
            # append the current answer choice to the prompt
            prompt = input_prompt + temp_ans

            input_ids = tokenizer(input_prompt, return_tensors="pt").input_ids.to(device)
            prompt_ids = tokenizer(prompt, return_tensors="pt").input_ids.to(device)
            outputs = self(prompt_ids)[0].squeeze(0)
            outputs = outputs.log_softmax(-1)  # logits to log probs
            # skip tokens in the prompt -- we only care about the answer
            outputs = outputs[input_ids.shape[-1] - 1: -1, :]
            prompt_ids = prompt_ids[0, input_ids.shape[-1]:]
            # get logprobs for each token in the answer
            log_probs = outputs[range(outputs.shape[0]), prompt_ids.squeeze(0)]
            scores_false.append(log_probs.sum().item())
        
        MC1, MC2 = self.MC_calcs(scores_true, scores_false)
        print("**********************************")
        print("**********************************")
        print(f"restore symptoms: {input['restore_evidences']}")
        print(f"diagosis: {input['correct_answer']}")
        print(f"single mc1: {MC1}")
        print(f"single mc2: {MC2}")
        print("**********************************")
        print("**********************************", flush=True)

        return MC1, MC2


    def intervention_infer_mc_score(
        self,
        input = None,
        tokenizer = None,
        device = None,
        auxiliary_model = None,
        auxiliary_tokenizer = None,
        dx_privacy = None,
        use_weight = True
    ):       
        #######################################
        #### calculate MC score
        #######################################

        # no privacy information leak, use base mc score calculation
        if len(input['restore_evidences']) == 0:
            return self.base_infer_mc_score(input['question_mask'], \
                                            input['correct_answer'], \
                                            [item for item in input['options'] if item != input['correct_answer']], \
                                             tokenizer, device)

        ref_true = format_ans(input['correct_answer'])
        ref_false_list = [format_ans(item) for item in input['options'] if item != input['correct_answer']]
        print(f"use weight: {use_weight}")
        if use_weight:
            re_evi_attention, multiplier = self.get_attention_diff(auxiliary_model, auxiliary_tokenizer, input['restore_evidences'], input['keep_evidences'], input['question_init'], device)
        input_prompt = format_questions_mc(input['question_mask'])
        layers_to_intervene = []
        for flattened_idx in self.model.intervention_heads:
            layer, _ = LlamaForCausalLM.flattened_idx_to_layer_head(flattened_idx, self.config.num_attention_heads)
            layers_to_intervene.append(f"model.layers.{layer}.self_attn.head_out")
        layers_to_intervene = list(set(layers_to_intervene))

        # true answer
        scores_true=[]
        prompt = input_prompt + ref_true
        input_ids = tokenizer(input_prompt, return_tensors="pt").input_ids.to(device)
        prompt_ids = tokenizer(prompt, return_tensors="pt").input_ids.to(device)

        # print(f"use weight: {use_weight}")
        if use_weight:
            intervene = partial(lt_modulated_vector_add_new_direction, llamaModel=self, restore_evidences=input['restore_evidences'], \
                                    re_evi_attention = re_evi_attention, multiplier = multiplier, dx_privacy=dx_privacy)
        else:
            intervene = partial(lt_modulated_vector_add_new_direction_noWeight, llamaModel=self, restore_evidences=input['restore_evidences'], \
                                    dx_privacy=dx_privacy)

        with TraceDict(self, layers_to_intervene, edit_output=intervene) as ret: 
            outputs = self(prompt_ids)[0].squeeze(0)
        
        outputs = outputs.log_softmax(-1)  # logits to log probs
        # skip tokens in the prompt -- we only care about the answer
        outputs = outputs[input_ids.shape[-1] - 1: -1, :]
        prompt_ids = prompt_ids[0, input_ids.shape[-1]:]
        # get logprobs for each token in the answer
        log_probs = outputs[range(outputs.shape[0]), prompt_ids.squeeze(0)]
        scores_true.append(log_probs.sum().item())


        # false answer
        scores_false=[]
        for temp_ans in ref_false_list:
            # append the current answer choice to the prompt
            prompt = input_prompt + temp_ans

            input_ids = tokenizer(input_prompt, return_tensors="pt").input_ids.to(device)
            prompt_ids = tokenizer(prompt, return_tensors="pt").input_ids.to(device)
            with TraceDict(self, layers_to_intervene, edit_output=intervene) as ret: 
                outputs = self(prompt_ids)[0].squeeze(0)
            outputs = outputs.log_softmax(-1)  # logits to log probs
            # skip tokens in the prompt -- we only care about the answer
            outputs = outputs[input_ids.shape[-1] - 1: -1, :]
            prompt_ids = prompt_ids[0, input_ids.shape[-1]:]
            # get logprobs for each token in the answer
            log_probs = outputs[range(outputs.shape[0]), prompt_ids.squeeze(0)]
            scores_false.append(log_probs.sum().item())

        MC1, MC2 = self.MC_calcs(scores_true, scores_false)
        print("**********************************")
        print("**********************************")
        print(f"restore symptoms: {input['restore_evidences']}")
        print(f"diagosis: {input['correct_answer']}")
        print(f"single mc1: {MC1}")
        print(f"single mc2: {MC2}")
        print("**********************************")
        print("**********************************", flush=True)

        return MC1, MC2



    #====================
    # 测试生成的质量
    # input: dict
    #   - restore_evidences
    #   - keep_evidences
    #   - question_init
    #   - question_mask
    #   - correct_answer
    #====================

    def cal_ppl(
        self,
        use_intervention = False,
        question = "",
        answer = "",
        intervene = None,
        layers_to_intervene = None,
        tokenizer = None,
        device = None
    ):
        # cal ppl
        question_ids = tokenizer(question, return_tensors="pt").input_ids.to(device)
        answer_ids = tokenizer(answer, return_tensors='pt').input_ids[:, 1:].to(device)
        question_answer_ids = torch.concat((question_ids, answer_ids), dim=1)
        
        if use_intervention:
            with TraceDict(self, layers_to_intervene, edit_output=intervene) as ret: 
                outputs = self(question_answer_ids)
        else:
            outputs = self(question_answer_ids)


        logits = outputs['logits'][0, question_ids.shape[-1]-1:-1, :]
        loss = CrossEntropyLoss(reduction='mean')(logits.view(-1, logits.shape[-1]), answer_ids.view(-1))

        return torch.exp(loss).detach().cpu().numpy()

    

    def base_generate(
        self,
        question = None,
        tokenizer = None,
        device = None,
        output_len = 256,
        top_k = 3
    ):
        input_prompt = format_questions_mc(question)
        input_ids = tokenizer(input_prompt, return_tensors="pt").input_ids.to(device)
        max_len = input_ids.shape[-1] + output_len

        model_gen_tokens = self.generate(input_ids, top_k=top_k, max_length=max_len, num_return_sequences=1,)[:, input_ids.shape[-1]:]

        model_gen_str = tokenizer.decode(model_gen_tokens[0], skip_special_tokens=True)
        model_gen_str = model_gen_str.strip()

        return model_gen_str

    def intervention_generate(
        self,
        input = None,
        tokenizer = None,
        device = None,
        auxiliary_model = None,
        auxiliary_tokenizer = None,
        output_len = 256,
        top_k = 3,
        dx_privacy = None,
        use_weight = True
    ):
        
        # no privacy information leak, use base generate
        if len(input['restore_evidences']) == 0:
            return self.base_generate(input['question_mask'])

        print(f"use weight: {use_weight}")
        if use_weight:
            re_evi_attention, multiplier = self.get_attention_diff(auxiliary_model, auxiliary_tokenizer, input['restore_evidences'], input['keep_evidences'], input['question_init'], device)

        input_prompt = format_questions_mc(input['question_mask'])
        input_ids = tokenizer(input_prompt, return_tensors="pt").input_ids.to(device)
        max_len = input_ids.shape[-1] + output_len

        layers_to_intervene = []
        for flattened_idx in self.model.intervention_heads:
            layer, head = LlamaForCausalLM.flattened_idx_to_layer_head(flattened_idx, self.config.num_attention_heads)
            layers_to_intervene.append(f"model.layers.{layer}.self_attn.head_out")
        layers_to_intervene = list(set(layers_to_intervene))
        
        if use_weight:
            intervene = partial(lt_modulated_vector_add_new_direction, llamaModel=self, restore_evidences=input['restore_evidences'], \
                                    re_evi_attention = re_evi_attention, multiplier = multiplier, dx_privacy=dx_privacy)
        else:
            intervene = partial(lt_modulated_vector_add_new_direction_noWeight, llamaModel=self, restore_evidences=input['restore_evidences'], \
                                    dx_privacy=dx_privacy)
        
        with TraceDict(self, layers_to_intervene, edit_output=intervene) as ret: 

            model_gen_tokens = self.generate(input_ids, top_k=top_k, max_length=max_len, num_return_sequences=1, \
                                            do_sample = False)[:, input_ids.shape[-1]:]

        model_gen_str = tokenizer.decode(model_gen_tokens[0], skip_special_tokens=True)
        model_gen_str = model_gen_str.strip()
        output_len = model_gen_tokens.shape[1]
        return model_gen_str
    

    def intervention_generate_speed(
        self,
        input = None,
        tokenizer = None,
        device = None,
        auxiliary_model = None,
        auxiliary_tokenizer = None,
        output_len = 256,
        top_k = 3,
        dx_privacy = None
    ):
        
        # no privacy information leak, use base generate
        if len(input['restore_evidences']) == 0:
            return self.base_generate(input['question_mask'])

        re_evi_attention, multiplier \
            = self.get_attention_diff(auxiliary_model, auxiliary_tokenizer, input['restore_evidences'], input['keep_evidences'], \
                                      input['question_init'], device)
        # print(multiplier)
        input_prompt = format_questions_mc(input['question_mask'])
        input_ids = tokenizer(input_prompt, return_tensors="pt").input_ids.to(device)
        max_len = input_ids.shape[-1] + output_len

        layers_to_intervene = []
        for flattened_idx in self.model.intervention_heads:
            layer, head = LlamaForCausalLM.flattened_idx_to_layer_head(flattened_idx, self.config.num_attention_heads)
            layers_to_intervene.append(f"model.layers.{layer}.self_attn.head_out")
        layers_to_intervene = list(set(layers_to_intervene))
        
        intervene = partial(lt_modulated_vector_add_new_direction_speed, llamaModel=self, restore_evidences=input['restore_evidences'], \
                                re_evi_attention= re_evi_attention, multiplier = multiplier, dx_privacy =dx_privacy)
        
        with TraceDict(self, layers_to_intervene, edit_output=intervene) as ret: 
            model_gen_tokens = self.generate(input_ids, top_k=top_k, max_length=max_len, num_return_sequences=1, \
                                            do_sample = False)[:, input_ids.shape[-1]:]

        model_gen_str = tokenizer.decode(model_gen_tokens[0], skip_special_tokens=True)
        model_gen_str = model_gen_str.strip()
        output_len = model_gen_tokens.shape[1]
        return model_gen_str, model_gen_tokens.shape[1]


    #=========================
    # helper
    #=========================
    def get_interventions(
        self,
        input = None,
        tokenizer = None,
        device = None,
        auxiliary_model = None,
        auxiliary_tokenizer = None,
        epsilon = 75
    ):
        print(f"Used epsilon: {epsilon}")
        if epsilon >=0:
            dx_privacy = DX_privacy(epsilon)
        re_evi_attention, multiplier \
            = self.get_attention_diff(auxiliary_model, auxiliary_tokenizer, input['restore_evidences'], input['keep_evidences'], \
                                      input['question_init'], device)
        input_prompt = format_questions_mc(input['question_mask'])
        input_ids = tokenizer(input_prompt, return_tensors="pt").input_ids.to(device)

        directions = []
        for hid, flattened_idx in enumerate(self.model.intervention_heads):
            layer, head = LlamaForCausalLM.flattened_idx_to_layer_head(flattened_idx, self.config.num_attention_heads)
            all_direction = 0
            for idx, evidence in enumerate(input['restore_evidences']):
                eid = self.model.evidences2id[evidence] 
                direction_pick = torch.zeros(self.model.evidences_len * self.model.interHead_len).to(device)
                direction_pick[eid * self.model.interHead_len + hid] = 1
                direction = self.model.head_out_intervention(direction_pick.unsqueeze(0)).squeeze(0)
                all_direction += direction * re_evi_attention[idx]
            directions.append(all_direction)
        directions = torch.concat(directions, dim=0)
        directions = directions / torch.norm(directions, p=2, dim=-1)
        if epsilon >=0:
            directions = dx_privacy.add_noise_to_embedding(directions)
        return directions

@add_start_docstrings(
    """
    The LLaMa Model transformer with a sequence classification head on top (linear layer).

    [`LlamaForSequenceClassification`] uses the last token in order to do the classification, as other causal models
    (e.g. GPT-2) do.

    Since it does classification on the last token, it requires to know the position of the last token. If a
    `pad_token_id` is defined in the configuration, it finds the last token that is not a padding token in each row. If
    no `pad_token_id` is defined, it simply takes the last value in each row of the batch. Since it cannot guess the
    padding tokens when `inputs_embeds` are passed instead of `input_ids`, it does the same (take the last value in
    each row of the batch).
    """,
    LLAMA_START_DOCSTRING,
)
class LlamaForSequenceClassification(LlamaPreTrainedModel):
    def __init__(self, config):
        super().__init__(config)
        self.num_labels = config.num_labels
        self.model = LlamaModel(config)
        self.score = nn.Linear(config.hidden_size, self.num_labels, bias=False)

        # Initialize weights and apply final processing
        self.post_init()

    def get_input_embeddings(self):
        return self.model.embed_tokens

    def set_input_embeddings(self, value):
        self.model.embed_tokens = value

    @add_start_docstrings_to_model_forward(LLAMA_INPUTS_DOCSTRING)
    def forward(
        self,
        input_ids: torch.LongTensor = None,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_values: Optional[List[torch.FloatTensor]] = None,
        inputs_embeds: Optional[torch.FloatTensor] = None,
        labels: Optional[torch.LongTensor] = None,
        use_cache: Optional[bool] = None,
        output_attentions: Optional[bool] = None,
        output_hidden_states: Optional[bool] = None,
        return_dict: Optional[bool] = None,
    ) -> Union[Tuple, SequenceClassifierOutputWithPast]:
        r"""
        labels (`torch.LongTensor` of shape `(batch_size,)`, *optional*):
            Labels for computing the sequence classification/regression loss. Indices should be in `[0, ...,
            config.num_labels - 1]`. If `config.num_labels == 1` a regression loss is computed (Mean-Square loss), If
            `config.num_labels > 1` a classification loss is computed (Cross-Entropy).
        """
        return_dict = return_dict if return_dict is not None else self.config.use_return_dict

        transformer_outputs = self.model(
            input_ids,
            attention_mask=attention_mask,
            position_ids=position_ids,
            past_key_values=past_key_values,
            inputs_embeds=inputs_embeds,
            use_cache=use_cache,
            output_attentions=output_attentions,
            output_hidden_states=output_hidden_states,
            return_dict=return_dict,
        )
        hidden_states = transformer_outputs[0]
        logits = self.score(hidden_states)

        if input_ids is not None:
            batch_size = input_ids.shape[0]
        else:
            batch_size = inputs_embeds.shape[0]

        if self.config.pad_token_id is None and batch_size != 1:
            raise ValueError("Cannot handle batch sizes > 1 if no padding token is defined.")
        if self.config.pad_token_id is None:
            sequence_lengths = -1
        else:
            if input_ids is not None:
                sequence_lengths = (torch.eq(input_ids, self.config.pad_token_id).long().argmax(-1) - 1).to(
                    logits.device
                )
            else:
                sequence_lengths = -1

        pooled_logits = logits[torch.arange(batch_size, device=logits.device), sequence_lengths]

        loss = None
        if labels is not None:
            labels = labels.to(logits.device)
            if self.config.problem_type is None:
                if self.num_labels == 1:
                    self.config.problem_type = "regression"
                elif self.num_labels > 1 and (labels.dtype == torch.long or labels.dtype == torch.int):
                    self.config.problem_type = "single_label_classification"
                else:
                    self.config.problem_type = "multi_label_classification"

            if self.config.problem_type == "regression":
                loss_fct = MSELoss()
                if self.num_labels == 1:
                    loss = loss_fct(pooled_logits.squeeze(), labels.squeeze())
                else:
                    loss = loss_fct(pooled_logits, labels)
            elif self.config.problem_type == "single_label_classification":
                loss_fct = CrossEntropyLoss()
                loss = loss_fct(pooled_logits.view(-1, self.num_labels), labels.view(-1))
            elif self.config.problem_type == "multi_label_classification":
                loss_fct = BCEWithLogitsLoss()
                loss = loss_fct(pooled_logits, labels)
        if not return_dict:
            output = (pooled_logits,) + transformer_outputs[1:]
            return ((loss,) + output) if loss is not None else output

        return SequenceClassifierOutputWithPast(
            loss=loss,
            logits=pooled_logits,
            past_key_values=transformer_outputs.past_key_values,
            hidden_states=transformer_outputs.hidden_states,
            attentions=transformer_outputs.attentions,
        )
