from abc import ABC, abstractmethod

import numpy as np
from sklearn.preprocessing import MinMaxScaler
from pandarallel import pandarallel
import pandas as pd
from typing import Tuple, List
from numpy.typing import NDArray

class BaseEditor(ABC):
    """A set of prior knowledge for each samples per label.

    Calculating and caching prior knowledge, density or likelihood, of samples for each class.
    
    Attributes:
        X: A dataframe of entire dataset.
        X_train: A dataframe of training dataset.
        y: A series of label.
        pl_priors: prior knowledge for each samlpes per label.
        continuous_cols: A list of continuous features to digitize.
        scaler: A minmax scaler for likelihoods.
    """
    def __init__(self,
                 X: pd.DataFrame,
                 X_train: pd.DataFrame,
                 y: pd.Series,
                 continuous_cols: List[str],
                 n_jobs: int
    ) -> None:
        self.X = X
        self.X_train = X_train.copy()
        self.y = y.copy()
        self.pl_criteria = None
        self.continuous_cols = continuous_cols
        self.scaler = MinMaxScaler()
        
        pandarallel.initialize(progress_bar=False, nb_workers = n_jobs, use_memory_fs=False)
        
        self.init_criteria()
        self.init_scaler()
        self.caching_criteria()
    
    @abstractmethod
    def init_criteria(self) -> None:
        pass
    
    @abstractmethod
    def init_scaler(self) -> None:
        pass
    
    @abstractmethod
    def caching_criteria(self) -> None:
        pass
    
    @abstractmethod
    def scale_criteria(self, criteria: NDArray[np.float32]) -> NDArray[np.float32]:
        pass
    
    def get_criteria(self, idx: NDArray[np.float32]) -> NDArray[np.float32]:
        return self.scale_criteria(self.pl_criteria.loc[idx, :])