from abc import ABC, abstractmethod

from sklearn.metrics import f1_score, roc_auc_score

import numpy as np

class Scorer(ABC, object):

    def __init__(self, metric_params):
        self.metric_params = metric_params

    @abstractmethod
    def __call__(self, y, y_hat):
        pass

class F1_Score(Scorer):

    def __init__(self, metric_params):
        super().__init__(metric_params)

    def __call__(self, 
                 y: np.array, 
                 y_probs: np.array
        ) -> float:
        
        y_hat = y_probs.argmax(1)
        return f1_score(y, y_hat, **self.metric_params)

class ROC_AUC_Score(Scorer):
    def __init__(self, metric_params):
        super().__init__(metric_params)
    
    def __call__(self, 
                 y: np.array, 
                 y_probs: np.array
        ) -> float:
        return roc_auc_score(y, y_probs[:, 1], **self.metric_params)

class Multi_ROC_AUC_Score(Scorer):
    def __init__(self, metric_params):
        super().__init__(metric_params)
    
    def __call__(self, 
                 y: np.array, 
                 y_probs: np.array
        ) -> float:
        return roc_auc_score(y, y_probs, **self.metric_params)