import torch
import numpy as np
from tqdm import tqdm
from functools import partial
import torchvision.transforms as T
from ...modules.diffusionmodules.util import make_ddim_sampling_parameters, make_ddim_timesteps, noise_like, extract_into_tensor


class DDIMSampler(object):
    def __init__(self, model, schedule="linear", **kwargs):
        super().__init__()
        self.model = model
        self.ddpm_num_timesteps = model.num_timesteps
        self.schedule = schedule

    def register_buffer(self, name, attr):
        if type(attr) == torch.Tensor:
            if attr.device != torch.device("cuda"):
                attr = attr.to(torch.device("cuda"))
        setattr(self, name, attr)

    def make_schedule(self, ddim_num_steps, ddim_discretize="uniform", ddim_eta=0., verbose=True):
        self.ddim_timesteps = make_ddim_timesteps(ddim_discr_method=ddim_discretize, num_ddim_timesteps=ddim_num_steps,
                                                  num_ddpm_timesteps=self.ddpm_num_timesteps,verbose=verbose)
        alphas_cumprod = self.model.alphas_cumprod
        assert alphas_cumprod.shape[0] == self.ddpm_num_timesteps, 'alphas have to be defined for each timestep'
        # print(self.model.__class__.__name__, 'using DDIM Sampler with', self.schedule, 'schedule')
        to_torch = lambda x: x.clone().detach().to(torch.float32).to(self.model.device)

        self.register_buffer('betas', to_torch(self.model.betas))
        self.register_buffer('alphas_cumprod', to_torch(alphas_cumprod))
        self.register_buffer('alphas_cumprod_prev', to_torch(self.model.alphas_cumprod_prev))

        # calculations for diffusion q(x_t | x_{t-1}) and others
        self.register_buffer('sqrt_alphas_cumprod', to_torch(np.sqrt(alphas_cumprod.cpu())))
        self.register_buffer('sqrt_one_minus_alphas_cumprod', to_torch(np.sqrt(1. - alphas_cumprod.cpu())))
        self.register_buffer('log_one_minus_alphas_cumprod', to_torch(np.log(1. - alphas_cumprod.cpu())))
        self.register_buffer('sqrt_recip_alphas_cumprod', to_torch(np.sqrt(1. / alphas_cumprod.cpu())))
        self.register_buffer('sqrt_recipm1_alphas_cumprod', to_torch(np.sqrt(1. / alphas_cumprod.cpu() - 1)))

        # ddim sampling parameters
        ddim_sigmas, ddim_alphas, ddim_alphas_prev = make_ddim_sampling_parameters(alphacums=alphas_cumprod.cpu(),
                                                                                   ddim_timesteps=self.ddim_timesteps,
                                                                                   eta=ddim_eta,verbose=verbose)
        self.register_buffer('ddim_sigmas', ddim_sigmas)
        self.register_buffer('ddim_alphas', ddim_alphas)
        self.register_buffer('ddim_alphas_prev', ddim_alphas_prev)
        self.register_buffer('ddim_sqrt_one_minus_alphas', np.sqrt(1. - ddim_alphas))
        sigmas_for_original_sampling_steps = ddim_eta * torch.sqrt(
            (1 - self.alphas_cumprod_prev) / (1 - self.alphas_cumprod) * (
                        1 - self.alphas_cumprod / self.alphas_cumprod_prev))
        self.register_buffer('ddim_sigmas_for_original_num_steps', sigmas_for_original_sampling_steps)

    @torch.no_grad()
    def sample(self,
               S,
               batch_size,
               shape,
               conditioning=None,
               callback=None,
               normals_sequence=None,
               img_callback=None,
               quantize_x0=False,
               eta=0.,
               mask=None,
               x0=None,
               temperature=1.,
               noise_dropout=0.,
               score_corrector=None,
               corrector_kwargs=None,
               verbose=True,
               x_T=None,
               log_every_t=1,
               unconditional_guidance_scale=1.,
               unconditional_conditioning=None,
               controller=None,
               **kwargs
               ):
        if conditioning is not None:
            if isinstance(conditioning, dict):
                cbs = conditioning[list(conditioning.keys())[0]].shape[0]
                if cbs != batch_size:
                    print(f"Warning: Got {cbs} conditionings but batch-size is {batch_size}")
            else:
                if conditioning.shape[0] != batch_size:
                    print(f"Warning: Got {conditioning.shape[0]} conditionings but batch-size is {batch_size}")

        self.make_schedule(ddim_num_steps=S, ddim_eta=eta, verbose=verbose)
        # sampling
        C, H, W = shape[-3:]
        size = (batch_size, C, H, W)

        # print(controller)
        samples, intermediates = self.ddim_sampling(conditioning, size,
                                                    callback=callback,
                                                    img_callback=img_callback,
                                                    quantize_denoised=quantize_x0,
                                                    mask=mask, x0=x0,
                                                    ddim_use_original_steps=False,
                                                    noise_dropout=noise_dropout,
                                                    temperature=temperature,
                                                    score_corrector=score_corrector,
                                                    corrector_kwargs=corrector_kwargs,
                                                    x_T=x_T,
                                                    log_every_t=log_every_t,
                                                    unconditional_guidance_scale=unconditional_guidance_scale,
                                                    unconditional_conditioning=unconditional_conditioning,
                                                    controller=controller,
                                                    **kwargs
                                                    )
        return samples, intermediates

    # FIXME
    @torch.no_grad()
    def sample_amg(self, S, batch_size, shape, conditioning=None, callback=None, normals_sequence=None,
                   img_callback=None, quantize_x0=False, eta=0., mask=None, x0=None, temperature=1.,
                   noise_dropout=0., score_corrector=None, corrector_kwargs=None, verbose=True,
                   x_T=None, log_every_t=1, unconditional_guidance_scale=1., unconditional_conditioning=None,
                   controller=None, nn_search=None, c_sim=1.0, c_spe=1.0, c_dup=1.0, **kwargs):
        """
        AMG sampling method - NO GRADIENTS REQUIRED
        Uses the original AMG approach: modify epsilon predictions directly based on similarity scores
        """
        
        if nn_search is None:
            print("Warning: nn_search not provided, falling back to standard sampling")
            return self.sample(S, batch_size, shape, conditioning, callback, normals_sequence,
                             img_callback, quantize_x0, eta, mask, x0, temperature,
                             noise_dropout, score_corrector, corrector_kwargs, verbose,
                             x_T, log_every_t, unconditional_guidance_scale, unconditional_conditioning,
                             controller, **kwargs)
            
        if conditioning is not None:
            if isinstance(conditioning, dict):
                cbs = conditioning[list(conditioning.keys())[0]].shape[0]
                if cbs != batch_size:
                    print(f"Warning: Got {cbs} conditionings but batch-size is {batch_size}")
            else:
                if conditioning.shape[0] != batch_size:
                    print(f"Warning: Got {conditioning.shape[0]} conditionings but batch-size is {batch_size}")

        self.make_schedule(ddim_num_steps=S, ddim_eta=eta, verbose=verbose)
        C, H, W = shape[-3:]
        size = (batch_size, C, H, W)

        samples, intermediates = self.ddim_sampling_amg(
            conditioning, size,
            callback=callback,
            img_callback=img_callback,
            quantize_denoised=quantize_x0,
            mask=mask, x0=x0,
            ddim_use_original_steps=False,
            noise_dropout=noise_dropout,
            temperature=temperature,
            score_corrector=score_corrector,
            corrector_kwargs=corrector_kwargs,
            x_T=x_T,
            log_every_t=log_every_t,
            unconditional_guidance_scale=unconditional_guidance_scale,
            unconditional_conditioning=unconditional_conditioning,
            controller=controller,
            nn_search=nn_search,
            c_sim=c_sim,
            c_spe=c_spe,
            c_dup=c_dup,
            **kwargs
        )
        return samples, intermediates

    @torch.no_grad()
    def ddim_sampling_amg(self, cond, shape, x_T=None, ddim_use_original_steps=False,
                          callback=None, timesteps=None, quantize_denoised=False,
                          mask=None, x0=None, img_callback=None, log_every_t=1,
                          temperature=1., noise_dropout=0., score_corrector=None, corrector_kwargs=None,
                          unconditional_guidance_scale=1., unconditional_conditioning=None,
                          controller=None, nn_search=None, c_sim=1.0, c_spe=1.0, c_dup=1.0, **kwargs):
        """
        AMG DDIM sampling - completely gradient-free, following original paper implementation
        """
        device = self.model.betas.device
        b = shape[0]
        
        if x_T is None:
            img = torch.randn(shape, device=device)
        else:
            img = x_T

        if timesteps is None:
            timesteps = self.ddpm_num_timesteps if ddim_use_original_steps else self.ddim_timesteps
        elif timesteps is not None and not ddim_use_original_steps:
            subset_end = int(min(timesteps / self.ddim_timesteps.shape[0], 1) * self.ddim_timesteps.shape[0]) - 1
            timesteps = self.ddim_timesteps[:subset_end]

        intermediates = {'x_inter': [img], 'pred_x0': [img], 'uncond_noise': [], 'text_noise': [], 'timesteps': timesteps}
        time_range = reversed(range(0,timesteps)) if ddim_use_original_steps else np.flip(timesteps)
        total_steps = timesteps if ddim_use_original_steps else timesteps.shape[0]

        iterator = tqdm(time_range, desc='AMG DDIM Sampler', total=total_steps)

        # Cache for neighbor conditioning to avoid repeated computation
        neighbor_cache = {}

        for i, step in enumerate(iterator):
            index = total_steps - i - 1
            ts = torch.full((b,), step, device=device, dtype=torch.long)

            if controller is not None:
                controller.set_current_step(i)

            if mask is not None:
                assert x0 is not None
                img_orig = self.model.q_sample(x0, ts)
                img = img_orig * mask + (1. - mask) * img

            # AMG-enhanced sampling step
            outs = self.p_sample_ddim_amg(
                img, cond, ts, index=index, use_original_steps=ddim_use_original_steps,
                quantize_denoised=quantize_denoised, temperature=temperature,
                noise_dropout=noise_dropout, score_corrector=score_corrector,
                corrector_kwargs=corrector_kwargs,
                unconditional_guidance_scale=unconditional_guidance_scale,
                unconditional_conditioning=unconditional_conditioning,
                nn_search=nn_search, c_sim=c_sim, c_spe=c_spe, c_dup=c_dup,
                step_index=i, total_steps=total_steps, neighbor_cache=neighbor_cache, **kwargs
            )
            img, pred_x0, model_uncond, model_t = outs
            
            if callback: callback(i)
            if img_callback: img_callback(pred_x0, i)

            if index % log_every_t == 0 or index == total_steps - 1:
                intermediates['x_inter'].append(img)
                intermediates['pred_x0'].append(pred_x0)
                intermediates['uncond_noise'].append(model_uncond)
                intermediates['text_noise'].append(model_t)
                
        return img, intermediates

    @torch.no_grad()
    def p_sample_ddim_amg(self, x, c, t, index, repeat_noise=False, use_original_steps=False, quantize_denoised=False,
                          temperature=1., noise_dropout=0., score_corrector=None, corrector_kwargs=None,
                          unconditional_guidance_scale=1., unconditional_conditioning=None,
                          dynamic_threshold=None, nn_search=None, c_sim=1.0, c_spe=1.0, c_dup=1.0,
                          step_index=0, total_steps=50, neighbor_cache=None, **kwargs):
        """
        AMG-enhanced DDIM step following the original paper's approach:
        1. Compute standard CFG prediction
        2. Check similarity to training data
        3. If high similarity, modify epsilon prediction directly (no gradients!)
        """
        b, *_, device = *x.shape, x.device

        # Standard classifier-free guidance prediction  
        if unconditional_conditioning is None or unconditional_guidance_scale == 1.:
            model_output = self.model.apply_model(x, t, c)
            model_uncond = model_t = model_output
        else:
            # Handle dictionary-style conditioning efficiently
            if isinstance(c, dict):
                # Batch both conditions in one forward pass
                x_in = torch.cat([x, x])
                t_in = torch.cat([t, t])
                
                c_in = {}
                for k in c:
                    if isinstance(c[k], torch.Tensor):
                        c_in[k] = torch.cat([unconditional_conditioning[k], c[k]])
                    else:
                        c_in[k] = c[k]  # Non-tensor values stay the same
                        
                combined_output = self.model.apply_model(x_in, t_in, c_in)
                model_uncond, model_t = combined_output.chunk(2)
            else:
                # Fallback for non-dict conditioning
                x_in = torch.cat([x] * 2)
                t_in = torch.cat([t] * 2)
                if isinstance(c, list):
                    c_in = [torch.cat([unconditional_conditioning[i], c[i]]) for i in range(len(c))]
                else:
                    c_in = torch.cat([unconditional_conditioning, c])
                model_uncond, model_t = self.model.apply_model(x_in, t_in, c_in).chunk(2)
                
            model_output = model_uncond + unconditional_guidance_scale * (model_t - model_uncond)

        # Convert to epsilon if needed
        if self.model.parameterization == "v":
            e_t = self.model.predict_eps_from_z_and_v(x, t, model_output)
        else:
            e_t = model_output

        if score_corrector is not None:
            assert self.model.parameterization == "eps", 'not implemented'
            e_t = score_corrector.modify_score(self.model, e_t, x, t, c, **corrector_kwargs)

        # Get DDIM parameters
        alphas = self.model.alphas_cumprod if use_original_steps else self.ddim_alphas
        alphas_prev = self.model.alphas_cumprod_prev if use_original_steps else self.ddim_alphas_prev
        sqrt_one_minus_alphas = self.model.sqrt_one_minus_alphas_cumprod if use_original_steps else self.ddim_sqrt_one_minus_alphas
        sigmas = self.model.ddim_sigmas_for_original_num_steps if use_original_steps else self.ddim_sigmas
        
        a_t = torch.full((b, 1, 1, 1), alphas[index], device=device)
        a_prev = torch.full((b, 1, 1, 1), alphas_prev[index], device=device)
        sigma_t = torch.full((b, 1, 1, 1), sigmas[index], device=device)
        sqrt_one_minus_at = torch.full((b, 1, 1, 1), sqrt_one_minus_alphas[index], device=device)

        # Predict x0 for similarity check
        if self.model.parameterization != "v":
            pred_x0 = (x - sqrt_one_minus_at * e_t) / a_t.sqrt()
        else:
            pred_x0 = self.model.predict_start_from_z_and_v(x, t, model_output)

        if quantize_denoised:
            pred_x0, _, *_ = self.model.first_stage_model.quantize(pred_x0)

        # ========== AMG GUIDANCE (NO GRADIENTS) ==========
        if nn_search is not None: # and step_index % 3 == 0:
            try:
                # Get similarity score by checking current prediction against training data
                similarity_score, neighbor_caption = self._compute_similarity_lightweight(
                    pred_x0, nn_search, step_index
                )

                # Dynamic threshold from paper (Equation 10)
                lambda_t = self._compute_parabolic_threshold(step_index, total_steps)
                
                # Apply AMG guidance if similarity exceeds threshold
                if similarity_score > lambda_t:
                    
                    # Guidance Strategy 1: Despecification Guidance (G_spe)
                    # Equation 14: G_spe = -s1 * (ε_θ(x_t, y) - ε_θ(x_t))
                    s1 = max(min(c_spe * similarity_score, unconditional_guidance_scale - 1), 0)
                    g_spe = -s1 * (model_t - model_uncond)

                    # Guidance Strategy 2: Caption Deduplication Guidance (G_dup)
                    # Equation 16: G_dup = -s2 * (ε_θ(x_t, y_N) - ε_θ(x_t))
                    g_dup = torch.zeros_like(e_t)
                    if neighbor_caption:
                        # Use cache to avoid repeated encoding
                        cache_key = f"{neighbor_caption}_{b}"
                        if cache_key not in neighbor_cache:
                            neighbor_text_cond = self.model.get_learned_conditioning([neighbor_caption])
                            if isinstance(c, dict):
                                neighbor_cond = {"context": neighbor_text_cond.repeat(b, 1, 1)}
                                # Copy essential keys only
                                for k in ["num_frames"]:
                                    if k in c:
                                        neighbor_cond[k] = c[k]
                            else:
                                neighbor_cond = neighbor_text_cond.repeat(b, 1, 1)
                            neighbor_cache[cache_key] = neighbor_cond
                        else:
                            neighbor_cond = neighbor_cache[cache_key]
                            
                        try:
                            e_t_neighbor = self.model.apply_model(x, t, neighbor_cond)
                            s2 = max(min(c_dup * similarity_score, unconditional_guidance_scale - s1 - 1), 0)
                            g_dup = -s2 * (e_t_neighbor - model_uncond)
                        except:
                            g_dup = torch.zeros_like(e_t)

                    # Guidance Strategy 3: Dissimilarity Guidance (G_sim) - SIMPLIFIED
                    # Instead of computing gradients, we add noise in direction away from memorization
                    g_sim = torch.zeros_like(e_t)
                    if similarity_score > lambda_t * 1.2:  # Only for very high similarity
                        # Add small amount of noise to break similarity pattern
                        noise_scale = c_sim * (similarity_score - lambda_t) * 0.1
                        g_sim = noise_scale * torch.randn_like(e_t)

                    # Combine guidance terms (Equation 10)
                    g_amg = g_spe + g_dup + g_sim
                    e_t = e_t + g_amg
                    
            except Exception as e:
                print(f"Warning: AMG guidance failed at step {step_index}: {e}")

        if dynamic_threshold is not None:
            raise NotImplementedError()

        # Standard DDIM update
        dir_xt = (1. - a_prev - sigma_t**2).sqrt() * e_t
        noise = sigma_t * noise_like(x.shape, device, repeat_noise) * temperature
        if noise_dropout > 0.:
            noise = torch.nn.functional.dropout(noise, p=noise_dropout)
        x_prev = a_prev.sqrt() * pred_x0 + dir_xt + noise

        if unconditional_guidance_scale > 0:
            effective_model_t = model_t + (g_amg / unconditional_guidance_scale)
        else:
            effective_model_t = model_t

        return x_prev, pred_x0, model_uncond, effective_model_t

    @torch.no_grad()
    def _compute_similarity_lightweight(self, pred_x0, nn_search, step_index):
        """
        Lightweight similarity computation without gradients
        """
        try:
            # Decode at lower resolution for speed
            pred_x0_small = torch.nn.functional.interpolate(
                pred_x0, size=(64, 64), mode='bilinear', align_corners=False
            )
            pred_x0_pixel = self.model.decode_first_stage(pred_x0_small)
            
            # Convert to PIL (take first image in batch)
            pred_img = pred_x0_pixel[0].clamp(-1, 1)
            pred_img = (pred_img + 1.0) / 2.0
            pred_img_pil = T.ToPILImage()(pred_img.cpu())
            
            # Get similarity from NN search
            similarity_score, _, neighbor_caption = nn_search.find_nearest_neighbor(pred_img_pil)
            
            return float(similarity_score), neighbor_caption
            
        except Exception as e:
            raise e

    def _compute_parabolic_threshold(self, step, total_steps):
        """
        Compute parabolic threshold as described in the paper
        λ_t = a + (b-a)e^(-ct)
        """
        t_normalized = step / total_steps
        a = -1.95  # asymptotic value
        b = -1.5   # value at t=0  
        c = -0.025 # shape parameter
        
        lambda_t = a + (b - a) * np.exp(c * t_normalized)
        return lambda_t

    @torch.no_grad()
    def ddim_sampling(self, cond, shape,
                      x_T=None, ddim_use_original_steps=False,
                      callback=None, timesteps=None, quantize_denoised=False,
                      mask=None, x0=None, img_callback=None, log_every_t=1,
                      temperature=1., noise_dropout=0., score_corrector=None, corrector_kwargs=None,
                      unconditional_guidance_scale=1., unconditional_conditioning=None,
                      controller=None, **kwargs):
        device = self.model.betas.device
        b = shape[0]
        if x_T is None:
            img = torch.randn(shape, device=device)
        else:
            img = x_T

        if timesteps is None:
            timesteps = self.ddpm_num_timesteps if ddim_use_original_steps else self.ddim_timesteps
        elif timesteps is not None and not ddim_use_original_steps:
            subset_end = int(min(timesteps / self.ddim_timesteps.shape[0], 1) * self.ddim_timesteps.shape[0]) - 1
            timesteps = self.ddim_timesteps[:subset_end]

        intermediates = {'x_inter': [img], 'pred_x0': [img], 'uncond_noise': [], 'text_noise': [], 'timesteps': timesteps,}
        time_range = reversed(range(0,timesteps)) if ddim_use_original_steps else np.flip(timesteps)
        total_steps = timesteps if ddim_use_original_steps else timesteps.shape[0]

        iterator = tqdm(time_range, desc='DDIM Sampler', total=total_steps)

        for i, step in enumerate(iterator):
            index = total_steps - i - 1
            ts = torch.full((b,), step, device=device, dtype=torch.long)

            # If amg_params are present, add step-specific info to them
            if 'amg_params' in kwargs and kwargs['amg_params'] is not None:
                kwargs['amg_params']['step_index'] = i
                kwargs['amg_params']['total_steps'] = total_steps
            # Notify the controller that a new step is beginning
            if controller is not None:
                controller.set_current_step(i) # TODO track DDIM or DDPM steps?
                # print(controller.current_step)

            if mask is not None:
                assert x0 is not None
                img_orig = self.model.q_sample(x0, ts)  # TODO: deterministic forward pass?
                img = img_orig * mask + (1. - mask) * img

            outs = self.p_sample_ddim(img, cond, ts, index=index, use_original_steps=ddim_use_original_steps,
                                      quantize_denoised=quantize_denoised, temperature=temperature,
                                      noise_dropout=noise_dropout, score_corrector=score_corrector,
                                      corrector_kwargs=corrector_kwargs,
                                      unconditional_guidance_scale=unconditional_guidance_scale,
                                      unconditional_conditioning=unconditional_conditioning, **kwargs)
            img, pred_x0, model_uncond, model_t = outs
            if callback: callback(i)
            if img_callback: img_callback(pred_x0, i)

            if index % log_every_t == 0 or index == total_steps - 1:
                intermediates['x_inter'].append(img)
                intermediates['pred_x0'].append(pred_x0)
                intermediates['uncond_noise'].append(model_uncond)
                intermediates['text_noise'].append(model_t)
        return img, intermediates

    @torch.no_grad()
    def p_sample_ddim(self, x, c, t, index, repeat_noise=False, use_original_steps=False, quantize_denoised=False,
                      temperature=1., noise_dropout=0., score_corrector=None, corrector_kwargs=None,
                      unconditional_guidance_scale=1., unconditional_conditioning=None,
                      dynamic_threshold=None, amg_params=None, **kwargs):
        b, *_, device = *x.shape, x.device

        if unconditional_conditioning is None or unconditional_guidance_scale == 1.:
            model_output = self.model.apply_model(x, t, c)
        else:
            x_in = torch.cat([x] * 2)
            t_in = torch.cat([t] * 2)
            if isinstance(c, dict):
                assert isinstance(unconditional_conditioning, dict)
                c_in = dict()
                for k in c:
                    if isinstance(c[k], list):
                        c_in[k] = [torch.cat([
                            unconditional_conditioning[k][i],
                            c[k][i]]) for i in range(len(c[k]))]
                    elif isinstance(c[k], torch.Tensor):
                        c_in[k] = torch.cat([
                                unconditional_conditioning[k],
                                c[k]])
                    else:
                        assert c[k] == unconditional_conditioning[k]
                        c_in[k] = c[k]
            elif isinstance(c, list):
                c_in = list()
                assert isinstance(unconditional_conditioning, list)
                for i in range(len(c)):
                    c_in.append(torch.cat([unconditional_conditioning[i], c[i]]))
            else:
                c_in = torch.cat([unconditional_conditioning, c])
            model_uncond, model_t = self.model.apply_model(x_in, t_in, c_in).chunk(2)
            # model_t = self.model.apply_model(x, t, c, **kwargs)
            # model_uncond = self.model.apply_model(x, t, unconditional_conditioning, **kwargs)
            model_output = model_uncond + unconditional_guidance_scale * (model_t - model_uncond)

        if self.model.parameterization == "v":
            print("using v!")
            e_t = self.model.predict_eps_from_z_and_v(x, t, model_output)
        else:
            e_t = model_output
        # =================== NEW AMG INJECTION BLOCK ===================
        if amg_params is not None:
            g_spe = g_sim = 0.0
            # Unpack the parameters we need for AMG
            nn_search = amg_params.get("nn_search")
            step_index = amg_params.get("step_index")
            total_steps = amg_params.get("total_steps")
            spe_coeff = amg_params.get("spe_coeff" , 0.7)
            dup_coeff  = amg_params.get("dup_coeff" , 0.7)
            sim_coeff = amg_params.get("sim_coeff" , 0.7)
            
            # Only apply guidance in the second half of sampling for stability
            if nn_search is not None: #and step_index > (total_steps // 2):
                # This is your corrected, safer AMG logic from the previous step.
                # It now safely modifies e_t inside the original function.
                try:
                    # We need pred_x0 for the similarity check, so calculate it early
                    alphas = self.model.alphas_cumprod if use_original_steps else self.ddim_alphas
                    a_t_early = torch.full((b, 1, 1, 1), alphas[index], device=device)
                    sqrt_one_minus_at_early = torch.full((b, 1, 1, 1), (self.model.sqrt_one_minus_alphas_cumprod if use_original_steps else self.ddim_sqrt_one_minus_alphas)[index], device=device)
                    pred_x0_for_sim = (x - sqrt_one_minus_at_early * e_t) / a_t_early.sqrt()

                    similarity_score, neighbor_caption = self._compute_similarity_lightweight(
                        pred_x0_for_sim, nn_search, step_index
                    )
                    lambda_t = self._compute_parabolic_threshold(step_index, total_steps)

                    if similarity_score > lambda_t:
                        # Use the safer, corrected guidance logic
                        g_spe = -spe_coeff * (model_t - model_uncond)
                        g_sim = sim_coeff * (model_uncond - model_t)
                        # (Add g_dup logic here if needed)
                        e_t = e_t + g_spe + g_sim # Apply the safe nudge
                except Exception as e:
                    print(f"Warning: AMG guidance failed at step {step_index}: {e}")
            # ===============================================================

        if score_corrector is not None:
            assert self.model.parameterization == "eps", 'not implemented'
            e_t = score_corrector.modify_score(self.model, e_t, x, t, c, **corrector_kwargs)

        alphas = self.model.alphas_cumprod if use_original_steps else self.ddim_alphas
        alphas_prev = self.model.alphas_cumprod_prev if use_original_steps else self.ddim_alphas_prev
        sqrt_one_minus_alphas = self.model.sqrt_one_minus_alphas_cumprod if use_original_steps else self.ddim_sqrt_one_minus_alphas
        sigmas = self.model.ddim_sigmas_for_original_num_steps if use_original_steps else self.ddim_sigmas
        # select parameters corresponding to the currently considered timestep
        a_t = torch.full((b, 1, 1, 1), alphas[index], device=device)
        a_prev = torch.full((b, 1, 1, 1), alphas_prev[index], device=device)
        sigma_t = torch.full((b, 1, 1, 1), sigmas[index], device=device)
        sqrt_one_minus_at = torch.full((b, 1, 1, 1), sqrt_one_minus_alphas[index],device=device)

        # current prediction for x_0
        if self.model.parameterization != "v":
            pred_x0 = (x - sqrt_one_minus_at * e_t) / a_t.sqrt()
        else:
            pred_x0 = self.model.predict_start_from_z_and_v(x, t, model_output)

        if quantize_denoised:
            pred_x0, _, *_ = self.model.first_stage_model.quantize(pred_x0)

        if dynamic_threshold is not None:
            raise NotImplementedError()

        # direction pointing to x_t
        dir_xt = (1. - a_prev - sigma_t**2).sqrt() * e_t
        noise = sigma_t * noise_like(x.shape, device, repeat_noise) * temperature
        if noise_dropout > 0.:
            noise = torch.nn.functional.dropout(noise, p=noise_dropout)
        x_prev = a_prev.sqrt() * pred_x0 + dir_xt + noise
        if amg_params is not None and nn_search is not None: # and step_index > (total_steps // 2):
            # print("AMG specific output", model_t + g_spe + g_sim)
            return x_prev, pred_x0, model_uncond, model_t + g_spe + g_sim
        return x_prev, pred_x0, model_uncond, model_t

    @torch.no_grad()
    def stochastic_encode(self, x0, t, use_original_steps=False, noise=None):
        # fast, but does not allow for exact reconstruction
        # t serves as an index to gather the correct alphas
        if use_original_steps:
            sqrt_alphas_cumprod = self.sqrt_alphas_cumprod
            sqrt_one_minus_alphas_cumprod = self.sqrt_one_minus_alphas_cumprod
        else:
            sqrt_alphas_cumprod = torch.sqrt(self.ddim_alphas)
            sqrt_one_minus_alphas_cumprod = self.ddim_sqrt_one_minus_alphas

        if noise is None:
            noise = torch.randn_like(x0)
        return (extract_into_tensor(sqrt_alphas_cumprod, t, x0.shape) * x0 +
                extract_into_tensor(sqrt_one_minus_alphas_cumprod, t, x0.shape) * noise)

    @torch.no_grad()
    def decode(self, x_latent, cond, t_start, unconditional_guidance_scale=1.0, unconditional_conditioning=None,
               use_original_steps=False, **kwargs):

        timesteps = np.arange(self.ddpm_num_timesteps) if use_original_steps else self.ddim_timesteps
        timesteps = timesteps[:t_start]

        time_range = np.flip(timesteps)
        total_steps = timesteps.shape[0]

        iterator = tqdm(time_range, desc='Decoding image', total=total_steps)
        x_dec = x_latent
        for i, step in enumerate(iterator):
            index = total_steps - i - 1
            ts = torch.full((x_latent.shape[0],), step, device=x_latent.device, dtype=torch.long)
            x_dec, _ = self.p_sample_ddim(x_dec, cond, ts, index=index, use_original_steps=use_original_steps,
                                          unconditional_guidance_scale=unconditional_guidance_scale,
                                          unconditional_conditioning=unconditional_conditioning, **kwargs)
        return x_dec