from torchvision.models import ResNet, ResNet50_Weights
from torchvision.models.resnet import Bottleneck
import torch
import torch.nn as nn

from timm.models.vision_transformer import VisionTransformer, checkpoint_filter_fn, build_model_with_cfg
from torchvision.models import VisionTransformer as ViT_torchvision
from transformers import ViTModel, ViTFeatureExtractor

# for timm loader
class ViT_custom(VisionTransformer):
    def forward(self, x, get_embeddings=False):
        x = self.forward_features(x)
        # x.shape = bs, 197, 768
        if self.attn_pool is not None:
            x = self.attn_pool(x)
        elif self.global_pool == 'avg':
            x = x[:, self.num_prefix_tokens:].mean(dim=1)
        elif self.global_pool:
            x = x[:, 0]  # class token <-- default go to here
        # x.shape = bs, 768
        x = self.fc_norm(x)
        feat = self.head_drop(x)
        x = self.head(feat)
        
        if get_embeddings: return x, feat
        return x

# for torchvision loader
class ViTBASE(nn.Module):
    def __init__(self, num_classes=10):
        super(ViTBASE, self).__init__()
        self.base = ViTModel.from_pretrained('google/vit-base-patch16-224')
        self.final = nn.Linear(self.base.config.hidden_size, num_classes)
        self.num_classes = num_classes
        self.relu = nn.ReLU()

    def forward(self, pixel_values, get_embeddings=False):
        outputs = self.base(pixel_values=pixel_values)
        logits = self.final(outputs.last_hidden_state[:,0])

        if get_embeddings: return logits, outputs.last_hidden_state[:,0]
        return logits

def vit_base_patch16_224(pretrained: bool = False, **kwargs) -> VisionTransformer:
    """ ViT-Base (ViT-B/16) from original paper (https://arxiv.org/abs/2010.11929).
    ImageNet-1k weights fine-tuned from in21k @ 224x224, source https://github.com/google-research/vision_transformer.
    """
    model_args = dict(patch_size=16, embed_dim=768, depth=12, num_heads=12, num_classes=21843)
    model = _create_vision_transformer('vit_base_patch16_224', pretrained=pretrained, **dict(model_args, **kwargs))
    return model

def _create_vision_transformer(variant: str, pretrained: bool = False, **kwargs) -> VisionTransformer:
    if kwargs.get('features_only', None):
        raise RuntimeError('features_only not implemented for Vision Transformer models.')

    _filter_fn = checkpoint_filter_fn

    # FIXME attn pool (currently only in siglip) params removed if pool disabled, is there a better soln?
    strict = True

    return build_model_with_cfg(
        ViT_custom,
        variant,
        pretrained,
        pretrained_filter_fn=_filter_fn,
        pretrained_strict=strict,
        **kwargs,
    )

class ResNet_custom(ResNet):

    def forward(self, x, get_embeddings=False):
        # See note [TorchScript super()]
        x = self.conv1(x)
        x = self.bn1(x)
        x = self.relu(x)
        x = self.maxpool(x)

        x = self.layer1(x)
        x = self.layer2(x)
        x = self.layer3(x)
        x = self.layer4(x)

        x = self.avgpool(x)
        embeddings = torch.flatten(x, 1)
        x = self.fc(embeddings)

        if get_embeddings: return x, embeddings
        return x
    
def resnet50(pretrained=False):
    model = ResNet_custom(Bottleneck, [3, 4, 6, 3])
    if pretrained:
        weights = ResNet50_Weights.IMAGENET1K_V1
        model.load_state_dict(weights.get_state_dict(progress=False, check_hash=True))
    return model

# timm loader
def ViTLOAD(ckpt="./checkpoints/ViT_imagenet_ori0.pth"):
    model = vit_base_patch16_224().cuda()
    model.load_state_dict(torch.load(ckpt, map_location='cpu').state_dict())
    return model

# torchvision loader
def TV_ViTLOAD(ckpt="./checkpoints/ViT_imagenet_ori0.pth", num_classes=10):
    model = ViTBASE(num_classes=num_classes).cuda()
    model.load_state_dict(torch.load(ckpt, map_location='cpu').state_dict())
    return model