import os
import torch
from timm.models import create_model
from util.lr_decay import create_optimizer, LayerDecayValueAssigner

from mae_lite.exps.timm_imagenet_exp import Exp as BaseExp, Model
from mae_lite.layers import build_lr_scheduler
from loguru import logger
from mae_lite.models.models_vit import *


class Exp(BaseExp):
    def __init__(self, batch_size, max_epoch=100):
        super(Exp, self).__init__(batch_size, max_epoch)
        # MAE
        self.layer_decay = 0.85
        self.global_pool = True

        # dataset & model
        self.dataset = "ImageNet"
        self.img_size = 224

        # optimizer
        self.opt = "adamw"
        self.opt_eps = 1e-8
        self.opt_betas = (0.9, 0.999)
        self.momentum = 0.9
        self.weight_decay = 0.05
        self.clip_grad = None

        # schedule
        self.sched = "warmcos_scale"
        self.basic_lr_per_img = 1e-3 / 256
        self.warmup_lr = 0.0
        self.min_lr = 1e-6
        self.warmup_epochs = 5

        # augmentation & regularization
        self.color_jitter = 0.4
        self.aa = "rand-m9-mstd0.5-inc1"
        self.reprob = 0.25
        self.remode = "pixel"
        self.recount = 1
        # self.resplit = False
        self.mixup = 0.8
        self.cutmix = 1.0
        self.smoothing = 0.1
        self.train_interpolation = "bicubic"
        # self.drop = 0.0
        # self.drop_connect = None
        self.drop_path = 0.1
        self.attn_drop_rate = 0.0
        # self.drop_block = None
        self.encoder_arch = "vit_tiny_patch16"

        # self.num_workers = 10
        self.weights_prefix = "model"
        # self.print_interval = 10
        # self.enable_tensorboard = True
        self.pretrain_exp_name = None
        self.save_folder_prefix = "ft_"

    @property
    def exp_name(self):
        assert self.pretrain_exp_name is not None, "Please provide a valid 'pretrain_exp_name'!"
        return os.path.join(self.pretrain_exp_name, "{}eval".format(self.save_folder_prefix))

    @exp_name.setter
    def exp_name(self, value):
        # exp_name is generated by exp.pretrain_exp_name and exp.save_folder_prefix.
        # Directly setting does not affect.
        pass

    def get_model(self):
        if "model" not in self.__dict__:
            encoder = create_model(
                self.encoder_arch,
                pretrained=self.pretrained,
                num_classes=self.num_classes,
                drop_rate=self.drop,
                drop_path_rate=self.drop_path,
                attn_drop_rate=self.attn_drop_rate,
                drop_block_rate=self.drop_block,
                global_pool=self.global_pool,
            )
            self.model = Model(self, encoder)
        return self.model

    def get_optimizer(self):
        if "optimizer" not in self.__dict__:
            if "lr" not in self.__dict__:
                self.lr = self.basic_lr_per_img * self.batch_size

            num_layers = len(self.get_model().model.blocks)
            if self.layer_decay < 1.0:
                assigner = LayerDecayValueAssigner(
                    list(self.layer_decay ** (num_layers + 1 - i) for i in range(num_layers + 2))
                )
            else:
                assigner = None

            if assigner is not None:
                logger.info("Assigned values = %s" % str(assigner.values))

            skip_weight_decay_list = self.get_model().model.no_weight_decay()
            logger.info("Skip weight decay list: {}".format(skip_weight_decay_list))

            self.optimizer = create_optimizer(
                self,
                self.get_model().model,
                skip_list=skip_weight_decay_list,
                get_num_layer=assigner.get_layer_id if assigner is not None else None,
                get_layer_scale=assigner.get_scale if assigner is not None else None,
            )
        return self.optimizer

    def get_lr_scheduler(self):
        if "lr" not in self.__dict__:
            self.lr = self.basic_lr_per_img * self.batch_size
        if "warmup_lr" not in self.__dict__:
            self.warmup_lr = self.warmup_lr_per_img * self.batch_size
        if "min_lr" not in self.__dict__:
            self.min_lr = self.min_lr_per_img * self.batch_size

        optimizer = self.get_optimizer()
        iters_per_epoch = len(self.get_data_loader()["train"])
        scheduler = build_lr_scheduler(
            self.sched,
            optimizer,
            self.lr,
            total_steps=iters_per_epoch * self.max_epoch,
            warmup_steps=iters_per_epoch * self.warmup_epochs,
            warmup_lr_start=self.warmup_lr,
            end_lr=self.min_lr,
        )
        return scheduler

    def set_current_state(self, current_step, ckpt_path=None):
        if current_step == 0:
            # load pretrain ckpt
            if ckpt_path is None:
                assert self.pretrain_exp_name is not None, "Please provide a valid 'pretrain_exp_name'!"
                ckpt_path = os.path.join(self.output_dir, self.pretrain_exp_name, "last_epoch_ckpt.pth.tar")
            logger.info("Load pretrained checkpoints from {}.".format(ckpt_path))
            msg = self.set_model_weights(ckpt_path, map_location="cpu")
            logger.info("Model params {} are not loaded".format(msg.missing_keys))
            logger.info("State-dict params {} are not used".format(msg.unexpected_keys))

    def set_model_weights(self, ckpt_path, map_location="cpu"):
        if not os.path.isfile(ckpt_path):
            from torch.nn.modules.module import _IncompatibleKeys

            logger.info("No checkpoints found! Training from scratch!")
            return _IncompatibleKeys(missing_keys=None, unexpected_keys=None)
        ckpt = torch.load(ckpt_path, map_location="cpu")
        weights_prefix = self.weights_prefix
        if not weights_prefix:
            state_dict = {"model." + k: v for k, v in ckpt["model"].items()}
        else:
            if weights_prefix and not weights_prefix.endswith("."):
                weights_prefix += "."
            if all(key.startswith("module.") for key in ckpt["model"].keys()):
                weights_prefix = "module." + weights_prefix
            state_dict = {k.replace(weights_prefix, "model."): v for k, v in ckpt["model"].items()}
        msg = self.get_model().load_state_dict(state_dict, strict=False)
        return msg


if __name__ == "__main__":
    exp = Exp(2)
    exp.update({"pretrain_exp_name": "mae_lite/mae_lite_exp"})
    print(exp.exp_name)
    model = exp.get_model()
    loader = exp.get_data_loader()
    opt = exp.get_optimizer()
    scheduler = exp.get_lr_scheduler()
