import argparse
import json
import re

import anthropic
import numpy as np
from tenacity import retry, stop_after_attempt, wait_chain, wait_fixed
from tqdm import tqdm

MULTIPLE_CHOICE_TASKS = [
    "temporal_sequences",
    "disambiguation_qa",
    "date_understanding",
    "tracking_shuffled_objects_three_objects",
    "penguins_in_a_table",
    "geometric_shapes",
    "snarks",
    "ruin_names",
    "tracking_shuffled_objects_seven_objects",
    "tracking_shuffled_objects_five_objects",
    "logical_deduction_three_objects",
    "hyperbaton",
    "logical_deduction_five_objects",
    "logical_deduction_seven_objects",
    "movie_recommendation",
    "salient_translation_error_detection",
    "reasoning_about_colored_objects",
]
FREE_FORM_TASKS = [
    "multistep_arithmetic_two",
    "navigate",
    "dyck_languages",
    "word_sorting",
    "sports_understanding",
    "boolean_expressions",
    "object_counting",
    "formal_fallacies",
    "causal_judgement",
    "web_of_lies",
]
global api_key

parser = argparse.ArgumentParser()
parser.add_argument("--api_key", type=str, default="sk")
parser.add_argument("--model_index", type=str, default="claude-v1")
parser.add_argument("--task", type=str, default="all", choices=["all", "multiple_choice", "free_form"])
args = parser.parse_args()


@retry(wait=wait_chain(*[wait_fixed(3) for i in range(3)] + [wait_fixed(5) for i in range(2)] + [wait_fixed(10)]))
def completion_with_backoff(model_index, messages):
    client = anthropic.Client(api_key)
    response = client.completion(
        prompt=f"{anthropic.HUMAN_PROMPT} {messages}{anthropic.AI_PROMPT}",
        stop_sequences=[anthropic.HUMAN_PROMPT],
        model=model_index,
        max_tokens_to_sample=1000,
    )
    # print(response["completion"])
    return response["completion"]


def extract_ans_old(ans, mode):
    ans_line = ans.split("answer is ")
    if len(ans_line) == 1:
        return [ans]
    else:
        ans = ans_line[-1].strip()

    if mode == "multiple_choice":
        options = [
            "(A)",
            "(B)",
            "(C)",
            "(D)",
            "(E)",
            "(F)",
            "(G)",
            "(H)",
            "(I)",
            "(J)",
            "(K)",
            "(L)",
            "(M)",
            "(N)",
            "(O)",
            "(P)",
            "(Q)",
            "(R)",
            "(S)",
            "(T)",
            "(U)",
            "(V)",
            "(W)",
            "(X)",
            "(Y)",
            "(Z)",
        ]
        for option in options:
            if option in ans:
                ans = option[1]
                break
        return [ans]
    elif mode == "free_form":
        if ans[-1] == ".":
            ans = ans[:-1]
        return [ans]


def extract_ans(ans, mode):
    patterns = [
        "answer is ",
        "answer is: ",
        "choice is ",
        "evaluates to ",
        "evaluates to: ",
        "expression evaluates to ",
        "expression is ",
        "may answer "
        "result is ",
        "result is: ",
    ]

    options = [
        "(A)",
        "(B)",
        "(C)",
        "(D)",
        "(E)",
        "(F)",
        "(G)",
        "(H)",
        "(I)",
        "(J)",
        "(K)",
        "(L)",
        "(M)",
        "(N)",
        "(O)",
        "(P)",
        "(Q)",
        "(R)",
        "(S)",
        "(T)",
        "(U)",
        "(V)",
        "(W)",
        "(X)",
        "(Y)",
        "(Z)",
    ]

    lines = []
    temps = []
    ans_lines = []

    for pattern in patterns:
        matches = re.findall(f"{pattern}(.*?)[\.|\n]", ans)
        if matches == []:
            break
        if matches:
            lines.append(matches[-1])
            break

    if len(lines) != 0:
        for line in lines:
            ans_wl = line
            ans_fw = line.split(" ")[0]
            ans_ld = line.split(" ")[-1]

            ans_wl in temps or temps.append(ans_wl)
            ans_fw in temps or temps.append(ans_fw)
            ans_ld in temps or temps.append(ans_ld)
    else:
        lines = ans.split("\n")
        lines = [line for line in lines if line.strip() != ""]

        ans_fw = lines[0].split(" ")[0]
        ans_ld = lines[-1].split(" ")[-1]

        ans_fw in temps or temps.append(ans_fw)
        ans_ld in temps or temps.append(ans_ld)

    for temp in temps:
        if len(temp) > 0:
            if temp[-1] in [".", ","]:
                temp = temp[:-1]
            ans_lines.append(temp)

    if mode == "multiple_choice":
        for option in options:
            if option in ans_lines[0]:
                return [option[1]]
        return [ans_lines[0]]
    elif mode == "free_form":
        return ans_lines


def run_tasks(tasks, mode, model_index):
    for task in tqdm(tasks):
        # print("Testing %s ..." % task)
        acc = 0
        task_data = json.load(open("data/%s.json" % task))

        # exp 01
        task_prompt = open("lib_prompt/%s.txt" % task, "r").read()

        # exp 02
        # task_prompt = open("lib_prompt_multiround_claude_instant/%s.txt" % task, "r").read()

        # exp 03
        # task_prompt = open("lib_prompt_multiround_claude_instant_test/%s.txt" % task, "r").read()

        print_first = False
        with open("outputs/test_%s_%s.txt" % (model_index.replace("-", "_"), task), "w") as fd:
            for q_ in tqdm(task_data["examples"]):
                # TODO Refactoring Template Builder
                # TODO The current one is specially made for Claude. "Let's think step by step." is included in Human.

                # exp 01
                q = "\nQ: " + q_["input"]
                prompt_q = task_prompt + "\n" + q + "\nA: Let's think step by step."

                # exp 02
                # q = "\nHuman: " + q_["input"]
                # prompt_q = task_prompt + q + "\n" + "\nAssistant:\nLet's think step by step.\n"

                # exp 03
                # q = "\nHuman: " + q_["input"]
                # prompt_q = task_prompt + q + "\nLet's think step by step.\n" + "\nAssistant:\n"

                if print_first:
                    print("First prompt: ")
                    print(prompt_q)
                    print_first = False

                response = completion_with_backoff(model_index, prompt_q)

                ans_model = response
                ans_ = extract_ans(ans_model, mode)

                if mode == "multiple_choice":
                    a = q_["target"][1]
                elif mode == "free_form":
                    a = q_["target"]

                ans_eval = False

                if a in ans_:
                    acc += 1
                    ans_eval = True
                else:
                    ans_eval = False

                fd.write(
                    f"[Q:]\n{q}\n[Q_P:]\n{prompt_q}\n[A_model:]\n{ans_model}\n[A_extract:]\n{ans_}\n[A_target:]\n{a}\n[A_match:]\n{ans_eval}\n"
                )
            print("%s acc %.4f" % (task, acc / len(task_data["examples"])))
            fd.write("%s acc %.4f" % (task, acc / len(task_data["examples"])))


def main(args, multiple_choice_tasks=MULTIPLE_CHOICE_TASKS, free_form_tasks=FREE_FORM_TASKS):
    api_key = args.api_key
    model_index = args.model_index
    run_multiple_choice = args.task == "all" or args.task == "multiple_choice"
    run_free_form = args.task == "all" or args.task == "free_form"

    if run_multiple_choice:
        run_tasks(multiple_choice_tasks, mode="multiple_choice", model_index=model_index)
    if run_free_form:
        run_tasks(free_form_tasks, mode="free_form", model_index=model_index)
    return


if __name__ == "__main__":
    main(args)
