# coding=utf-8
# Copyright 2021 The HuggingFace Inc. team. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import argparse
from ast import arg
from curses import raw
from itertools import chain
import logging
import math
import os
from sched import scheduler
import sys
import copy
import json
import random
from pathlib import Path

import datasets
from datasets import load_dataset, load_metric, load_from_disk, ClassLabel
import torch
from torch.utils.data import DataLoader
import torch.nn.functional as F
import torch.distributions as dist
from tqdm.auto import tqdm

import transformers
from accelerate import Accelerator
from huggingface_hub import Repository
from transformers import (
    AdamW,
    AutoConfig,
    AutoModel,
    AutoModelForSequenceClassification,
    AutoModelForMultipleChoice,
    AutoModelForMaskedLM,
    AutoTokenizer,
    DataCollatorWithPadding,
    DataCollatorForLanguageModeling,
    PretrainedConfig,
    SchedulerType,
    default_data_collator,
    get_scheduler,
    set_seed,
)

from models import Variational, Generator, VAE, DataCollactorForLanguageModelingMC, VariationalForMC
from utils import check_dir, ECE

logger = logging.getLogger(__name__)

cls_task_list = ["snli", "mnli", "qqp", "TwitterPPDB"]
mc_task_list = ["swag", "hellaswag"]

task_to_keys = {
    "mnli": ("premise", "hypothesis"),
    "snli": ("premise", "hypothesis"),
    "qqp": ("question1", "question2"),
    "TwitterPPDB": ("question1", "question2"),
}


def parse_args():
    parser = argparse.ArgumentParser(description="Finetune a transformers model on a text classification task")
    parser.add_argument(
        "--task_name",
        type=str,
        default=None,
        help="The name of the glue task to train on.",
    )
    parser.add_argument(
        "--train_file", type=str, default=None, help="A csv or a json file containing the training data."
    )
    parser.add_argument(
        "--validation_file", type=str, default=None, help="A csv or a json file containing the validation data."
    )
    parser.add_argument(
        "--max_length",
        type=int,
        default=256,
        help=(
            "The maximum total input sequence length after tokenization. Sequences longer than this will be truncated,"
            " sequences shorter will be padded if `--pad_to_max_lengh` is passed."
        ),
    )
    parser.add_argument(
        "--pad_to_max_length",
        action="store_true",
        help="If passed, pad all samples to `max_length`. Otherwise, dynamic padding is used.",
    )
    parser.add_argument(
        "--model_name_or_path",
        type=str,
        help="Path to pretrained model or model identifier from huggingface.co/models.",
        required=True,
    )
    parser.add_argument(
        "--use_slow_tokenizer",
        action="store_true",
        help="If passed, will use a slow tokenizer (not backed by the 🤗 Tokenizers library).",
    )
    parser.add_argument(
        "--per_device_train_batch_size",
        type=int,
        default=8,
        help="Batch size (per device) for the training dataloader.",
    )
    parser.add_argument(
        "--per_device_eval_batch_size",
        type=int,
        default=1,
        help="Batch size (per device) for the evaluation dataloader.",
    )
    parser.add_argument(
        "--learning_rate",
        type=float,
        default=5e-5,
        help="Initial learning rate (after the potential warmup period) to use.",
    )
    parser.add_argument("--weight_decay", type=float, default=0.0, help="Weight decay to use.")
    parser.add_argument("--num_train_epochs", type=int, default=3, help="Total number of training epochs to perform.")
    parser.add_argument(
        "--max_train_steps",
        type=int,
        default=None,
        help="Total number of training steps to perform. If provided, overrides num_train_epochs.",
    )
    parser.add_argument(
        "--gradient_accumulation_steps",
        type=int,
        default=1,
        help="Number of updates steps to accumulate before performing a backward/update pass.",
    )
    parser.add_argument(
        "--lr_scheduler_type",
        type=SchedulerType,
        default="linear",
        help="The scheduler type to use.",
        choices=["linear", "cosine", "cosine_with_restarts", "polynomial", "constant", "constant_with_warmup"],
    )
    parser.add_argument(
        "--num_warmup_steps", type=int, default=0, help="Number of steps for the warmup in the lr scheduler."
    )
    parser.add_argument("--output_dir", type=str, default=None, help="Where to store the final model.")
    parser.add_argument("--conf_dir", type=str, default=None, help="Where to store the output of model.")
    parser.add_argument("--seed", type=int, default=None, help="A seed for reproducible training.")
    parser.add_argument("--push_to_hub", action="store_true", help="Whether or not to push the model to the Hub.")
    parser.add_argument(
        "--hub_model_id", type=str, help="The name of the repository to keep in sync with the local `output_dir`."
    )
    parser.add_argument("--hub_token", type=str, help="The token to use to push to the Model Hub.")
    parser.add_argument("--do_eval", action="store_true")
    parser.add_argument("--model_task", type=str, default=None)
    parser.add_argument("--log_epoch", action="store_true")
    parser.add_argument("--delta", type=str, default=None)
    parser.add_argument("--delta_config", type=str, default="")
    parser.add_argument("--lora_r", type=int, default=8)
    parser.add_argument("--lora_alpha", type=int, default=8)
    parser.add_argument("--delta_path", type=str, default=None)
    parser.add_argument("--ckpt_path", type=str, default=None)
    parser.add_argument("--ft_path", type=str, default=None)
    parser.add_argument("--mlm_prob", type=float)
    parser.add_argument("--plot", action="store_true")
    parser.add_argument("--fig_title", type=str, default="test")
    args = parser.parse_args()

    # Sanity checks
    if args.task_name is None and args.train_file is None and args.validation_file is None:
        raise ValueError("Need either a task name or a training/validation file.")
    else:
        if args.train_file is not None:
            extension = args.train_file.split(".")[-1]
            assert extension in ["csv", "json"], "`train_file` should be a csv or a json file."
        if args.validation_file is not None:
            extension = args.validation_file.split(".")[-1]
            assert extension in ["csv", "json"], "`validation_file` should be a csv or a json file."
    return args


def main():
    args = parse_args()
    if args.do_eval:
        # Initialize the accelerator. We will let the accelerator handle device placement for us in this example.
        accelerator = Accelerator()
        # Make one log on every process with the configuration for debugging.
        logging.basicConfig(
            format="%(asctime)s - %(levelname)s - %(name)s - %(message)s",
            datefmt="%m/%d/%Y %H:%M:%S",
            level=logging.INFO,
        )
        logger.info(accelerator.state)

        # Setup logging, we only want one process per machine to log things on the screen.
        # accelerator.is_local_main_process is only True for one process per machine.
        logger.setLevel(logging.INFO if accelerator.is_local_main_process else logging.ERROR)
        if accelerator.is_local_main_process:
            datasets.utils.logging.set_verbosity_warning()
            transformers.utils.logging.set_verbosity_info()
        else:
            datasets.utils.logging.set_verbosity_error()
            transformers.utils.logging.set_verbosity_error()

        # If passed along, set the training seed now.
        if args.seed is not None:
            set_seed(args.seed)

        accelerator.wait_for_everyone()

        # Get the datasets: you can either provide your own CSV/JSON training and evaluation files (see below)
        # or specify a GLUE benchmark task (the dataset will be downloaded automatically from the datasets Hub).

        # For CSV/JSON files, this script will use as labels the column called 'label' and as pair of sentences the
        # sentences in columns called 'sentence1' and 'sentence2' if such column exists or the first two columns not named
        # label if at least two columns are provided.

        # If the CSVs/JSONs contain only one non-label column, the script does single sentence classification on this
        # single column. You can easily tweak this behavior (see below)

        # In distributed training, the load_dataset function guarantee that only one local process can concurrently
        # download the dataset.
        if args.task_name is not None:
            # Downloading and loading a dataset from the hub.
            try:
                raw_datasets = load_from_disk(f'./data/processed_data/{args.task_name}')
            except:
                raw_datasets = load_from_disk(f'./data/{args.task_name}')
        else:
            # Loading the dataset from local csv or json file.
            data_files = {}
            if args.train_file is not None:
                data_files["train"] = args.train_file
            if args.validation_file is not None:
                data_files["validation"] = args.validation_file
            extension = (args.train_file if args.train_file is not None else args.valid_file).split(".")[-1]
            raw_datasets = load_dataset(extension, data_files=data_files)
        # See more about loading any type of standard or custom dataset at
        # https://huggingface.co/docs/datasets/loading_datasets.html.



        # Load pretrained model and tokenizer
        #
        # In distributed training, the .from_pretrained methods guarantee that only one local process can concurrently
        # download model & vocab.
        config = AutoConfig.from_pretrained(args.model_name_or_path)
        tokenizer = AutoTokenizer.from_pretrained(args.model_name_or_path, use_fast=not args.use_slow_tokenizer)
        
        if args.ckpt_path is None:
            model = AutoModelForMaskedLM.from_pretrained(
                args.model_name_or_path,
                from_tf=bool(".ckpt" in args.model_name_or_path),
                config=config,
            )
            if args.ft_path is not None:
                config_ft = AutoConfig.from_pretrained(args.ft_path)
                model_ft = AutoModelForMultipleChoice.from_pretrained(
                    args.ft_path,
                    from_tf=bool(".ckpt" in args.model_name_or_path),
                    config=config_ft,
                )
                model.roberta = model_ft.roberta
            if args.delta_path is not None:
                from opendelta import AutoDeltaConfig
                from opendelta.auto_delta import AutoDeltaModel
                # Pooler.
                model_ft = AutoModelForMultipleChoice.from_pretrained(
                        args.model_name_or_path,
                        from_tf=bool(".ckpt" in args.model_name_or_path),
                        config=config,
                    )
                delta_args = json.loads(Path(args.delta_config).read_text())
                delta_config = AutoDeltaConfig.from_dict(delta_args)
                delta_model = AutoDeltaModel.from_config(delta_config, backbone_model=model_ft)
                delta_model.freeze_module(set_state_dict = True)
                delta_model.log(delta_ratio=True, trainable_ratio=True, visualization=True)
                model_ft.load_state_dict(torch.load(os.path.join(args.delta_path, "model.ckpt")), strict=False)
                model.roberta = model_ft.roberta
                # Sanity Check
                # print(model.roberta.encoder.layer[0].attention.self.query.lora.lora_A.shape)
        else:
            model_pretrained = AutoModelForMaskedLM.from_pretrained(
                args.model_name_or_path,
                from_tf=bool(".ckpt" in args.model_name_or_path),
                config=config,
            )

            model_mc = AutoModel.from_pretrained(
                args.model_name_or_path,
                from_tf=bool(".ckpt" in args.model_name_or_path),
                config=config,
                add_pooling_layer=False,
            )
            model_mc.resize_token_embeddings(len(tokenizer))

            variational = VariationalForMC(model_mc)
            generator = Generator(copy.deepcopy(model_pretrained.lm_head), config)
            model = VAE(config, variational, generator)
            model.load_state_dict(torch.load(os.path.join(args.ckpt_path, "model.ckpt")), strict=False)
        
        # Preprocessing the datasets
        if args.task_name in cls_task_list:
            # Labels
            if args.task_name is not None:
                is_regression = args.task_name == "stsb"
                if not is_regression:
                    try:
                        label_list = raw_datasets["train"].features["label"].names
                        num_labels = len(label_list)
                    except:
                        # A useful fast method:
                        # https://huggingface.co/docs/datasets/package_reference/main_classes.html#datasets.Dataset.unique
                        label_list = raw_datasets["train"].unique("label")
                        label_list.sort()  # Let's sort it for determinism
                        num_labels = len(label_list)
            else:
                # Trying to have good defaults here, don't hesitate to tweak to your needs.
                is_regression = raw_datasets["train"].features["label"].dtype in ["float32", "float64"]
                if is_regression:
                    num_labels = 1
                else:
                    # A useful fast method:
                    # https://huggingface.co/docs/datasets/package_reference/main_classes.html#datasets.Dataset.unique
                    label_list = raw_datasets["train"].unique("label")
                    label_list.sort()  # Let's sort it for determinism
                    num_labels = len(label_list)

            sentence1_key, sentence2_key = task_to_keys[args.task_name]

            # Some models have set the order of the labels to use, so let's make sure we do use it.
            label_to_id = None
            if (
                model.config.label2id != PretrainedConfig(num_labels=num_labels).label2id
                and args.task_name is not None
                and not is_regression
            ):
                # Some have all caps in their config, some don't.
                label_name_to_id = {k.lower(): v for k, v in model.config.label2id.items()}
                if list(sorted(label_name_to_id.keys())) == list(sorted(label_list)):
                    logger.info(
                        f"The configuration of the model provided the following label correspondence: {label_name_to_id}. "
                        "Using it!"
                    )
                    label_to_id = {i: label_name_to_id[label_list[i]] for i in range(num_labels)}
                else:
                    logger.warning(
                        "Your model seems to have been trained with labels, but they don't match the dataset: ",
                        f"model labels: {list(sorted(label_name_to_id.keys()))}, dataset labels: {list(sorted(label_list))}."
                        "\nIgnoring the model labels as a result.",
                    )
            elif args.task_name is None:
                label_to_id = {v: i for i, v in enumerate(label_list)}
            
            if label_to_id is not None:
                model.config.label2id = label_to_id
                model.config.id2label = {id: label for label, id in config.label2id.items()}
            elif args.task_name is not None and not is_regression:
                model.config.label2id = {l: i for i, l in enumerate(label_list)}
                model.config.id2label = {id: label for label, id in config.label2id.items()}

            padding = "max_length" if args.pad_to_max_length else False

            glue_task = ["mnli", "qqp"]

            def preprocess_function(examples):
                # Tokenize the texts
                texts = (
                    (examples[sentence1_key],) if sentence2_key is None else (examples[sentence1_key], examples[sentence2_key])
                )
                result = tokenizer(*texts, padding=padding, max_length=args.max_length, truncation=True)

                if "label" in examples:
                    if label_to_id is not None and args.task_name not in glue_task:
                        # Map labels to IDs (not necessary for GLUE tasks)
                        result["labels"] = [label_to_id[l] for l in examples["label"]]
                    else:
                        # In all cases, rename the column to labels because the model will expect that.
                        result["labels"] = examples["label"]
                return result

            with accelerator.main_process_first():
                processed_datasets = raw_datasets.map(
                    preprocess_function,
                    batched=True,
                    remove_columns=raw_datasets["train"].column_names,
                    desc="Running tokenizer on dataset",
                )
            # eval_dataset = processed_datasets["validation_matched" if args.task_name == "mnli" else "validation"]
            # eval_dataset = processed_datasets["validation"]
            try:
                eval_dataset = processed_datasets["test"]
            except:
                eval_dataset = processed_datasets["validation"]
        
        elif args.task_name in mc_task_list:
            # Preprocessing the datasets.
            # First we tokenize all the texts.
            padding = "max_length" if args.pad_to_max_length else False

            # When using your own dataset or a different dataset from swag, you will probably need to change this.
            #NOTE: Only Swag & Hellaswag are supported now.
            if args.task_name == 'swag':
                ending_names = [f"ending{i}" for i in range(4)]
                context_name = "sent1"
                question_header_name = "sent2"

                def get_sentences(examples):
                    first_sentences = [[context] * 4 for context in examples[context_name]]
                    question_headers = examples[question_header_name]
                    second_sentences = [
                        [f"{header} {examples[end][i]}" for end in ending_names] for i, header in enumerate(question_headers)
                    ]
                    return first_sentences, second_sentences
            elif args.task_name =="hellaswag":
                ending_names = "endings"
                context_name = "ctx_a"
                question_header_name = "ctx_b"

                def get_sentences(examples):
                    first_sentences = [[context] * 4 for context in examples[context_name]]
                    question_headers = examples[question_header_name]
                    second_sentences = [
                        [f"{header} {end}" for end in examples[ending_names][i]] for i, header in enumerate(question_headers)
                    ]
                    return first_sentences, second_sentences
            else:
                raise NotImplementedError()
            column_names = raw_datasets["train"].column_names
            label_column_name = "label" if "label" in column_names else "labels"
            if args.task_name == "hellaswag":
                raw_datasets["train"] = raw_datasets["train"].cast_column(label_column_name, ClassLabel(num_classes=4, names=['0', '1', '2', '3'], id=None))
                raw_datasets["validation"] = raw_datasets["validation"].cast_column(label_column_name, ClassLabel(num_classes=4, names=['0', '1', '2', '3'], id=None))
                raw_datasets["test"] = raw_datasets["test"].cast_column(label_column_name, ClassLabel(num_classes=4, names=['0', '1', '2', '3'], id=None))
            def preprocess_function(examples):
                first_sentences, second_sentences = get_sentences(examples)
                labels = examples[label_column_name]
                
                # Flatten out
                first_sentences = list(chain(*first_sentences))
                second_sentences = list(chain(*second_sentences))

                # Tokenize
                tokenized_examples = tokenizer(
                    first_sentences,
                    second_sentences,
                    max_length=args.max_length,
                    padding=padding,
                    truncation=True,
                )
                # Un-flatten
                tokenized_inputs = {k: [v[i : i + 4] for i in range(0, len(v), 4)] for k, v in tokenized_examples.items()}
                tokenized_inputs["labels"] = labels
                return tokenized_inputs

            with accelerator.main_process_first():
                processed_datasets = raw_datasets.map(
                    preprocess_function, batched=True, remove_columns=raw_datasets["train"].column_names
                )
            eval_dataset = processed_datasets["test"]
        
        else:
            # Use processing from run_mlm_no_trainer.py
            column_names = raw_datasets["train"].column_names
            try:
                raw_datasets = raw_datasets["test"]
            except:
                raw_datasets = raw_datasets["validation"]
            text_column_name = "text" if "text" in column_names else column_names[0]
            max_seq_length = args.max_length
            def tokenize_function(examples):
                return tokenizer(examples[text_column_name], return_special_tokens_mask=True)

            with accelerator.main_process_first():
                tokenized_datasets = raw_datasets.map(
                    tokenize_function,
                    batched=True,
                    num_proc=8,
                    remove_columns=column_names,
                    desc="Running tokenizer on every text in dataset",
                )

            # Main data processing function that will concatenate all texts from our dataset and generate chunks of
            # max_seq_length.
            def group_texts(examples):
                # Concatenate all texts.
                concatenated_examples = {k: list(chain(*examples[k])) for k in examples.keys()}
                total_length = len(concatenated_examples[list(examples.keys())[0]])
                # We drop the small remainder, we could add padding if the model supported it instead of this drop, you can
                # customize this part to your needs.
                if total_length >= max_seq_length:
                    total_length = (total_length // max_seq_length) * max_seq_length
                # Split by chunks of max_len.
                result = {
                    k: [t[i : i + max_seq_length] for i in range(0, total_length, max_seq_length)]
                    for k, t in concatenated_examples.items()
                }
                return result
            
            with accelerator.main_process_first():
                tokenized_datasets = tokenized_datasets.map(
                    group_texts,
                    batched=True,
                    num_proc=8,
                    desc=f"Grouping texts in chunks of {max_seq_length}",
                )
            eval_dataset = tokenized_datasets
            
        



        # Log a few random samples from the training set:
        for index in random.sample(range(len(eval_dataset)), 3):
            logger.info(f"Sample {index} of the training set: {eval_dataset[index]}.")


        if args.task_name in mc_task_list:
            data_collator = DataCollactorForLanguageModelingMC(tokenizer, mlm_probability=args.mlm_prob)
        else:
            data_collator = DataCollatorForLanguageModeling(tokenizer, mlm_probability=args.mlm_prob)
        eval_dataloader = DataLoader(eval_dataset, collate_fn=data_collator, batch_size=args.per_device_eval_batch_size)

        # lm_model_pretrained, lm_model_FT, lm_model_Delta, lm_model_VAE_D, lm_model_VAE_P, eval_dataloader = accelerator.prepare(
        #     lm_model_pretrained, lm_model_FT, lm_model_Delta, lm_model_VAE_D, lm_model_VAE_P, eval_dataloader
        # )

        model, eval_dataloader = accelerator.prepare(
            model, eval_dataloader
        )

        model.eval()
        output_dicts = []
        
        for step, batch in enumerate(tqdm(eval_dataloader)):
            with torch.no_grad():
                labels = batch.pop("labels", None)
                if args.ckpt_path is None:
                    outputs = model(**batch)
                    logits_px = outputs.logits
                else:
                    logits_mlm_cls, z1 = model.variational(**batch, flat=False)
                    logits_px = model.generator(z1) # MLM + Classification

                mask_token_index = (labels.view(-1) != -100).nonzero().squeeze(-1)
                labels = labels.view(-1)[mask_token_index]
                logits = logits_px.detach().view(-1, model.config.vocab_size)[mask_token_index]
                for j in range(logits.size(0)):
                    probs = F.softmax(logits[j], -1)
                    output_dict = {
                    'true': labels[j].item(),
                    'pred': logits[j].argmax().item(),
                    'conf': probs.max().item(),
                    }
                    output_dicts.append(output_dict)
        
        check_dir(args.conf_dir)
        output_path = os.path.join(args.conf_dir, 'res.json')
        print(f'writing outputs to \'{output_path}\'')

        with open(output_path, 'w+') as f:
            for i, output_dict in enumerate(tqdm(output_dicts)):
                output_dict_str = json.dumps(output_dict)
                f.write(f'{output_dict_str}\n')
                
if __name__ == "__main__":
    main()