# coding=utf-8
# Copyright 2021 The HuggingFace Inc. team. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
""" Finetuning a 🤗 Transformers model for sequence classification on GLUE."""
import argparse
import copy
import logging
import math
import os
import sys
import json
import random
from pathlib import Path

import datasets
from datasets import load_dataset, load_metric, load_from_disk
import numpy as np
import torch
from torch.utils.data import DataLoader
import torch.nn.functional as F
from torch.nn import CrossEntropyLoss
import torch.distributions as dist
from tqdm.auto import tqdm

import transformers
from accelerate import Accelerator
from huggingface_hub import Repository
from transformers import (
    AdamW,
    AutoConfig,
    AutoModelForSequenceClassification,
    AutoModelForMaskedLM,
    AutoTokenizer,
    DataCollatorWithPadding,
    PretrainedConfig,
    SchedulerType,
    default_data_collator,
    get_scheduler,
    set_seed,
)
from models import Variational, Generator, VAE, DataCollatorForMultiTask
from utils import check_dir

logger = logging.getLogger(__name__)


task_to_keys = {
    "mnli": ("premise", "hypothesis"),
    "snli": ("premise", "hypothesis"),
    "qqp": ("question1", "question2"),
    "TwitterPPDB": ("question1", "question2"),
}


def parse_args():
    parser = argparse.ArgumentParser(description="Finetune a transformers model on a text classification task")
    parser.add_argument(
        "--task_name",
        type=str,
        default=None,
        help="The name of the glue task to train on.",
        choices=list(task_to_keys.keys()),
    )
    parser.add_argument(
        "--train_file", type=str, default=None, help="A csv or a json file containing the training data."
    )
    parser.add_argument(
        "--validation_file", type=str, default=None, help="A csv or a json file containing the validation data."
    )
    parser.add_argument(
        "--max_length",
        type=int,
        default=128,
        help=(
            "The maximum total input sequence length after tokenization. Sequences longer than this will be truncated,"
            " sequences shorter will be padded if `--pad_to_max_lengh` is passed."
        ),
    )
    parser.add_argument(
        "--pad_to_max_length",
        action="store_true",
        help="If passed, pad all samples to `max_length`. Otherwise, dynamic padding is used.",
    )
    parser.add_argument(
        "--model_name_or_path",
        type=str,
        help="Path to pretrained model or model identifier from huggingface.co/models.",
        required=True,
    )
    parser.add_argument(
        "--use_slow_tokenizer",
        action="store_true",
        help="If passed, will use a slow tokenizer (not backed by the 🤗 Tokenizers library).",
    )
    parser.add_argument(
        "--per_device_train_batch_size",
        type=int,
        default=8,
        help="Batch size (per device) for the training dataloader.",
    )
    parser.add_argument(
        "--per_device_eval_batch_size",
        type=int,
        default=8,
        help="Batch size (per device) for the evaluation dataloader.",
    )
    parser.add_argument("--output_dir", type=str, default=None, help="Where to store the final model.")
    parser.add_argument("--conf_dir", type=str, default=None, help="Where to store the output of model.")
    parser.add_argument("--seed", type=int, default=None, help="A seed for reproducible training.")
    parser.add_argument("--ckpt_path", type=str, default=None)
    parser.add_argument("--ft_path", type=str, default=None)
    parser.add_argument("--delta_path", type=str, default=None)
    parser.add_argument("--delta", type=str, default=None)
    parser.add_argument("--delta_config", type=str, default="")
    parser.add_argument("--num_iters", type=int, default=10)
    parser.add_argument("--rejection", action="store_true")
    parser.add_argument("--original_label", type=int, default=0)
    parser.add_argument("--push_to_hub", action="store_true", help="Whether or not to push the model to the Hub.")
    parser.add_argument(
        "--hub_model_id", type=str, help="The name of the repository to keep in sync with the local `output_dir`."
    )
    parser.add_argument("--hub_token", type=str, help="The token to use to push to the Model Hub.")
    args = parser.parse_args()

    # Sanity checks
    if args.task_name is None and args.train_file is None and args.validation_file is None:
        raise ValueError("Need either a task name or a training/validation file.")
    else:
        if args.train_file is not None:
            extension = args.train_file.split(".")[-1]
            assert extension in ["csv", "json"], "`train_file` should be a csv or a json file."
        if args.validation_file is not None:
            extension = args.validation_file.split(".")[-1]
            assert extension in ["csv", "json"], "`validation_file` should be a csv or a json file."

    if args.push_to_hub:
        assert args.output_dir is not None or args.do_train is False, "Need an `output_dir` to create a repo when `--push_to_hub` is passed."

    return args


def main():
    args = parse_args()

    # Initialize the accelerator. We will let the accelerator handle device placement for us in this example.
    accelerator = Accelerator()
    # Make one log on every process with the configuration for debugging.
    logging.basicConfig(
        format="%(asctime)s - %(levelname)s - %(name)s - %(message)s",
        datefmt="%m/%d/%Y %H:%M:%S",
        level=logging.INFO,
    )
    logger.info(accelerator.state)

    # Setup logging, we only want one process per machine to log things on the screen.
    # accelerator.is_local_main_process is only True for one process per machine.
    logger.setLevel(logging.INFO if accelerator.is_local_main_process else logging.ERROR)
    if accelerator.is_local_main_process:
        datasets.utils.logging.set_verbosity_warning()
        transformers.utils.logging.set_verbosity_info()
    else:
        datasets.utils.logging.set_verbosity_error()
        transformers.utils.logging.set_verbosity_error()

    # If passed along, set the training seed now.
    if args.seed is not None:
        set_seed(args.seed)

    accelerator.wait_for_everyone()

    # Get the datasets: you can either provide your own CSV/JSON training and evaluation files (see below)
    # or specify a GLUE benchmark task (the dataset will be downloaded automatically from the datasets Hub).

    # For CSV/JSON files, this script will use as labels the column called 'label' and as pair of sentences the
    # sentences in columns called 'sentence1' and 'sentence2' if such column exists or the first two columns not named
    # label if at least two columns are provided.

    # If the CSVs/JSONs contain only one non-label column, the script does single sentence classification on this
    # single column. You can easily tweak this behavior (see below)

    # In distributed training, the load_dataset function guarantee that only one local process can concurrently
    # download the dataset.
    if args.task_name is not None:
        # Downloading and loading a dataset from the hub.
        raw_datasets = load_from_disk(f'./data/processed_data/{args.task_name}')
    else:
        # Loading the dataset from local csv or json file.
        data_files = {}
        if args.train_file is not None:
            data_files["train"] = args.train_file
        if args.validation_file is not None:
            data_files["validation"] = args.validation_file
        extension = (args.train_file if args.train_file is not None else args.valid_file).split(".")[-1]
        raw_datasets = load_dataset(extension, data_files=data_files)
    # See more about loading any type of standard or custom dataset at
    # https://huggingface.co/docs/datasets/loading_datasets.html.

    # Labels
    if args.task_name is not None:
        is_regression = args.task_name == "stsb"
        if not is_regression:
            try:
                label_list = raw_datasets["train"].features["label"].names
                num_labels = len(label_list)
            except:
                # A useful fast method:
                # https://huggingface.co/docs/datasets/package_reference/main_classes.html#datasets.Dataset.unique
                label_list = raw_datasets["train"].unique("label")
                label_list.sort()  # Let's sort it for determinism
                num_labels = len(label_list)
    else:
        # Trying to have good defaults here, don't hesitate to tweak to your needs.
        is_regression = raw_datasets["train"].features["label"].dtype in ["float32", "float64"]
        if is_regression:
            num_labels = 1
        else:
            # A useful fast method:
            # https://huggingface.co/docs/datasets/package_reference/main_classes.html#datasets.Dataset.unique
            label_list = raw_datasets["train"].unique("label")
            label_list.sort()  # Let's sort it for determinism
            num_labels = len(label_list)

    # Load pretrained model and tokenizer
    #
    # In distributed training, the .from_pretrained methods guarantee that only one local process can concurrently
    # download model & vocab.
    config = AutoConfig.from_pretrained(args.model_name_or_path, num_labels=num_labels, finetuning_task=args.task_name)
    tokenizer = AutoTokenizer.from_pretrained(args.model_name_or_path, use_fast=not args.use_slow_tokenizer)
    
    if args.ckpt_path is None:
        model = AutoModelForMaskedLM.from_pretrained(
            args.model_name_or_path,
            from_tf=bool(".ckpt" in args.model_name_or_path),
            config=config,
        )
        if args.ft_path is not None:
            model_ft = AutoModelForSequenceClassification.from_pretrained(
                args.ft_path,
                from_tf=bool(".ckpt" in args.model_name_or_path),
                config=config,
            )
            model.roberta = model_ft.roberta
        if args.delta_path is not None:
            from opendelta import AutoDeltaConfig
            from opendelta.auto_delta import AutoDeltaModel
            model_ft = AutoModelForSequenceClassification.from_pretrained(
                    args.model_name_or_path,
                    from_tf=bool(".ckpt" in args.model_name_or_path),
                    config=config,
                )
            delta_args = json.loads(Path(args.delta_config).read_text())
            delta_config = AutoDeltaConfig.from_dict(delta_args)
            delta_model = AutoDeltaModel.from_config(delta_config, backbone_model=model)
            delta_model.freeze_module(set_state_dict = True)
            delta_model.log(delta_ratio=True, trainable_ratio=True, visualization=True)
            model.load_state_dict(torch.load(os.path.join(args.delta_path, "model.ckpt")), strict=False)
            # model.roberta = model_ft.roberta
            # Sanity Check
            # print(model.roberta.encoder.layer[0].attention.self.query.lora.lora_A.shape)
    else:
        model_pretrained = AutoModelForMaskedLM.from_pretrained(
            args.model_name_or_path,
            from_tf=bool(".ckpt" in args.model_name_or_path),
            config=config,
        )

        variational = Variational(copy.deepcopy(model_pretrained.roberta))
        generator = Generator(copy.deepcopy(model_pretrained.lm_head), config)
        model = VAE(config, variational, generator)
        model.load_state_dict(torch.load(os.path.join(args.ckpt_path, "model.ckpt")), strict=False)

    # Preprocessing the datasets
    if args.task_name is not None:
        sentence1_key, sentence2_key = task_to_keys[args.task_name]
    else:
        # Again, we try to have some nice defaults but don't hesitate to tweak to your use case.
        non_label_column_names = [name for name in raw_datasets["train"].column_names if name != "label"]
        if "sentence1" in non_label_column_names and "sentence2" in non_label_column_names:
            sentence1_key, sentence2_key = "sentence1", "sentence2"
        else:
            if len(non_label_column_names) >= 2:
                sentence1_key, sentence2_key = non_label_column_names[:2]
            else:
                sentence1_key, sentence2_key = non_label_column_names[0], None

    # Some models have set the order of the labels to use, so let's make sure we do use it.
    label_to_id = None
    if (
        model.config.label2id != PretrainedConfig(num_labels=num_labels).label2id
        and args.task_name is not None
        and not is_regression
    ):
        # Some have all caps in their config, some don't.
        label_name_to_id = {k.lower(): v for k, v in model.config.label2id.items()}
        if list(sorted(label_name_to_id.keys())) == list(sorted(label_list)):
            logger.info(
                f"The configuration of the model provided the following label correspondence: {label_name_to_id}. "
                "Using it!"
            )
            label_to_id = {i: label_name_to_id[label_list[i]] for i in range(num_labels)}
        else:
            logger.warning(
                "Your model seems to have been trained with labels, but they don't match the dataset: ",
                f"model labels: {list(sorted(label_name_to_id.keys()))}, dataset labels: {list(sorted(label_list))}."
                "\nIgnoring the model labels as a result.",
            )
    elif args.task_name is None:
        label_to_id = {v: i for i, v in enumerate(label_list)}
    
    if label_to_id is not None:
        model.config.label2id = label_to_id
        model.config.id2label = {id: label for label, id in config.label2id.items()}
    elif args.task_name is not None and not is_regression:
        model.config.label2id = {l: i for i, l in enumerate(label_list)}
        model.config.id2label = {id: label for label, id in config.label2id.items()}

    padding = "max_length" if args.pad_to_max_length else False

    glue_task = ["mnli", "qqp"]

    # Using test dataset only
    # raw_datasets = raw_datasets['test'].filter(lambda example: example["label"] == args.original_label).select(range(10))
    raw_datasets = raw_datasets['test'].shuffle(seed=42).select(range(100))
    print("num_samples: " + str(len(raw_datasets)))

    def preprocess_function_unmask(examples):
        # Tokenize the texts
        texts = (
            (examples[sentence1_key],) if sentence2_key is None else (examples[sentence1_key], examples[sentence2_key])
        )
        result = tokenizer(*texts, padding=padding, max_length=args.max_length, truncation=True)

        if "label" in examples:
            if label_to_id is not None and args.task_name not in glue_task:
                # Map labels to IDs (not necessary for GLUE tasks)
                result["labels"] = [label_to_id[l] for l in examples["label"]]
            else:
                # In all cases, rename the column to labels because the model will expect that.
                result["labels"] = examples["label"]
        return result
    
    def preprocess_function_mask(examples):
        # Tokenize the texts
        if sentence2_key is None:
            tokenized_sentence1 = tokenizer(examples[sentence1_key], padding=padding, max_length=args.max_length, truncation=True)
            # Mask
            sentence = tokenized_sentence1["input_ids"]
            tokenized_sentence1["input_ids"] = [sentence[0]] + (len(sentence) - 2) * [tokenizer.convert_tokens_to_ids(tokenizer.mask_token)] + sentence[-1:]
            result = tokenized_sentence1
        else:
            tokenized_sentence1 = tokenizer(examples[sentence1_key], padding=padding, max_length=args.max_length, truncation=True)
            tokenized_sentence2 = tokenizer(examples[sentence2_key], padding=padding, max_length=args.max_length, truncation=True)
            # Mask
            sentence = tokenized_sentence2["input_ids"]
            # tokenized_sentence2["input_ids"] = [sentence[-1]] + sentence[1:3] + (len(sentence) - 4) * [tokenizer.convert_tokens_to_ids(tokenizer.mask_token)] + sentence[-1:]
            tokenized_sentence2["input_ids"] = [sentence[-1]] + (len(sentence) - 2) * [tokenizer.convert_tokens_to_ids(tokenizer.mask_token)] + sentence[-1:]
            result = {
                "input_ids": tokenized_sentence1["input_ids"] + tokenized_sentence2["input_ids"],
                "attention_mask": tokenized_sentence1["attention_mask"] + tokenized_sentence2["attention_mask"],
            }

        if "label" in examples:
            if label_to_id is not None and args.task_name not in glue_task:
                # Map labels to IDs (not necessary for GLUE tasks)
                result["labels"] = label_to_id[examples["label"]]
            else:
                # In all cases, rename the column to labels because the model will expect that.
                result["labels"] = examples["label"]
            
        return result

    with accelerator.main_process_first():
        processed_datasets_unmasked = raw_datasets.map(
            preprocess_function_unmask,
            batched=True,
            remove_columns=raw_datasets.column_names,
            desc="Running tokenizer on dataset",
        )
        processed_datasets_masked = raw_datasets.map(
            preprocess_function_mask,
            batched=False,
            remove_columns=raw_datasets.column_names,
            desc="Running tokenizer on dataset",
        )

    eval_dataset = processed_datasets_masked
    eval_dataset_unmasked = processed_datasets_unmasked

    # Log a few random samples from the training set:
    for index in random.sample(range(len(eval_dataset)), 1):
        unmasked_sample = tokenizer.decode(eval_dataset_unmasked[index]["input_ids"])
        masked_sample = tokenizer.decode(eval_dataset[index]["input_ids"])
        logger.info(f"Sample {index} of the unmasked test set: {unmasked_sample}.")
        logger.info(f"Sample {index} of the masked test set: {masked_sample}.")

    # DataLoaders creation:
    if args.pad_to_max_length:
        # If padding was already done ot max length, we use the default data collator that will just convert everything
        # to tensors.
        data_collator = default_data_collator
    else:
        # Otherwise, `DataCollatorWithPadding` will apply dynamic padding for us (by padding to the maximum length of
        # the samples passed). When using mixed precision, we add `pad_to_multiple_of=8` to pad all tensors to multiple
        # of 8s, which will enable the use of Tensor Cores on NVIDIA hardware with compute capability >= 7.5 (Volta).
        data_collator = DataCollatorWithPadding(tokenizer, pad_to_multiple_of=(8 if accelerator.use_fp16 else None))

    eval_dataloader = DataLoader(eval_dataset, collate_fn=data_collator, batch_size=args.per_device_eval_batch_size)
    eval_dataloader_origin = DataLoader(eval_dataset_unmasked, collate_fn=data_collator, batch_size=args.per_device_eval_batch_size)
    mask_token_id = tokenizer.convert_tokens_to_ids(tokenizer.mask_token)

    
    loss_fct = CrossEntropyLoss()

    check_dir(args.output_dir)
    if args.rejection:
        if args.task_name in ["snli", "mnli"]:
            label_list = [0, 1, 2]
        elif args.task_name in ['qqp', 'TwitterPPDB']:
            label_list = [0, 1]
        for label in label_list:
            progress_bar = tqdm(range(len(eval_dataloader)), disable=not accelerator.is_local_main_process)
            output_path = os.path.join(args.output_dir, f"res_iter={args.num_iters}_label={label}.txt")
            with open(output_path, "w") as f:
                T = args.num_iters
                lower_0 = 0.1
                lower_T = 0.3
                lower_list = np.linspace(lower_0, lower_T, T)
                model.train()
                for (batch, batch_original) in tqdm(zip(eval_dataloader, eval_dataloader_origin)):
                    mask_pos = (batch["input_ids"] == mask_token_id) # Indicate the total masked area.
                    N = int(torch.sum(mask_pos.int())) # Total masked tokens (batched)
                    labels_original = batch.pop("labels", None)
                    labels = labels_original * 0 + label
                    for t in range(T):
                        iter_conter = 0
                        lower = lower_list[t]
                        while True:
                            u = lower + (1. - lower) * torch.rand(1)
                            
                            # Direct
                            model.train()
                            _, logits_px, z = model(batch) 
                            
                            # PPLM
                            # logits_cls_masked, z = model.variational(batch)
                            # loss_cls_masked = loss_fct(logits_cls_masked.view(-1, num_labels), labels)
                            # for _ in range(3):
                            #     dz = torch.autograd.grad(loss_cls_masked, z)
                            #     dz = F.normalize(torch.concat(dz), p=2, dim=-1)
                            #     z += dz
                            #     hs = torch.mean(z, dim=1, keepdim=True)
                            #     logits_cls_masked = model.variational.classifier(hs)
                            #     loss_cls_masked = loss_fct(logits_cls_masked.view(-1, num_labels), labels)
                            # logits_px = model.generator(z)

                            mlm_prob = F.softmax(logits_px, -1)[mask_pos, :] # mlm probability for masked pos.
                            max_prob, max_ids = torch.max(mlm_prob, dim=-1)
                        
                            n = round(N * (T-(t+1))/T) # Linear decay, decide numer of tokens to mask.
                            _, preserve = torch.topk(max_prob, N - n)
                            tokens = torch.zeros(N).long() + mask_token_id
                            tokens[preserve] = max_ids[preserve]
                            
                            batch_unchanged = batch["input_ids"].clone()
                            batch["input_ids"] = torch.masked_scatter(batch["input_ids"], mask_pos, tokens)

                            model.eval()
                            with torch.no_grad():
                                logits_cls_masked, _, _ = model(batch)
                            cls_prob = F.softmax(logits_cls_masked, -1)

                            if float(cls_prob[0,label]) > float(u):
                                break
                            batch["input_ids"] = batch_unchanged
                            iter_conter += 1
                            if iter_conter % 50 == 0:
                                lower *= 0.9            # If sampling for this particular label is too hard, we decrease the threshold.
                        
                    # Save batch results
                    for (sample, original) in zip(batch["input_ids"], batch_original["input_ids"]):
                        generated_label = torch.max(cls_prob, -1)[1]
                        f.write(f"Original : " + tokenizer.decode(original) + f' label: {int(labels_original)}' +  '\n')
                        f.write(f"Generated: " + tokenizer.decode(sample) + f' label: {int(generated_label)}' "\n")
                        if args.task_name in ["snli", "mnli"]:
                            f.write(f"entailment: {cls_prob[0,0]:.2f}, contradiction: {cls_prob[0,1]:.2f}, neutral: {cls_prob[0,2]:.2f}" + '\n\n')
                        elif args.task_name in ['qqp', 'TwitterPPDB']:
                            f.write(f"non-synonym: {cls_prob[0,0]:.2f}, synonym: {cls_prob[0,1]:.2f}" + '\n\n')
                    progress_bar.update(1)

    else:
        model.eval()
        progress_bar = tqdm(range(len(eval_dataloader)), disable=not accelerator.is_local_main_process)
        output_path = os.path.join(args.output_dir, f"res_iter={args.num_iters}.txt")
        with open(output_path, "w") as f:
            T = args.num_iters
            for (batch, batch_original) in tqdm(zip(eval_dataloader, eval_dataloader_origin)):
                mask_pos = (batch["input_ids"] == mask_token_id) # Indicate the total masked area.
                N = int(torch.sum(mask_pos.int())) # Total masked tokens (batched)
                labels = batch.pop("labels", None)
                for t in range(T):
                    # Predict
                    with torch.no_grad():
                        if args.ckpt_path is None:
                            outputs = model(**batch)
                            logits_px = outputs.logits
                        else:
                            logits_cls_masked, logits_px, z = model(batch) # MLM + Classification
                    
                    mlm_prob = F.softmax(logits_px, -1)[mask_pos, :] # mlm probability for masked pos.
                    max_prob, max_ids = torch.max(mlm_prob, dim=-1)
                    
                    # Mask
                    n = round(N * (T-(t+1))/T) # Linear decay, decide numer of tokens to mask.
                    _, preserve = torch.topk(max_prob, N - n)
                    tokens = torch.zeros(N).long() + mask_token_id
                    tokens[preserve] = max_ids[preserve]
                    batch["input_ids"] = torch.masked_scatter(batch["input_ids"], mask_pos, tokens)

                
                # Save batch results
                for (sample, original) in zip(batch["input_ids"], batch_original["input_ids"]):
                    
                    f.write(f"Original : " + tokenizer.decode(original) + f' label: {int(labels)}' +  '\n')
                    
                    if args.ckpt_path is not None:
                        cls_prob = F.softmax(logits_cls_masked, -1)
                        generated_label = torch.max(cls_prob, -1)[1]
                        f.write(f"Generated: " + tokenizer.decode(sample) + f' label: {int(generated_label)}' "\n")
                        if args.task_name in ["snli", "mnli"]:
                            f.write(f"entailment: {cls_prob[0,0]:.2f}, contradiction: {cls_prob[0,1]:.2f}, neutral: {cls_prob[0,2]:.2f}" + '\n\n')
                        elif args.task_name in ['qqp', 'TwitterPPDB']:
                            f.write(f"non-synonym: {cls_prob[0,0]:.2f}, synonym: {cls_prob[0,1]:.2f}" + '\n\n')
                    else:
                        f.write(f"Generated: " + tokenizer.decode(sample) + "\n")
                        f.write('\n\n')
                progress_bar.update(1)

            
    

if __name__ == "__main__":
    main()