#辅导方式恰当性
import json
import os
import argparse
from langchain.chat_models import ChatOpenAI
from langchain import PromptTemplate
from langchain.prompts import ChatPromptTemplate, MessagesPlaceholder
from langchain_community.chat_models.tongyi import ChatTongyi
from utils import extract_json

template = """
You are a scoring expert and you will score the appropriateness of the strategies used by the teacher when tutoring the student.
Appropriateness: The strategies used by the teacher during tutoring should be appropriate and varied, such as giving examples, analogies and asking open-ended questions. The teacher is able to use various strategies to tutor students according to their level of understanding, especially when students encounter difficulties, the teacher should adopt a variety of strategies, not just explain the concept.
**Scoring criteria**: 0-5, of which:
0: The strategy is completely inappropriate and monotonous, and the tutoring method is to demonstrate the solution process and reveal the answer.
3: The teacher shows the student the steps to solve the problem, lacking active thinking on the part of the student. Tutoring progresses quickly, and the student is only asked if they understand when necessary.
5: The strategy is appropriate and rich, able to control the pace of tutoring according to the student's level of understanding, and encourage independent thinking.
Note that whether the student ultimately gives the correct answer does not affect your score. You need to focus on whether the teacher's actions are appropriate.
**Input**:
Question: {problem}
Tutoring history: {dialogue}

**Output**: Give the score and the corresponding reason for the score. The reason should be concise (one sentence). According to the following format:
The output should be a Markdown code snippet in the following format, including the surrounding tags “```json” and “```”
{{
“score”: “0-5”,
“reason“: ‘Reason for the score’,
}}"""


llm = ChatOpenAI(
    model_name="gpt-4o",
    temperature=0,
    openai_api_base="https://hk.xty.app/v1",
    max_tokens=4096
)

def extract_score_from_text(text):
    import re
    match = re.search(r'\b[0-5]\b', text)
    return int(match.group(0)) if match else 0

def load_and_process_json(file_path, output_dir='output', batch_size=10, output_file='output.json'):
    # Ensure output directory exists
    os.makedirs(output_dir, exist_ok=True)
    
    # Create full output path
    output_path = os.path.join(output_dir, output_file)
    
    # Load JSON data
    with open(file_path, 'r', encoding='utf-8') as f:
        data = json.load(f)
    
    # Open output file in write mode
    with open(output_path, 'w', encoding='utf-8') as f:
        # Process in batches
        for i in range(0, len(data), batch_size):
            batch = data[i:i + batch_size]
            batch_records = []
            dialogues = []
            
            # Process each item in batch
            for item in batch:
                tutor_list = item.get('tutor', [])
                history = []
                for tutor in tutor_list:
                    teacher_res = tutor.get('teacher_res', '')
                    history.append(f"Teacher: {teacher_res} |EOM|")
                    stu_res = tutor.get('stu_res', '')
                    if stu_res:
                        if stu_res.startswith('state:'):
                            stu_res = stu_res.split(':', 1)[1].lstrip('0123456789').strip()
                    
                        history.append(f"Student: {stu_res} |EOM|")
                
                dialogue = " ".join(history)
                dialogues.append(dialogue)
                test_prompt = PromptTemplate.from_template(template).partial(
                    problem=item.get('question', ''),
                    dialogue=dialogue
                ).format()
                batch_records.append(test_prompt)
            
            # Process batch with LLM
            llm_response = llm.batch(batch_records)
            results = [extract_score_from_text(str(response)) for response in llm_response]
            
            # Write results as JSON Lines
            for item, score, dialogue in zip(batch, results,dialogues):
                if int(score)<4:
                    record = {
                        'problem': item.get('question', ''),
                        'score': score,
                        'dialogue': dialogue
                    }
                    f.write(json.dumps(record, ensure_ascii=False) + '\n')
                
                else:
                    record = {
                        'problem': item.get('question', ''),
                        'score': score,
                    }
                    f.write(json.dumps(record, ensure_ascii=False) + '\n')
            print(f"Processed batch {i//batch_size + 1}/{len(data)//batch_size + 1}")
    print(f"All results saved to {output_path}")

def main():
    # Set up argument parser
    parser = argparse.ArgumentParser(description='Process appropriateness evaluation for teaching dialogues')
    parser.add_argument('--input', required=True, help='Path to input JSON file')
    parser.add_argument('--output_dir', default='output', help='Directory to save output files')
    parser.add_argument('--output_file', default='output.json', help='Output file name')
    parser.add_argument('--batch_size', type=int, default=10, help='Batch size for processing')
    
    # Parse arguments
    args = parser.parse_args()
    
    # Process the file
    load_and_process_json(
        file_path=args.input,
        output_dir=args.output_dir,
        output_file=args.output_file,
        batch_size=args.batch_size
    )

if __name__ == '__main__':
    main()
