#总体性能
import json
import os
import argparse
from langchain.chat_models import ChatOpenAI
from langchain import PromptTemplate
from langchain.prompts import ChatPromptTemplate, MessagesPlaceholder
from langchain_community.chat_models.tongyi import ChatTongyi
from utils import extract_json


template = """
You are a scoring expert. The teacher is tutoring the student through dialogue. Please evaluate the overall ability of the teacher when tutoring the student.
Overall ability: The comprehensive ability demonstrated by the teacher during the tutoring process, including encouraging the student to think actively (active thinking means getting the student to think about the problem actively through questioning, etc., rather than the teacher showing the solution and then asking the student if they understand), a reasonable grasp of the pace of problem-solving, appropriate tutoring strategies, accuracy in imparting knowledge, and providing correct feedback when the student gives the wrong answer.
**Grading criteria**: 0-5, where
0: The teacher directly analyzes and demonstrates the solution process to the students, who are in a passive state of acceptance with no room to think. The teacher provides incorrect feedback to the students, which causes them to misunderstand.
5: The teacher demonstrates strong comprehensive skills, asks targeted questions to guide the students' thinking, encourages the students to solve problems on their own, and gives correct feedback to the students' responses.
Note that whether the student ultimately gives the correct answer and whether they successfully understand a concept does not affect your grading. You need to focus on the teacher's behavior.
**Input**:
**Question**: {problem}
**Reference answer (used to determine whether the teacher's explanation is correct)**: {answer}
**Tutoring history**: {dialogue}

**Output**: Give the score and the corresponding reason for the score. The reason should be concise (1 sentence). According to the following format:
The output should be a Markdown code snippet in the following format, including the surrounding tags “```json” and “```”
{{
“score”: “0-5”,
“reason“: ‘Reason for the score’,
}}"""



# gpt-3.5-turbo-1106
llm = ChatOpenAI(
    model_name="gpt-4o",
    temperature=0,
    openai_api_base="https://api.xty.app/v1",
    max_tokens=4096
)

def extract_score_from_text(text):
    import re
    match = re.search(r'\b[0-5]\b', text)
    return int(match.group(0)) if match else 0

def load_and_process_json(file_path, output_dir='output', batch_size=10, output_file='output.json'):
    # Ensure output directory exists
    os.makedirs(output_dir, exist_ok=True)
    
    # Create full output path
    output_path = os.path.join(output_dir, output_file)
    
    # Load JSON data
    with open(file_path, 'r', encoding='utf-8') as f:
        data = json.load(f)
    
    # Open output file in write mode
    with open(output_path, 'w', encoding='utf-8') as f:
        # Process in batches
        for i in range(0, len(data), batch_size):
            batch = data[i:i + batch_size]
            batch_records = []
            dialogues = []
            
            # Process each item in batch
            for item in batch:
                tutor_list = item.get('tutor', [])
                history = []
                for tutor in tutor_list:
                    teacher_res = tutor.get('teacher_res', '')
                    history.append(f"Teacher: {teacher_res} |EOM|")
                    stu_res = tutor.get('stu_res', '')
                    if stu_res:
                        if stu_res.startswith('state:'):
                            stu_res = stu_res.split(':', 1)[1].lstrip('0123456789').strip()
                    
                        history.append(f"Student: {stu_res} |EOM|")
                
                dialogue = " ".join(history)
                dialogues.append(dialogue)
                test_prompt = PromptTemplate.from_template(template).partial(
                    problem=item.get('question', ''),
                    answer=item.get('analysis', ''),
                    dialogue=dialogue
                ).format()
                batch_records.append(test_prompt)
            
            # Process batch with LLM
            llm_response = llm.batch(batch_records)
            results = [extract_score_from_text(str(response)) for response in llm_response]
            ress = extract_json(llm_response)
            # Write results as JSON Lines
            for item, score, llm_r, dialogue in zip(batch, results, ress, dialogues):
                if int(score)<4:
                    
                    record = {
                        'problem': item.get('question', ''),
                        'score': score,
                        'text': llm_r[0]['score'],
                        "resson" : llm_r[0]['reason'],
                        'dialogue': dialogue
                    }
                    f.write(json.dumps(record, ensure_ascii=False) + '\n')
                else:
                    record = {
                        'problem': item.get('question', ''),
                        'score': score,
                        'text': llm_r[0]['score'],
                        "resson" : llm_r[0]['reason'],
                    }
                    f.write(json.dumps(record, ensure_ascii=False) + '\n')

            print(f"Processed batch {i//batch_size + 1}/{len(data)//batch_size + 1}")
    
    print(f"All results saved to {output_path}")

def main():
    # Set up argument parser
    parser = argparse.ArgumentParser(description='Process overall proficiency evaluation for teaching dialogues')
    parser.add_argument('--input', required=True, help='Path to input JSON file')
    parser.add_argument('--output_dir', default='output', help='Directory to save output files')
    parser.add_argument('--output_file', default='output.json', help='Output file name')
    parser.add_argument('--batch_size', type=int, default=10, help='Batch size for processing')
    
    # Parse arguments
    args = parser.parse_args()
    
    # Process the file
    load_and_process_json(
        file_path=args.input,
        output_dir=args.output_dir,
        output_file=args.output_file,
        batch_size=args.batch_size
    )

if __name__ == '__main__':
    main()
