import json
import sys
import argparse
from pathlib import Path
from collections import defaultdict

def calculate_scores(output_dir):
    # 创建一个字典来存储每个文件的得分以及每个分组的总得分和数量
    group_scores = defaultdict(lambda: {'total_score': 0, 'count': 0})
    file_scores = {}

    # 遍历目录下的所有json文件
    for json_file in Path(output_dir).glob('*.json'):
        # 只处理文件名包含"gpt_res"的文件
        if "gpt_res" not in json_file.name.lower():
            continue
        
        # 提取文件名中的数字部分（例如 '6' 从 'gpt_res6_chemistry.json'）
        group_identifier = json_file.name.split('_')[1]  # 提取数字部分
        
        total_score = 0
        count = 0
        
        # 读取并处理每个json文件
        with open(json_file, 'r', encoding='utf-8') as f:
            for line in f:
                data = json.loads(line)
                score = int(data.get('score', 0))
                total_score += score
                count += 1
        
        # 保存该文件的得分
        if count > 0:
            file_scores[json_file.name] = total_score / count
            # 将该文件的得分加到对应分组的总得分中
            group_scores[group_identifier]['total_score'] += total_score
            group_scores[group_identifier]['count'] += count
        else:
            file_scores[json_file.name] = None  # 没有有效数据的文件
    
    # 输出每个文件的得分
    for file_name, score in file_scores.items():
        if score is not None:
            print(f"文件: {file_name} 的得分: {score:.2f}")
        else:
            print(f"文件: {file_name} 没有有效数据")
    
    # 输出每个分组的平均得分
    for group, stats in group_scores.items():
        if stats['count'] > 0:
            average = stats['total_score'] / stats['count']
            print(f"组 {group} 的平均得分: {average:.2f}")
        else:
            print(f"组 {group} 没有有效数据")

if __name__ == "__main__":
    # 设置命令行参数解析
    parser = argparse.ArgumentParser(description='处理包含gpt的文件夹中的json文件')
    parser.add_argument('--output_dir', type=str, required=True, help='包含json文件的目录路径')
    
    args = parser.parse_args()
    calculate_scores(args.output_dir)
