import torch
from torch import nn


def neural_substitution(x: torch.Tensor, convolutions: nn.ModuleList) -> torch.Tensor:
    """
    Stochastic neural substitution for branch-level connectivity

    Args:
        x (torch.Tensor): Input features of multiple convolutions.
        The shape is (N,C,H,W), representing number of features(not batch size), channels, height, and width, respectively.
        convolutions (nn.ModuleList): The list of multiple convolutions.
    Returns:
        'torch.Tensor': The substituted features that have passed multiple convolutions.
    """

    N, C, H, W = x.size()
    n_conv = len(convolutions)
    out_features = list()

    for conv_module in convolutions:
        out_features.append(conv_module(x))

    out_features = torch.cat(out_features, dim=0)
    out_features = out_features[torch.randperm(n_conv * N)]

    out_features = out_features.reshape(n_conv, N, C, H, W).sum(1)
    return out_features
