#!/bin/bash

# Check if seed and CUDA device index arguments are provided
if [ $# -lt 2 ]; then
    echo "Usage: $0 <seed> <cuda_index>"
    exit 1
fi

SEED=$1
CUDA_INDEX=$2

# Map CUDA index to device string, taskset core range, and tmux starting index
case $CUDA_INDEX in
    0)
        DEVICE="cuda:0"
        CORE_START="0"
        TMUX_START="0"
        ;;
    1)
        DEVICE="cuda:1"
        CORE_START="16"
        TMUX_START="6"
        ;;
    *)
        echo "Invalid CUDA index. Use 0 for cuda:0 or 1 for cuda:1."
        exit 1
        ;;
esac

# Function to generate the core range for taskset
generate_core_range() {
    local start=$1
    local end=$((start + 2))
    echo "$start-$end"
}

source activate tcmdp
# If not done download the agents from wandb and extract them
python script/extract_wandb_exp.py --entity="anonymous" --project="paper_vanilla_tc_rarl" --download_folder="Agents_wandb_logs/vanilla_tc_rarl" --handle_missing_seed=True
python script/gather_agents.py --all_logs_folder "Agents_wandb_logs" --output_folder "paper_agents"

# Function to run the command multiple times
run_command() {
    source activate tcmdp

    # Loop over each file that matches the pattern in the specified directory
    agent_file=paper_agents/vanilla_tc_rarl/$2/3/agent_$3_$2_3.pth
    taskset -c $1 python src/main_tc_adversary.py --agent_path $agent_file --agent_type td3 --env_name $2 --project_name paper_adversaryVanilla_tc_rarl --device $4 --radius 0.001 --max_steps 5000000 --nb_uncertainty_dim 3 --experiment_name tc_adv_$2_$3 --seed $3 --omniscient_adversary True --device $4
}

# Define core ranges for each taskset
CORE_RANGE_1=$(generate_core_range $CORE_START)
CORE_RANGE_2=$(generate_core_range $((CORE_START + 3)))
CORE_RANGE_3=$(generate_core_range $((CORE_START + 6)))
CORE_RANGE_4=$(generate_core_range $((CORE_START + 9)))
CORE_RANGE_5=$(generate_core_range $((CORE_START + 12)))

# tmux 1
tmux new-session -d -s tmux_$((TMUX_START + 0)) "$(typeset -f); run_command '$CORE_RANGE_1' 'Ant' $SEED $DEVICE"

# tmux 2
tmux new-session -d -s tmux_$((TMUX_START + 1)) "$(typeset -f); run_command '$CORE_RANGE_2' 'HalfCheetah' $SEED $DEVICE"

# tmux 3
tmux new-session -d -s tmux_$((TMUX_START + 2)) "$(typeset -f); run_command '$CORE_RANGE_3' 'Hopper' $SEED $DEVICE"

# tmux 4
tmux new-session -d -s tmux_$((TMUX_START + 3)) "$(typeset -f); run_command '$CORE_RANGE_4' 'HumanoidStandup' $SEED $DEVICE"

# tmux 5
tmux new-session -d -s tmux_$((TMUX_START + 4)) "$(typeset -f); run_command '$CORE_RANGE_5' 'Walker' $SEED $DEVICE"

echo "All tmux sessions started. Use 'tmux attach-session -t [session_name]' to attach to a session."
