import time
from copy import deepcopy
import argparse
import numpy as np
import torch
import torch.optim as optim
import torch.nn.functional as F
import hydra
import logging
import datetime

from data.load_dataset import load_data

from dhg.random import set_seed
from model.hds_ode import hds_ode
from dhg.metrics import HypergraphVertexClassificationEvaluator as Evaluator


parser = argparse.ArgumentParser()
parser.add_argument('--dataset', type=str, default="Cora-CA",
                    help='Dataset to use.')
parser.add_argument('--model_name', type=str, default="hds_ode",
                    help='model to use.')
parser.add_argument('--lr', type=float, default=1e-2,
                    help='Initial learning rate.')
parser.add_argument('--weight_decay', type=float, default=5e-4,
                    help='Weight decay (L2 loss on parameters).')
parser.add_argument('--repeat', type=int, default=5,
                    help='Number of repeat times to test.')
parser.add_argument('--epochs', type=int, default=200,
                    help='Number of epochs to train.')
parser.add_argument('--layer_num', type=int, default=40,
                    help='Number of layers.')
parser.add_argument('--step', type=int, default=20,
                    help='Steps of diffusion.')
parser.add_argument('--alpha_v', type=float, default=0.05,
                    help='alpha of vertices in diffusion.')
parser.add_argument('--alpha_e', type=float, default=0.9,
                    help='alpha of hyperedge in diffusion.')
parser.add_argument('--dropout', type=float, default=0.15,
                    help='Dropout rate')
parser.add_argument('--split', type=str, default='random',
                    help='type of split')
parser.add_argument('--num_per_class', type=int, default=10,
                    help='Number of training vertex per class.')
parser.add_argument('--num_development', type=int, default=1500,
                    help='Number of training+val vertices.')
parser.add_argument('--no-cuda', action='store_true', default=False,
                    help='Disables CUDA training.')
parser.add_argument('--seed', type=int, default=2022, help='Random seed.')

args = parser.parse_args()
args.cuda = not args.no_cuda and torch.cuda.is_available()

log = logging.getLogger(__name__)


def load_model(model_name, dim_features, num_classes):
    if model_name == 'hds_ode':
        return hds_ode(dim_features, num_classes,
                    args.layer_num, args.step, alpha_v=args.alpha_v, alpha_e=args.alpha_e, drop_rate=args.dropout)
    else:
        print("model doesn't exist")


def train(net, X, A, lbls, train_idx, optimizer, epoch):
    net.train()

    st = time.time()
    optimizer.zero_grad()
    outs = net(X, A)
    outs, lbls = outs[train_idx], lbls[train_idx]
    loss = F.cross_entropy(outs, lbls)
    loss.backward()
    optimizer.step()
    print(f"Epoch: {epoch}, Time: {time.time()-st:.5f}s, Loss: {loss.item():.5f}")
    return loss.item()


@torch.no_grad()
def infer(net, X, A, lbls, idx, evaluator, test=False):
    net.eval()

    outs = net(X, A)
    outs, lbls = outs[idx], lbls[idx]
    if not test:
        res = evaluator.validate(lbls, outs)
    else:
        res = evaluator.test(lbls, outs)
    return res


@hydra.main(version_base=None)
def main(cfg):
    log.info(args)
    start_time = datetime.datetime.strftime(datetime.datetime.now(), '%Y-%m-%d_%H:%M:%S')
    set_seed(args.seed)
    device = torch.device("cuda") if args.cuda else torch.device("cpu")
    evaluator = Evaluator(["accuracy", "f1_score", {"f1_score": {"average": "micro"}}])

    res_list = []
    for i in range(args.repeat):
        X, HG, lbl, dim_features, num_classes, train_mask, val_mask, test_mask = load_data(args.dataset, args.split,
                                                                                           args.num_per_class,
                                                                                           args.num_development)
        model = load_model(args.model_name, dim_features, num_classes)
        optimizer = optim.Adam(model.parameters(), lr=args.lr, weight_decay=args.weight_decay)

        X, lbl = X.to(device), lbl.to(device)
        HG = HG.to(device)
        model = model.to(device)

        best_state = None
        best_epoch, best_val = 0, 0
        for epoch in range(args.epochs):
            # train
            train(model, X, HG, lbl, train_mask, optimizer, epoch)
            # validation
            if epoch % 1 == 0:
                with torch.no_grad():
                    val_res = infer(model, X, HG, lbl, val_mask, evaluator)
                if val_res > best_val:
                    log.info(f"update best: {val_res:.5f}")
                    best_epoch = epoch
                    best_val = val_res
                    best_state = deepcopy(model.state_dict())
        log.info("\ntrain finished!")
        log.info(f"best val: {best_val:.5f}")
        # test
        log.info("test...")
        model.load_state_dict(best_state)
        res = infer(model, X, HG, lbl, test_mask, evaluator, test=True)
        log.info(f"final result: epoch: {best_epoch}")
        log.info(res)
        log.info('#' * 30)
        res_list.append(res['accuracy'])

    log.info(f"mean: {np.mean(np.array(res_list))}, std: {np.std(np.array(res_list))}")
    log.info(f"{(np.array(res_list) - np.mean(np.array(res_list))).tolist()}")


if __name__ == '__main__':
    main()
