import numpy as np

class Traders:

    def __init__(self, supp_env='uniform', retail_env='cap_linear', params=()):
        # Initialize the supplier and retailer environments
        self.supp_env = supp_env
        self.retail_env = retail_env

        # params is used for environments that require parameters
        self.params = params

        # count is used to create a deterministic adversary sequence (for lifting Assumption 1.1)
        self.count = 0

    def feedback(self, p: float):
        # Verify that the price is within the correct bounds
        if not (0 <= p <= 1):
            raise ValueError("Price must be in [0,1]")

        # Handle the case of the symmetric tri-rectangle distribution first (for lifting Assumptions 1.1 and 1.2)
        if self.supp_env in {"mu", "F_mu_adversary"} and self.retail_env == self.supp_env:
            if self.supp_env == "mu":
                # Select one of the three rectangles uniformly at random
                rect = np.random.randint(1, 4)
            else:
                # Select the rectangle deterministically according to the sequence i_t = {3,2,1,3,2,1,...}
                rect = 3 - (self.count % 3)
                self.count += 1

            # Draw supplier cost and retailer utility function from the selected rectangle
            if rect == 1:
                supplier_cost = np.random.uniform(0.0, 1 / 8)
                x = np.random.uniform(3 / 8, 4 / 8)
            elif rect == 2:
                supplier_cost = np.random.uniform(2 / 8, 3 / 8)
                x = np.random.uniform(7 / 8, 1.0)
            elif rect == 3:
                supplier_cost = np.random.uniform(4 / 8, 5 / 8)
                x = np.random.uniform(5 / 8, 6 / 8)

            U = lambda q: x if q == 1 else 0
            U_sharp = lambda price: 1.0 if x >= price else 0.0

        else:

            # Draw supplier costs from other types of distributions
            env = self.supp_env
            if env == "uniform":
                # Standard uniform distribution
                supplier_cost = np.random.uniform()

            elif env == "beta":
                # Beta distribution
                a, b = self.params
                supplier_cost = np.random.beta(a, b)

            elif env == "trunc_log_normal":
                # Truncated log-normal distribution
                # mu and sigma are the mean and st. dev. of log(X), respectively
                mu, sigma = self.params

                # Reject samples outside of [0,1]
                check = True
                while check:
                    v = np.random.lognormal(mu, sigma)
                    if 0.0 <= v <= 1.0:
                        supplier_cost = v
                        check=False

            elif env == "two_beta_mixture":
                # Mixture of two Beta distributions, with weights w and 1-w
                w, a1, b1, a2, b2 = self.params
                check = True
                while check:
                    v = np.random.beta(a1, b1) if np.random.rand() < w else np.random.beta(a2, b2)
                    if 0.0 <= v <= 1.0:
                        supplier_cost = v
                        check=False

            elif env == "two_point_cost":
                # Discrete distribution with support {0, x} (for lifting Assumption 1.3)
                x = self.params
                supplier_cost = np.random.choice([0, x])

            else:
                raise ValueError(f"Unknown supplier environment {env!r}")

            # Draw retailer utilities from other types of distributions
            r_env = self.retail_env

            if r_env == "cap_linear":
                a = np.random.beta(5, 2)
                q_cap = np.random.beta(2, 2)

                U = lambda q: min(a * q, a * q_cap)
                U_sharp = lambda price: q_cap if a >= price else 0.0

            elif r_env == "exp":
                Lambda = np.random.lognormal(0, 0.5)

                U = lambda q: (1 - np.exp(-Lambda * q)) / Lambda
                U_sharp = lambda price: 1.0 if price == 0.0 else min(-np.log(price) / Lambda, 1.0)

            elif r_env == "two_type_buyer":
                # Retailer utility used for lifting Assumption 1.3
                x_prime = self.params

                x = x_prime if np.random.rand() < 0.5 else 1.0

                U = lambda q: x if q == 1 else 0
                U_sharp = lambda price: 1.0 if x >= price else 0.0

            elif r_env == "quadratic":
                # Deterministic retailer utility used for lifting Assumption 1.4
                U = lambda q: -(q - 0.7) ** 2 + 0.5
                U_sharp = lambda price: 0.7 - 0.5 * price

            else:
                raise ValueError(f"Unknown retailer environment {r_env!r}")

        supplier_accept = int(p >= supplier_cost)
        q_star = U_sharp(p)
        U_star = U(q_star) - p * q_star

        return supplier_cost, supplier_accept, U_star, q_star
