from typing import Optional, Tuple, Union

import torch
import torch.nn as nn
import torch.nn.functional as F
from torch import Tensor
from torch_sparse import SparseTensor, matmul, set_diag

from torch_geometric.nn.conv import MessagePassing
from torch_geometric.nn.conv.gcn_conv import gcn_norm
from torch_geometric.typing import Adj, OptTensor, OptPairTensor, Size
from torch_geometric.nn.dense.linear import Linear

from math import log
from torch.nn import Parameter
from torch_geometric.nn.inits import glorot, zeros
from torch_geometric.utils import add_self_loops, remove_self_loops, softmax

def cal_coeff(x1, x2):
    out1 = x1.mean(dim=1, keepdim=True)
    out2 = x2.mean(dim=1, keepdim=True)
    out = torch.cat([out1, out2], dim=1)
    alpha = F.softmax(out, dim=1)
    return torch.narrow(alpha, 1, 0, 1), torch.narrow(alpha, 1, 1, 1)

class KAggre(MessagePassing):

    _cached_edge_index: Optional[Tuple[Tensor, Tensor]]
    _cached_adj_t: Optional[SparseTensor]

    def __init__(self, K: int = 1, cached: bool = False, add_self_loops: bool = True, **kwargs):
        kwargs.setdefault('aggr', 'add')
        super().__init__(**kwargs)

        self.K = K
        self.cached = cached
        self.add_self_loops = add_self_loops
        self._cached_edge_index = None
        self._cached_adj_t = None
        self.reset_parameters()

    def reset_parameters(self):
        self._cached_edge_index = None
        self._cached_adj_t = None

    def forward(self, x: Tensor, edge_index: Adj,
                edge_weight: OptTensor = None) -> Tensor:
        """"""
        if isinstance(edge_index, Tensor):
            cache = self._cached_edge_index
            if cache is None:
                edge_index, edge_weight = gcn_norm(  # yapf: disable
                    edge_index, edge_weight, x.size(self.node_dim), False,
                    self.add_self_loops, dtype=x.dtype)
                if self.cached:
                    self._cached_edge_index = (edge_index, edge_weight)
            else:
                edge_index, edge_weight = cache[0], cache[1]
        elif isinstance(edge_index, SparseTensor):
            cache = self._cached_adj_t
            if cache is None:
                edge_index = gcn_norm(  # yapf: disable
                    edge_index, edge_weight, x.size(self.node_dim), False,
                    self.add_self_loops, dtype=x.dtype)
                if self.cached:
                    self._cached_adj_t = edge_index
            else:
                edge_index = cache

        for _ in range(self.K):
            # propagate_type: (x: Tensor, edge_weight: OptTensor)
            x = self.propagate(edge_index, x=x, edge_weight=edge_weight,
                                size=None)

        return x

    def message(self, x_j: Tensor, edge_weight: Tensor) -> Tensor:
        return edge_weight.view(-1, 1) * x_j

    def message_and_aggregate(self, adj_t: SparseTensor, x: Tensor) -> Tensor:
        return matmul(adj_t, x, reduce=self.aggr)

    def __repr__(self) -> str:
        return (f'{self.__class__.__name__}(K={self.K})')

class newConv(nn.Module):
    def __init__(self, in_channels: int, out_channels: int, K: int = 1, cached: bool = False, add_self_loops: bool = True, bias: bool = True):
        super().__init__()
        self.Aggre_int = KAggre(K, cached, add_self_loops)
        self.Aggre_ext = KAggre(K, cached, add_self_loops)
        self.lin = Linear(in_channels, out_channels, bias=bias)

    def forward(self, x: Tensor, inf_edge_index: Adj, edge_index: Adj,
                inf_edge_weight: OptTensor = None, edge_weight: OptTensor = None) -> Tensor:
        """"""
        if inf_edge_index is None:
            out = self.Aggre_int(x, edge_index, edge_weight)
            return self.lin(out)
        x_ext = self.Aggre_ext(x, inf_edge_index, inf_edge_weight)
        x_int = self.Aggre_int(x, edge_index, edge_weight)
        a_ext, a_int = cal_coeff(x_ext, x_int)
        out = a_ext * x_ext + a_int * x_int
        return self.lin(out)

class new2Conv(MessagePassing):
    def __init__(self, K: int, channels: int, alpha: float, theta: float = None,
                 layer: int = None, shared_weights: bool = True,
                 cached: bool = False, add_self_loops: bool = True, **kwargs):

        kwargs.setdefault('aggr', 'add')
        super().__init__(**kwargs)

        self.channels = channels
        self.alpha = alpha
        self.beta = 1.
        if theta is not None or layer is not None:
            assert theta is not None and layer is not None
            self.beta = log(theta / layer + 1)

        self.Aggre_int = KAggre(K, cached, add_self_loops)
        self.Aggre_ext = KAggre(K, cached, add_self_loops)

        self.weight1 = Parameter(torch.Tensor(channels, channels))

        if shared_weights:
            self.register_parameter('weight2', None)
        else:
            self.weight2 = Parameter(torch.Tensor(channels, channels))

        self.reset_parameters()

    def reset_parameters(self):
        glorot(self.weight1)
        glorot(self.weight2)

    def lin_propagate(self, x, x_0):
        x.mul_(1 - self.alpha)
        x_0 = self.alpha * x_0[:x.size(0)]

        if self.weight2 is None:
            out = x.add_(x_0)
            out = torch.addmm(out, out, self.weight1, beta=1. - self.beta,
                              alpha=self.beta)
        else:
            out = torch.addmm(x, x, self.weight1, beta=1. - self.beta,
                              alpha=self.beta)
            out += torch.addmm(x_0, x_0, self.weight2, beta=1. - self.beta,
                               alpha=self.beta)
        return out

    def forward(self, x: Tensor, x_0: Tensor, inf_edge_index: Adj, edge_index: Adj,
                inf_edge_weight: OptTensor = None, edge_weight: OptTensor = None) -> Tensor:
        """"""
        if inf_edge_index is None:
            out = self.Aggre_int(x, edge_index, edge_weight)
            return self.lin_propagate(out, x_0)

        x_ext = self.Aggre_ext(x, inf_edge_index, inf_edge_weight)
        x_int = self.Aggre_int(x, edge_index, edge_weight)
        a_ext, a_int = cal_coeff(x_ext, x_int)
        out = a_ext * x_ext + a_int * x_int

        return self.lin_propagate(out, x_0)

    def message(self, x_j: Tensor, edge_weight: OptTensor) -> Tensor:
        return x_j if edge_weight is None else edge_weight.view(-1, 1) * x_j

    def message_and_aggregate(self, adj_t: SparseTensor, x: Tensor) -> Tensor:
        return matmul(adj_t, x, reduce=self.aggr)

    def __repr__(self) -> str:
        return (f'{self.__class__.__name__}({self.channels}, '
                f'alpha={self.alpha}, beta={self.beta})')

class newSGConv(nn.Module):
    def __init__(self, in_channels: int, out_channels: int, K: int = 1, cached: bool = False, add_self_loops: bool = True, bias: bool = True):
        super().__init__()
        self.K = K
        self.Aggre_int = KAggre(1, cached, add_self_loops)
        self.Aggre_ext = KAggre(1, cached, add_self_loops)
        self.lin = Linear(in_channels, out_channels, bias=bias)

    def propagate(self, x, inf_edge_index, edge_index, inf_edge_weight, edge_weight):
        x_ext = self.Aggre_ext(x, inf_edge_index, inf_edge_weight)
        x_int = self.Aggre_int(x, edge_index, edge_weight)
        a_ext, a_int = cal_coeff(x_ext, x_int)
        out = a_ext * x_ext + a_int * x_int
        return out

    def forward(self, x: Tensor, inf_edge_index: Adj, edge_index: Adj,
                inf_edge_weight: OptTensor = None, edge_weight: OptTensor = None) -> Tensor:
        """"""
        for _ in range(self.K):
            if inf_edge_index is None:
                x = self.Aggre_int(x, edge_index, edge_weight)
            else:
                x = self.propagate(x, inf_edge_index, edge_index, inf_edge_weight, edge_weight)
        return self.lin(x)

class newGATConv(MessagePassing):
    def __init__(
        self,
        in_channels: Union[int, Tuple[int, int]],
        out_channels: int,
        heads: int = 1,
        concat: bool = True,
        negative_slope: float = 0.2,
        dropout: float = 0.0,
        add_self_loops: bool = True,
        edge_dim: Optional[int] = None,
        fill_value: Union[float, Tensor, str] = 'mean',
        bias: bool = True,
        **kwargs,
    ):
        kwargs.setdefault('aggr', 'add')
        super().__init__(node_dim=0, **kwargs)

        self.in_channels = in_channels
        self.out_channels = out_channels
        self.heads = heads
        self.concat = concat
        self.negative_slope = negative_slope
        self.dropout = dropout
        self.add_self_loops = add_self_loops
        self.edge_dim = edge_dim
        self.fill_value = fill_value

        # In case we are operating in bipartite graphs, we apply separate
        # transformations 'lin_src' and 'lin_dst' to source and target nodes:
        if isinstance(in_channels, int):
            self.lin_src = Linear(in_channels, heads * out_channels,
                                  bias=False, weight_initializer='glorot')
            self.lin_dst = self.lin_src
        else:
            self.lin_src = Linear(in_channels[0], heads * out_channels, False,
                                  weight_initializer='glorot')
            self.lin_dst = Linear(in_channels[1], heads * out_channels, False,
                                  weight_initializer='glorot')

        # The learnable parameters to compute attention coefficients:
        self.att_src = Parameter(torch.Tensor(1, heads, out_channels))
        self.att_dst = Parameter(torch.Tensor(1, heads, out_channels))
        self.att_src_inf = Parameter(torch.Tensor(1, heads, out_channels))
        self.att_dst_inf = Parameter(torch.Tensor(1, heads, out_channels))

        if edge_dim is not None:
            self.lin_edge = Linear(edge_dim, heads * out_channels, bias=False,
                                   weight_initializer='glorot')
            self.att_edge = Parameter(torch.Tensor(1, heads, out_channels))
        else:
            self.lin_edge = None
            self.register_parameter('att_edge', None)

        if bias and concat:
            self.bias = Parameter(torch.Tensor(heads * out_channels))
        elif bias and not concat:
            self.bias = Parameter(torch.Tensor(out_channels))
        else:
            self.register_parameter('bias', None)

        self.reset_parameters()

    def reset_parameters(self):
        self.lin_src.reset_parameters()
        self.lin_dst.reset_parameters()
        if self.lin_edge is not None:
            self.lin_edge.reset_parameters()
        glorot(self.att_src)
        glorot(self.att_dst)
        glorot(self.att_src_inf)
        glorot(self.att_dst_inf)
        glorot(self.att_edge)
        zeros(self.bias)

    def graph_attention(self, x, att_src, att_dst, edge_index, edge_attr=None, size=None):
        x_src, x_dst = x

        # Next, we compute node-level attention coefficients, both for source
        # and target nodes (if present):
        alpha_src = (x_src * att_src).sum(dim=-1)
        alpha_dst = None if x_dst is None else (x_dst * att_dst).sum(-1)
        alpha = (alpha_src, alpha_dst)

        if self.add_self_loops:
            if isinstance(edge_index, Tensor):
                # We only want to add self-loops for nodes that appear both as
                # source and target nodes:
                num_nodes = x_src.size(0)
                if x_dst is not None:
                    num_nodes = min(num_nodes, x_dst.size(0))
                num_nodes = min(size) if size is not None else num_nodes
                edge_index, edge_attr = remove_self_loops(
                    edge_index, edge_attr)
                edge_index, edge_attr = add_self_loops(
                    edge_index, edge_attr, fill_value=self.fill_value,
                    num_nodes=num_nodes)
            elif isinstance(edge_index, SparseTensor):
                if self.edge_dim is None:
                    edge_index = set_diag(edge_index)
                else:
                    raise NotImplementedError(
                        "The usage of 'edge_attr' and 'add_self_loops' "
                        "simultaneously is currently not yet supported for "
                        "'edge_index' in a 'SparseTensor' form")

        # edge_updater_type: (alpha: OptPairTensor, edge_attr: OptTensor)
        alpha = self.edge_updater(edge_index, alpha=alpha, edge_attr=edge_attr)

        # propagate_type: (x: OptPairTensor, alpha: Tensor)
        out = self.propagate(edge_index, x=x, alpha=alpha, size=size)
        return out

    def forward(self, x: Union[Tensor, OptPairTensor], inf_edge_index: Adj, edge_index: Adj,
                inf_edge_weight: OptTensor = None, edge_weight: OptTensor = None, size: Size = None):

        H, C = self.heads, self.out_channels

        # We first transform the input node features. If a tuple is passed, we
        # transform source and target node features via separate weights:
        if isinstance(x, Tensor):
            assert x.dim() == 2, "Static graphs not supported in 'GATConv'"
            x_src = x_dst = self.lin_src(x).view(-1, H, C)
        else:  # Tuple of source and target node features:
            x_src, x_dst = x
            assert x_src.dim() == 2, "Static graphs not supported in 'GATConv'"
            x_src = self.lin_src(x_src).view(-1, H, C)
            if x_dst is not None:
                x_dst = self.lin_dst(x_dst).view(-1, H, C)

        x = (x_src, x_dst)

        x_ext = self.graph_attention(x, self.att_src_inf, self.att_dst_inf, inf_edge_index, inf_edge_weight, size)
        x_int = self.graph_attention(x, self.att_src, self.att_dst, edge_index, edge_weight, size)

        if self.concat:
            x_ext = x_ext.view(-1, self.heads * self.out_channels)
            x_int = x_int.view(-1, self.heads * self.out_channels)
        else:
            x_ext = x_ext.mean(dim=1)
            x_int = x_int.mean(dim=1)

        a_ext, a_int = cal_coeff(x_ext, x_int)
        out = a_ext * x_ext + a_int * x_int

        if self.bias is not None:
            out += self.bias

        return out


    def edge_update(self, alpha_j: Tensor, alpha_i: OptTensor,
                    edge_attr: OptTensor, index: Tensor, ptr: OptTensor,
                    size_i: Optional[int]) -> Tensor:
        # Given edge-level attention coefficients for source and target nodes,
        # we simply need to sum them up to "emulate" concatenation:
        alpha = alpha_j if alpha_i is None else alpha_j + alpha_i

        if edge_attr is not None and self.lin_edge is not None:
            if edge_attr.dim() == 1:
                edge_attr = edge_attr.view(-1, 1)
            edge_attr = self.lin_edge(edge_attr)
            edge_attr = edge_attr.view(-1, self.heads, self.out_channels)
            alpha_edge = (edge_attr * self.att_edge).sum(dim=-1)
            alpha = alpha + alpha_edge

        alpha = F.leaky_relu(alpha, self.negative_slope)
        alpha = softmax(alpha, index, ptr, size_i)
        alpha = F.dropout(alpha, p=self.dropout, training=self.training)
        return alpha


    def message(self, x_j: Tensor, alpha: Tensor) -> Tensor:
        return alpha.unsqueeze(-1) * x_j

    def __repr__(self) -> str:
        return (f'{self.__class__.__name__}({self.in_channels}, '
                f'{self.out_channels}, heads={self.heads})')
