# Copyright 2024 Bytedance Ltd. and/or its affiliates
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""
Preprocess the NuminaMath-CoT dataset to parquet format
"""

import os
import datasets

from verl.utils.hdfs_io import copy, makedirs
import argparse

from verl.utils.reward_score.math import remove_boxed, last_boxed_only_string


def extract_solution(solution_str):
    return remove_boxed(last_boxed_only_string(solution_str))

def make_prefix(raw_question, template_type):
    # instruction_prefix provided by DeepSeek
    if template_type == 'base':
        instruction_prefix = f"""A conversation between User and Assistant. The user asks a question, and the Assistant solves it. The assistant first thinks about the reasoning process in the mind and then provides the user with the final answer.
User: {raw_question} Show your work in <think> </think> tags. And return the final answer within \\boxed{{}}.
Assistant: Let me solve this step by step.
<think>"""
    elif template_type == 'qwen-instruct':
        """This works for Qwen Instruct Models"""
        instruction_prefix = f"""<|im_start|>system\nYou are a helpful assistant. You first thinks about the reasoning process in the mind and then provides the user with the answer.<|im_end|>\n<|im_start|>user\n {raw_question} Show your work in <think> </think> tags. And return the final answer within \\boxed{{}}.<|im_end|>\n<|im_start|>assistant\nLet me solve this step by step.\n<think>"""
    return instruction_prefix


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument('--local_dir', default='~/data/math')
    parser.add_argument('--hdfs_dir', default=None)
    parser.add_argument('--template_type', type=str, default='base')

    args = parser.parse_args()

    # data_source = 'lighteval/MATH' # original data link
    data_source = 'AI-MO/NuminaMath-CoT'

    dataset = datasets.load_dataset(data_source, trust_remote_code=True)

    train_dataset = dataset['train']
    test_dataset = dataset['test']

    train_dataset = train_dataset.filter(
        lambda ex: last_boxed_only_string(ex["solution"]) is not None                  # ← predicate
        # or:   lambda ex: r"\boxed" in ex["solution"]
    )
    print(len(train_dataset))
    train_dataset = train_dataset.filter(
        lambda ex: extract_solution(ex["solution"]) is not None                  # ← predicate
        # or:   lambda ex: r"\boxed" in ex["solution"]
    )
    test_dataset = test_dataset.filter(
        lambda ex: last_boxed_only_string(ex["solution"]) is not None            # ← predicate
        # or:   lambda ex: r"\boxed" in ex["solution"]
    )
    print(len(test_dataset))
    test_dataset = test_dataset.filter(
        lambda ex: extract_solution(ex["solution"]) is not None            # ← predicate
        # or:   lambda ex: r"\boxed" in ex["solution"]
    )

    # after this, train_dataset contains only the wanted rows
    print(len(train_dataset))
    print(len(test_dataset))

    # instruction_following = "Let's think step by step and output the final answer within \\boxed{}."

    # add a row to each data item that represents a unique id
    def make_map_fn(split):

        def process_fn(example, idx):
            # pop no need colums
            source = example.pop('source')
            messages = example.pop('messages')
            
            question_raw = example.pop('problem')

            # question = question + ' ' + instruction_following
            question = make_prefix(question_raw, template_type=args.template_type)

            answer_raw = example.pop('solution')

            solution = extract_solution(answer_raw)
            data = {
                "data_source": data_source,
                "prompt": [{
                    "role": "user",
                    "content": question
                }],
                "ability": "math",
                "reward_model": {
                    "style": "rule",
                    "ground_truth": solution
                },
                "extra_info": {
                    'split': split,
                    'index': idx,
                    'answer': answer_raw,
                    "question": question_raw,
                }
            }
            return data

        return process_fn

    train_dataset = train_dataset.map(function=make_map_fn('train'), with_indices=True)
    test_dataset = test_dataset.map(function=make_map_fn('test'), with_indices=True)

    local_dir = args.local_dir
    hdfs_dir = args.hdfs_dir

    train_dataset.to_parquet(os.path.join(local_dir, 'train.parquet'))
    test_dataset.to_parquet(os.path.join(local_dir, 'test.parquet'))

    if hdfs_dir is not None:
        makedirs(hdfs_dir)

        copy(src=local_dir, dst=hdfs_dir)
