# GLoRA: Geometric Low-Rank Adaptation

GLoRA (Geometric Low-Rank Adaptation) is a theoretically grounded framework designed to optimize large language models by addressing computational efficiency and expressivity trade-offs during fine-tuning. This repository includes essential scripts and tools for implementing GLoRA.

## Table of Contents

- [Installation](#installation)
- [Repository Structure](#repo)
- [Usage](#usage)

## Installation

To get started with GLoRA, clone this repository and install the required dependencies:

```bash
git clone https://github.com/yourusername/glora.git
cd glora
pip install -r requirements.txt
```

## Repository Structure
This repository contains the following scripts:

1. `compute_intrinsic_dimensions.py`: Computes the intrinsic dimensions of data representation using the Two Nearest Neighbors technique.

2. `run_glue_glora.py`: Runs the GLoRA adaptation process on the GLUE benchmark datasets.

3. `tune_hyperparameters.py`: Tunes hyperparameters for optimal GLoRA performance, we focus on learning rate, weight decay, warmup ratio and type of scheduler, but it can be adapted.

4. `store_activations.py`: Stores the activations of model layers to be used for intrinsic dimension estimation.

5. `store_data.py`: Handles data storage and preprocessing for GLUE benchmark experiments.

## Usage
We start by storing the datasets
```bash
for task in rte cola sst2 stsb qnli mrpc
do 
    python store_datasets.py --task_name $task \
    --output_dir "glue_datasets"
done
```

Then, we compute the activations and store them
```bash
for task in cola rte sst2 stsb mrpc qnli
do
CUDA_VISIBLE_DEVICES=0,1,2,3,4,5,6,7 torchrun --nproc_per_node=8 store_activations.py \
    --task_name $task \
    --output_dir "glue_datasets" \
    --model_name_or_path microsoft/deberta-v3-base
done
```

After that, we can compute the intrinsic dimensions

```bash
for task in cola rte sst2 stsb mrpc qnli 
do
    python compute_intrinsic_dimensions.py \
        --task_name  $task 
done
```

Finally, we can tune the hyperparameters as follows
```bash
for task in sst2 
do
    CUDA_VISIBLE_DEVICES=0,1,2,3,4,5,6,7  torchrun --nproc_per_node=8 tune_hyperparameters.py \
        --task_name $task \
        --output_dir "glue_datasets" \
        --model_name_or_path microsoft/deberta-v3-base 
done
```

Once, we obtain the optimal hyperparameters, we set them on the `run_glue_glora.py` file, then we launch the fien-tuning:

```bash
for task in sst2 
do
    CUDA_VISIBLE_DEVICES=0,1,2,3,4,5,6,7  torchrun --nproc_per_node=8 run_glue_glora.py \
        --task_name $task \
        --output_dir "glue_datasets" \
        --model_name_or_path microsoft/deberta-v3-base 
done
```
