from typing import Iterable

import numpy as np

def board_to_image(board: "Board") -> np.array:
    """
    :param board: An array that corresponds to the
    board representation of some state for some
    driving_gridworld environment.
    """
    def color256_to_1000(c):
        return int(c / 255.0 * 999)

    COLOUR_FG = {
        ' ': (color256_to_1000(205), color256_to_1000(205), color256_to_1000(193)),
        '|': (0, 0, 0), 
        'd': (color256_to_1000(102), color256_to_1000(102), color256_to_1000(0)),
        'C': (0, 999, 999),
        'b': (color256_to_1000(128), color256_to_1000(0), color256_to_1000(128)),
        'p': (987, 623, 145),
        '^': (color256_to_1000(39), color256_to_1000(232), color256_to_1000(51))
    }
    return np.array([[COLOUR_FG[chr(x)] for x in row] for row in board]) / 1000

def make_board_from_info(
        car_col: int,
        car_speed: int,
        obstacles: Iterable["Obstacle"],
        headlight_range: int,
) -> "Board":
    """
    Construct a board from some state info.
    :param car_col: The column that the car sits in.
    :param car_speed: The speed of the car.
    :param obstacles: An iterable of the obstacles in the board.
    :param headlight_range: The headlight range of the environment.
    :returns: a board corresponding to the information provided.
    """
    board = [[124, 100, 32, 32, 100, 124, 32] for _ in range(headlight_range + 1)]
    board[-1][car_col + 1] = 67
    for obstacle in obstacles:
        _, row, col, _, _ = obstacle
        board[row][col + 1] = 112
    for i in range(car_speed):
        board[len(board) - 1 - i][-1] = 94
    return np.array(board)

def standardize_y_axis(figs: Iterable["Figure"]) -> None:
    """
    Standardizes the y axes for all of the figs given.
    Ie, makes their limits the same.
    This is done in place, hence returning None.
    """
    y_min_tot, y_max_tot = float('inf'), -float('inf')
    for fig in figs:
        ax0, ax1 = fig.axes
        y_min, y_max = ax1.get_ylim()
        y_min_tot = min(y_min_tot, y_min)
        y_max_tot = max(y_max_tot, y_max)
    for fig in figs:
        ax0, ax1 = fig.axes
        ax1.set_ylim((y_min_tot, y_max_tot))
