import os

import numpy as np  # type: ignore
import torch
from torch.distributions import Normal
from torch.distributions.kl import kl_divergence
from torch.utils.data import DataLoader

from dataloaders.utils import PBPDataset
from models.deep_ensemble import Model
from models.set_encoder import SetEncoder
from models.swag import SWAG
from utils import Args, ModelSaveDict, Stats, get_mixture_mu_var


def schedule(epoch: int, args: Args) -> float:
    t = epoch / args.swag_start
    lr_ratio = args.swag_lr / args.swag_lr_init
    if t <= 0.5:
        factor = 1.0
    elif t <= 0.9:
        factor = 1.0 - (1.0 - lr_ratio) * (t - 0.5) / 0.4
    else:
        factor = lr_ratio
    return args.swag_lr_init * factor


def adjust_learning_rate(optimizer: torch.optim.SGD, lr: float) -> float:
    for param_group in optimizer.param_groups:
        param_group["lr"] = lr
    return lr


def train(args: Args, run: int) -> ModelSaveDict:
    if args.train_set is None or args.test_set is None:
        raise ValueError("train or test cannot be none")

    model_args = [args.x_dim, args.h_dim, args.y_dim]
    model = Model(*model_args).to(args.device)

    swag_model = SWAG(
        Model,
        args.device,
        no_cov_mat=False,
        max_num_models=args.model_n,
        model_args=model_args,
    ).to(args.device)

    gen = SetEncoder(args.x_dim, args.h_dim, args.x_dim).to(args.device)
    gen_opt = torch.optim.Adam(gen.parameters(), lr=args.beta_lr)

    optimizer = torch.optim.SGD(
        model.parameters(),
        lr=args.swag_lr_init,
        momentum=args.swag_momentum,
        weight_decay=args.weight_decay,
    )

    exp_norm = torch.ones(args.x_dim).norm().float() * args.norm_c
    args.swag_start = args.epochs // 2

    for epoch in range(args.epochs):
        lr = schedule(epoch, args)
        adjust_learning_rate(optimizer, lr)
        for i, ((x, y, _), (x_phi, _, _)) in enumerate(
            zip(args.train_set, args.train_set)
        ):
            # for i, (x, y, _) in enumerate(args.train_set):
            if x.size(0) < 6:
                continue

            x, y = x.to(args.device), y.to(args.device)
            x_phi = x_phi.to(args.device)

            b, ft = x.size()
            if epoch > args.epochs // 2:
                gen_mu, gen_logvar, _ = gen(x)
                x_gen = Normal(gen_mu, torch.exp(gen_logvar / 2)).rsample()
                x = torch.cat((x, x_gen), dim=0)

            mu, logvar = model(x)
            logvar = torch.clamp(logvar, args.logvar_min, args.logvar_max)

            mu_nat, logvar_nat = mu[:b], logvar[:b]
            mu_gen, logvar_gen = mu[b:], logvar[b:]

            loss = (torch.exp(-logvar_nat) * (y - mu_nat) ** 2 + logvar_nat).mean()

            if epoch > args.epochs // 2:
                d, _ = (
                    ((x[b:].unsqueeze(1).detach() - x[:b].unsqueeze(0)) ** 2)
                    .sum(dim=2)
                    .min(dim=1)
                )

                # print(f"min nat dist: {min_nat_dist}")
                w = 1 - torch.exp(-d / (2 * args.ls ** 2))

                # kl divergence
                kl = kl_divergence(
                    Normal(mu_gen, torch.exp(logvar_gen / 2)), Normal(mu_gen, 1.0)
                )
                loss += (w * kl).sum()

            # optimize phase one
            optimizer.zero_grad()
            loss.backward()
            optimizer.step()

            # PHASE TWO: generate, forward, minimize generatro obbjective
            gen_mu, gen_logvar, subsets = gen(x_phi)
            x_gen = Normal(gen_mu, torch.exp(gen_logvar / 2)).rsample()

            # forward pass with the new samples
            mu, logvar = model(x_gen)
            logvar = torch.clamp(logvar, args.logvar_min, args.logvar_max)

            gen_entropy = Normal(gen_mu, torch.exp(gen_logvar / 2)).entropy().mean()
            entropy = Normal(mu, torch.exp(logvar / 2)).entropy().mean()

            # the loss here should be towards low entropy, away from the data
            nat_dist = (
                ((x_gen.unsqueeze(1) - x_phi[subsets]) ** 2).sum(dim=2).mean(dim=1)
            )
            # print(((x_gen.unsqueeze(1) - x_phi[subsets]) ** 2).size())

            nat_dist = nat_dist[nat_dist > (exp_norm * 1)]
            if nat_dist.size(0) == 0:
                nat_dist = torch.tensor(0.0)

            loss = entropy - gen_entropy + nat_dist.mean()

            if torch.any(torch.isinf(loss)) or torch.any(torch.isnan(loss)):
                raise OverflowError(
                    f"x gen: {x_gen} nat_dist:{nat_dist} nat_dist:{nat_dist.mean()}"
                )

            gen_opt.zero_grad()
            loss.backward()
            gen_opt.step()

        if epoch + 1 >= args.swag_start:
            # maybe this doesn't need to be done on every epoch (it didn't in the original)
            swag_model.collect_model(model)
            if epoch == 0 or epoch % 5 == 0 or epoch == args.epochs - 1:
                swag_model.sample(0.0)

    models: ModelSaveDict = {
        os.path.join(
            args.get_model_dir(run), "model.pt",
        ): swag_model.cpu().state_dict(),
        os.path.join(args.get_model_dir(run), "encoder.pt",): gen.cpu().state_dict(),
    }

    return models


def eval(args: Args, run: int, models: ModelSaveDict) -> Stats:
    if args.train_set is None or args.test_set is None:
        raise ValueError("train or test cannot be none")

    model_args = [args.x_dim, args.h_dim, args.y_dim]
    model = Model(*model_args).to(args.device)
    swag_model = SWAG(
        Model,
        args.device,
        no_cov_mat=False,
        max_num_models=args.model_n,
        model_args=model_args,
    ).to(args.device)

    swag_model.load_state_dict(
        models[os.path.join(args.get_model_dir(run), "model.pt",)]
    )

    gen = SetEncoder(args.x_dim, args.h_dim, args.x_dim).to(args.device)
    gen.load_state_dict(models[os.path.join(args.get_model_dir(run), "encoder.pt")])
    gen.eval()

    t = Stats(args.test_set.dataset, args)
    y_mu, y_sigma = args.test_set.dataset.get_y_moments()  # type: ignore
    model.eval()
    with torch.no_grad():
        for i, (x, y, _) in enumerate(args.test_set):
            x, y = x.to(args.device), y.to(args.device)

            mus, logvars = swag_model.mc(x, args.samples)
            logvars = torch.clamp(logvars, args.logvar_min, args.logvar_max)

            mu, var = get_mixture_mu_var(mus, logvars)
            sigma = torch.sqrt(var)

            # calculate log likelihood before denormalizing, can cause nan otherwise
            log_lik = Normal(mu, sigma).log_prob(y).sum().item()

            # denormalize the data
            mu = mu * y_sigma + y_mu
            sigma *= y_sigma
            y = y * y_sigma + y_mu

            t.set_history(mu, sigma, y)
            t.set_regression_stats(mu, sigma, y, log_lik)

        t.normalize_stats(len(args.test_set.dataset))
        t.ll -= np.log(y_sigma)  # brings std normal space back to input

    return t
