import os

import torch

from models.deep_ensemble import ModelConv as Model
from models.swag import SWAG
from utils import Args, ClassificationStats, ModelSaveDict, get_conv_args


def schedule(epoch: int, args: Args) -> float:
    t = epoch / args.swag_start
    lr_ratio = args.swag_lr / args.swag_lr_init
    if t <= 0.5:
        factor = 1.0
    elif t <= 0.9:
        factor = 1.0 - (1.0 - lr_ratio) * (t - 0.5) / 0.4
    else:
        factor = lr_ratio
    return args.swag_lr_init * factor


def adjust_learning_rate(optimizer: torch.optim.SGD, lr: float) -> float:
    for param_group in optimizer.param_groups:
        param_group["lr"] = lr
    return lr


def train(args: Args, run: int) -> ModelSaveDict:
    if args.train_set is None or args.test_set is None:
        raise ValueError("train or test cannot be none")

    model_args = [(args.x_dim, args.x_dim), args.h_dim, args.y_dim, get_conv_args(args)]
    model = Model(*model_args).to(args.device)  # type: ignore

    swag_model = SWAG(
        Model,
        args.device,
        no_cov_mat=False,
        max_num_models=args.model_n,
        model_args=model_args,
    ).to(args.device)

    optimizer = torch.optim.SGD(
        model.parameters(),
        lr=args.swag_lr_init,
        momentum=args.swag_momentum,
        weight_decay=args.weight_decay,
    )

    criterion = torch.nn.NLLLoss()
    args.swag_start = args.epochs // 2

    for epoch in range(args.epochs):
        lr = schedule(epoch, args)
        adjust_learning_rate(optimizer, lr)
        model.train()
        for i, (x, y) in enumerate(args.train_set):
            x, y = (
                x.to(args.device, non_blocking=True),
                y.to(args.device, non_blocking=True),
            )

            yhat = model(x)
            loss = criterion(torch.log_softmax(yhat, dim=1), y)

            if torch.any(torch.isinf(loss)) or torch.any(torch.isnan(loss)):
                raise OverflowError("overflow")

            optimizer.zero_grad()
            loss.backward()
            optimizer.step()

        if epoch >= args.swag_start:
            # maybe this doesn't need to be done on every epoch (it didn't in the original)
            swag_model.collect_model(model)
            if epoch == 0 or epoch % 5 == 0 or epoch == args.epochs - 1:
                swag_model.sample(0.0)

        if epoch % 10 == 0:
            t = ClassificationStats(
                args.test_set.dataset, args, length=100 * args.batch_size * 10,
            )
            model.eval()
            with torch.no_grad():
                for i, (x, y) in enumerate(args.test_set):
                    x, y = x.to(args.device), y.to(args.device)

                    yhat = model(x)
                    t.set_multiclass(y, yhat)

                    if i == 99:
                        break

            t.calc_stats()
            print(f"epoch: {epoch} accuracy: {t.accuracy:.4f} cal: {t.cal_error:.4f}")

    models: ModelSaveDict = {
        os.path.join(
            args.get_model_dir(run, classification=True), "model.pt",
        ): swag_model.cpu().state_dict(),
    }

    return models


def eval(args: Args, run: int, models: ModelSaveDict) -> ClassificationStats:
    if args.train_set is None or args.test_set is None:
        raise ValueError("train or test cannot be none")

    model_args = [(args.x_dim, args.x_dim), args.h_dim, args.y_dim, get_conv_args(args)]
    model = Model(*model_args).to(args.device)  # type: ignore
    swag_model = SWAG(
        Model,
        args.device,
        no_cov_mat=False,
        max_num_models=args.model_n,
        model_args=model_args,
    ).to(args.device)

    swag_model.load_state_dict(
        models[os.path.join(args.get_model_dir(run, classification=True), "model.pt",)]
    )

    t = ClassificationStats(args.test_set.dataset, args)
    model.eval()
    with torch.no_grad():
        for i, (x, y) in enumerate(args.test_set):
            x, y = x.to(args.device), y.to(args.device)

            yhat = swag_model.mc_class(x, args.samples).mean(dim=0)
            t.set_multiclass(y, yhat)

    t.calc_stats()
    return t
