import numpy as np
import math
import torch
import core.func as func


def get_skip(alphas, betas):
    N = len(betas) - 1
    skip_alphas = np.ones([N + 1, N + 1], dtype=betas.dtype)
    for s in range(N + 1):
        skip_alphas[s, s + 1:] = alphas[s + 1:].cumprod()
    skip_betas = np.zeros([N + 1, N + 1], dtype=betas.dtype)
    for t in range(N + 1):
        prod = betas[1: t + 1] * skip_alphas[1: t + 1, t]
        skip_betas[:t, t] = (prod[::-1].cumsum())[::-1]
    return skip_alphas, skip_betas


class Schedule(object):
    def __init__(self, betas, alphas=None):
        r""" for n>=1, betas[n] is the variance of q(xn|xn-1)
             for n=0,  betas[0]=0
        """

        self.betas = betas
        self.alphas = 1. - self.betas if alphas is None else alphas
        self.N = len(betas) - 1

        assert isinstance(self.betas, np.ndarray) and self.betas[0] == 0
        assert isinstance(self.alphas, np.ndarray) and self.alphas[0] == 1
        assert len(self.betas) == len(self.alphas)

        # skip_alphas[s, t] = alphas[s + 1: t + 1].prod()
        self.skip_alphas, self.skip_betas = get_skip(self.alphas, self.betas)
        self.cum_alphas = self.skip_alphas[0]  # cum_alphas = alphas.cumprod()
        self.cum_betas = self.skip_betas[0]
        self.snr = self.cum_alphas / self.cum_betas

    def tilde_beta(self, s, t):
        return self.skip_betas[s, t] * self.cum_betas[s] / self.cum_betas[t]

    def sample(self, x0):  # sample from q(xn|x0), where n is uniform
        n = np.random.choice(list(range(1, self.N + 1)), (len(x0),))
        eps = torch.randn_like(x0)
        xn = func.stp(self.cum_alphas[n] ** 0.5, x0) + func.stp(self.cum_betas[n] ** 0.5, eps)
        return n, eps, xn

    def sub_sample(self, x0, num_steps=10):
        from .trajectory import _choice_steps_linear

        steps = _choice_steps_linear(self.N, num_steps)
        n = np.random.choice(steps, (len(x0),))
        eps = torch.randn_like(x0)
        xn = func.stp(self.cum_alphas[n] ** 0.5, x0) + func.stp(self.cum_betas[n] ** 0.5, eps)
        return n, eps, xn

    def split_sample(self, x0, split_idx, part):
        if part == 0:
            n = np.random.choice(list(range(1, split_idx + 1)), (len(x0),))
        elif part == 1:
            n = np.random.choice(list(range(split_idx, self.N + 1)), (len(x0),))
        else:
            raise ValueError(f'part={part}')
        eps = torch.randn_like(x0)
        xn = func.stp(self.cum_alphas[n] ** 0.5, x0) + func.stp(self.cum_betas[n] ** 0.5, eps)
        return n, eps, xn

    def __repr__(self):
        return f'Schedule({self.betas[:10]}..., {self.N})'


class NamedSchedule(Schedule):
    def __init__(self, schedule_name, N):
        betas = np.append(0., get_named_beta_schedule(schedule_name, N))
        super().__init__(betas)
        self.schedule_name = schedule_name
        self.N = N

    def __repr__(self):
        return f'NamedSchedule({self.schedule_name}, {self.N})'


def get_named_beta_schedule(schedule_name, num_diffusion_timesteps):
    """
    Get a pre-defined beta schedule for the given name.

    The beta schedule library consists of beta schedules which remain similar
    in the limit of num_diffusion_timesteps.
    Beta schedules may be added, but should not be removed or changed once
    they are committed to maintain backwards compatibility.
    """
    if schedule_name == "linear":
        # Linear schedule from Ho et al, extended to work for any number of
        # diffusion steps.
        scale = 1000 / num_diffusion_timesteps
        beta_start = scale * 0.0001
        beta_end = scale * 0.02
        return np.linspace(
            beta_start, beta_end, num_diffusion_timesteps, dtype=np.float64
        )
    elif schedule_name == "cosine":
        return betas_for_alpha_bar(
            num_diffusion_timesteps,
            lambda t: math.cos((t + 0.008) / 1.008 * math.pi / 2) ** 2,
        )
    else:
        raise NotImplementedError(f"unknown beta schedule: {schedule_name}")


def betas_for_alpha_bar(num_diffusion_timesteps, alpha_bar, max_beta=0.999):
    """
    Create a beta schedule that discretizes the given alpha_t_bar function,
    which defines the cumulative product of (1-beta) over time from t = [0,1].

    :param num_diffusion_timesteps: the number of betas to produce.
    :param alpha_bar: a lambda that takes an argument t from 0 to 1 and
                      produces the cumulative product of (1-beta) up to that
                      part of the diffusion process.
    :param max_beta: the maximum beta to use; use values lower than 1 to
                     prevent singularities.
    """
    betas = []
    for i in range(num_diffusion_timesteps):
        t1 = i / num_diffusion_timesteps
        t2 = (i + 1) / num_diffusion_timesteps
        betas.append(min(1 - alpha_bar(t2) / alpha_bar(t1), max_beta))
    return np.array(betas)
