#ifndef OPEN_SPIEL_ALGORITHMS_PCFR_H_
#define OPEN_SPIEL_ALGORITHMS_PCFR_H_

#include <string>
#include <utility>
#include <vector>

#include "open_spiel/spiel.h"
#include "info_state_values.hpp"
#include "average_policy.hpp"
#include "current_policy.hpp"
#include "cfr.hpp"

namespace open_spiel {
namespace algorithms {

class PCFRSolver : public MyCFRSolver {
public:
    static inline const std::string alg_name = "full_pfcr";

    PCFRSolver(const Game &game, double mutation_rate, int update_anchoring_interval = 0, const int asym_player_id = -1, const bool regret_matching_plus = false)
            : MyCFRSolver(game,
                          update_anchoring_interval > 0?
                              alg_name + "/mutation_rate_" + std::to_string(mutation_rate) + "_tsig_" + std::to_string(update_anchoring_interval) + "_asym_player_id_" + std::to_string(asym_player_id):
                              alg_name + "/mutation_rate_" + std::to_string(mutation_rate) + "_asym_player_id_" + std::to_string(asym_player_id),
                          regret_matching_plus),
              mutation_rate_(mutation_rate),
              update_anchoring_interval_(update_anchoring_interval),
              asym_player_id_(asym_player_id)
              {}

    PCFRSolver(const Game &game, const std::string name, double mutation_rate, const int update_anchoring_interval, const int asym_player_id = -1, const bool regret_matching_plus = false)
            : MyCFRSolver(game, name, regret_matching_plus),
              mutation_rate_(mutation_rate),
              update_anchoring_interval_(update_anchoring_interval),
              asym_player_id_(asym_player_id)
              {}

    virtual ~PCFRSolver() = default;

    // Performs one step of the PCFR algorithm.
    void EvaluateAndUpdatePolicy() override;

protected:
    std::vector<double> ComputeCounterFactualRegretForActionProbs(
            const State &state, const absl::optional<int> &alternating_player,
            const std::vector<double> &reach_probabilities, const int current_player,
            const std::vector<double> &info_state_policy, const std::vector<Action> &legal_actions,
            std::vector<double> *child_values_out) override;
    void UpdateAnchoringStrategy();

    const double mutation_rate_;
    const int update_anchoring_interval_;
    const int asym_player_id_;
};

class PCFRPlusSolver: public PCFRSolver {
public:
    static inline const std::string alg_name = "full_pfcr_plus";

    PCFRPlusSolver(const Game &game, double mutation_rate, int update_anchoring_interval = 0, int asym_player_id = -1)
            : PCFRSolver(game,
                          update_anchoring_interval > 0?
                              alg_name + "/mutation_rate_" + std::to_string(mutation_rate) + "_tsig_" + std::to_string(update_anchoring_interval) + "_asym_player_id_" + std::to_string(asym_player_id):
                              alg_name + "/mutation_rate_" + std::to_string(mutation_rate) + "_asym_player_id_" + std::to_string(asym_player_id),                          
                          mutation_rate,
                          update_anchoring_interval,
                          asym_player_id,
                          true) {}

    PCFRPlusSolver(const Game &game, const std::string name, double mutation_rate, int update_anchoring_interval = 0, int asym_player_id = -1)
            : PCFRSolver(game,
                          std::move(name),
                          mutation_rate,
                          update_anchoring_interval,
                          asym_player_id,
                          true) {}

    virtual ~PCFRPlusSolver() = default;

};

}  // namespace algorithms
}  // namespace open_spiel

#endif  // OPEN_SPIEL_ALGORITHMS_PCFR_H_