import argparse
import os
import os.path as osp

from easydict import EasyDict as edict

from dinoreg.utils.common import ensure_dir


_C = edict()

# random seed
_C.seed = 7351

# dirs
_C.working_dir = osp.dirname(osp.realpath(__file__))
_C.root_dir = osp.dirname(osp.dirname(_C.working_dir))
_C.exp_name = osp.basename(_C.working_dir)
_C.output_dir = osp.join(_C.root_dir, 'output', _C.exp_name)
_C.snapshot_dir = osp.join(_C.output_dir, 'snapshots')
_C.log_dir = osp.join(_C.output_dir, 'logs')
_C.event_dir = osp.join(_C.output_dir, 'events')
_C.feature_dir = osp.join(_C.output_dir, 'features')

ensure_dir(_C.output_dir)
ensure_dir(_C.snapshot_dir)
ensure_dir(_C.log_dir)
ensure_dir(_C.event_dir)
ensure_dir(_C.feature_dir)

# data
_C.data = edict()
_C.data.dataset_root = '---Need-to-Set---'

# train data
_C.train = edict()
_C.train.batch_size = 1
_C.train.num_workers = 8
_C.train.point_limit = 30000
_C.train.use_augmentation = True
_C.train.augmentation_noise = 0.01
_C.train.augmentation_min_scale = 0.8
_C.train.augmentation_max_scale = 1.2
_C.train.augmentation_shift = 2.0
_C.train.augmentation_rotation = 1.0

# test config
_C.test = edict()
_C.test.batch_size = 1
_C.test.num_workers = 8
_C.test.point_limit = None

# eval config
_C.eval = edict()
_C.eval.acceptance_overlap = 0.0
_C.eval.acceptance_radius = 1.0
_C.eval.inlier_ratio_threshold = 0.05
_C.eval.rre_threshold = 5.0
_C.eval.rte_threshold = 2.0

# ransac
_C.ransac = edict()
_C.ransac.distance_threshold = 0.3
_C.ransac.num_points = 4
_C.ransac.num_iterations = 50000

# optim
_C.optim = edict()
_C.optim.lr = 1e-4
_C.optim.lr_decay = 0.95
_C.optim.lr_decay_steps = 4
_C.optim.weight_decay = 1e-6
_C.optim.max_epoch = 60
_C.optim.grad_acc_steps = 1

# model - Geometric Branch
_C.geometric_branch = edict()
_C.geometric_branch.num_stages = 5
_C.geometric_branch.init_voxel_size = 0.3
_C.geometric_branch.kernel_size = 15
_C.geometric_branch.base_radius = 4.25
_C.geometric_branch.base_sigma = 2.0
_C.geometric_branch.init_radius = _C.geometric_branch.base_radius * _C.geometric_branch.init_voxel_size
_C.geometric_branch.init_sigma = _C.geometric_branch.base_sigma * _C.geometric_branch.init_voxel_size
_C.geometric_branch.group_norm = 32
_C.geometric_branch.input_dim = 1
_C.geometric_branch.init_dim = 64
_C.geometric_branch.output_dim = 256

# model - Visual Branch
_C.visual_branch = edict()
_C.visual_branch.backbone = 'dinov2-small'
_C.visual_branch.image_size = 378
_C.visual_branch.backbone_output_size = 384
_C.visual_branch.window_radius = 1
_C.visual_branch.fusion_output_size = 512

# model - Global
_C.model = edict()
_C.model.ground_truth_matching_radius = 0.6
_C.model.num_points_in_patch = 128
_C.model.num_sinkhorn_iterations = 100

# model - Coarse Matching
_C.coarse_matching = edict()
_C.coarse_matching.num_targets = 128
_C.coarse_matching.overlap_threshold = 0.1
_C.coarse_matching.num_correspondences = 128
_C.coarse_matching.dual_normalization = True

# model - VGTransformer
_C.vgtransformer = edict()
_C.vgtransformer.hidden_dim = 256
_C.vgtransformer.output_dim = 256
_C.vgtransformer.num_heads = 4
_C.vgtransformer.blocks = ['self', 'cross', 'self', 'cross', 'self', 'cross']
_C.vgtransformer.sigma_d = 4.8
_C.vgtransformer.sigma_a = 15
_C.vgtransformer.angle_k = 3

# model - Fine Matching
_C.fine_matching = edict()
_C.fine_matching.topk = 2
_C.fine_matching.acceptance_radius = 0.6
_C.fine_matching.mutual = True
_C.fine_matching.confidence_threshold = 0.05
_C.fine_matching.use_dustbin = False
_C.fine_matching.use_global_score = False
_C.fine_matching.correspondence_threshold = 3
_C.fine_matching.correspondence_limit = None
_C.fine_matching.num_refinement_steps = 5

# loss - Coarse level
_C.coarse_loss = edict()
_C.coarse_loss.positive_margin = 0.1
_C.coarse_loss.negative_margin = 1.4
_C.coarse_loss.positive_optimal = 0.1
_C.coarse_loss.negative_optimal = 1.4
_C.coarse_loss.log_scale = 40
_C.coarse_loss.positive_overlap = 0.1

# loss - Fine level
_C.fine_loss = edict()
_C.fine_loss.positive_radius = 0.6

# loss - Overall
_C.loss = edict()
_C.loss.weight_coarse_loss = 1.0
_C.loss.weight_fine_loss = 1.0


def make_cfg():
    return _C


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('--link_output', dest='link_output', action='store_true', help='link output dir')
    args = parser.parse_args()
    return args


def main():
    cfg = make_cfg()
    args = parse_args()
    if args.link_output:
        os.symlink(cfg.output_dir, 'output')


if __name__ == '__main__':
    main()
