import sys
import pandas as pd
import os
import csv
import time
import json
import argparse
import gurobipy as gp
from gurobipy import GRB


def gurobi_cal(lp_name, time_limit, seed):

    m = gp.read(lp_name)
    # m.Params.MIPGap = 5e-2
    m.Params.timeLimit = time_limit
    m.Params.seed = seed

    m.optimize()

    res={
        "Status":m.status,
        "Runtime":m.runtime,
        "ObjVal":m.objval,
        "ObjBound":m.objbound,
        "MIPGap":m.mipgap,
        "IterCount":m.itercount,
        "BarIterCount":m.baritercount,
        "NodeCount":m.nodecount,
        "SolCount":m.solcount,
    }

    return res


def gurobi_main(lp_path, problem, method,time_limit, gap_limit, limit_para):
    '''
    :param lp_path: string: lp所处的位置：比如说'data/instances/cauctions/'
    :param problem: string: 问题类型：'setcover', 'cauctions', 'facilities', 'indset'
    :param method: string: 使用求解器的类型：'GCNN' 或者 'SCIP_default'
    :param time_limit: int: 设置求解器求解的时间限制
    :param gap_limit: float:  设置求解器求解的gap限制
    :param limit_para: string:  测试问题类型：限制时间 则设置'time',限制gap则设置'gap'
    :return:
    :param result_df: 返回求解结果的csv dataframe
    '''
    argsGpu = -1
    instances = []
    device = "CPU" if argsGpu == -1 else "GPU"
    if limit_para == 'time':
        result_file = f"GUROBI_time_{time_limit}_"+problem+".csv"
    elif limit_para == 'gap':
        result_file = f"GUROBI_gap_{gap_limit}_"+problem+".csv"
    eval_dir = f"eval_results/{problem}"
    os.makedirs(eval_dir, exist_ok=True)
    result_file = f"{eval_dir}/{result_file}"

    catl_eval_instance = os.listdir(lp_path)
    for temp_eval_instance in catl_eval_instance:
        if 'MPS' in temp_eval_instance.upper() or 'LP' in temp_eval_instance.upper():
            instances += [
                {'type': 'original', 'path': lp_path + temp_eval_instance}]

    print(f"problem: {problem}")
    print(f"time limit: {time_limit} s")

    print("running GUROBI...")

    fieldnames = [
        'problem',
        'policy',
        'seed',
        'instance',
        'nnodes',
        'stime',
        'gap',
        'objVal',
        'status'
    ]

    with open(result_file, 'a', newline='') as csvfile:
        writer = csv.DictWriter(csvfile, fieldnames=fieldnames)
        writer.writeheader()

        for seed, instance in enumerate(instances):
            print(f"{instance['type']}: {instance['path']}...")

            m = gp.read(instance['path'])

            if limit_para == 'time':
                m.Params.timeLimit = time_limit
            elif limit_para == 'gap':
                m.Params.MIPGap = gap_limit
            m.Params.seed = seed

            m.optimize()

            res={
                'problem': problem,
                'policy': "GUROBI",
                'seed': seed,
                'instance': instance['path'],
                "status": 'optimal' if m.status==2 else 'unknown',
                'nnodes':m.nodecount,
                'stime':m.runtime,
                'gap':m.mipgap,
                'objVal':m.objval,
            }

            writer.writerow(res)
            csvfile.flush()
            print(res)

    result_df = pd.read_csv(result_file)

    return result_df


if __name__ == '__main__':
    problem = 'item_placement'
    lp_path = 'instances/'+problem+'/test/'
    method = 'GUROBI'
    time_limit = 900
    gap_limit = 0.05
    limit_para = 'time'
    result_df = gurobi_main(lp_path, problem, method,time_limit, gap_limit, limit_para)