import sys
import pandas as pd
import os
import csv
import time
import json
import argparse
import gurobipy as gp
from gurobipy import GRB


def gurobi_cal(lp_name, time_limit, seed):

    m = gp.read(lp_name)
    # m.Params.MIPGap = 5e-2
    m.Params.timeLimit = time_limit
    m.Params.seed = seed

    m.optimize()

    res={
        "Status":m.status,
        "Runtime":m.runtime,
        "ObjVal":m.objval,
        "ObjBound":m.objbound,
        "MIPGap":m.mipgap,
        "IterCount":m.itercount,
        "BarIterCount":m.baritercount,
        "NodeCount":m.nodecount,
        "SolCount":m.solcount,
    }

    return res


def gurobi_main(problem, lp_path):
    '''
    :param lp_path: string: the path of lp file
    :param problem: string: 'setcover', 'cauctions', 'facilities', 'indset'
    :return:
    :param result_df: 返回求解结果的csv dataframe
    '''
    time_limit = 900 # set the time limit for solver
    gap_limit = 0 # set gap limit for solver
    limit_para = 'time' # set the limit type for solver
    argsGpu = -1 # using GPU or not, -1 represent not use
    instances = []
    device = "CPU" if argsGpu == -1 else "GPU"
    if limit_para == 'time':
        result_file = f"GUROBI_time_{time_limit}_"+problem+".csv"
    elif limit_para == 'gap':
        result_file = f"GUROBI_gap_{gap_limit}_"+problem+".csv"
    eval_dir = f"eval_results/{problem}"
    os.makedirs(eval_dir, exist_ok=True)
    result_file = f"{eval_dir}/{result_file}"

    catl_eval_instance = os.listdir(lp_path)
    for temp_eval_instance in catl_eval_instance:
        if 'MPS' in temp_eval_instance.upper() or 'LP' in temp_eval_instance.upper():
            instances += [
                {'type': 'original', 'path': lp_path + temp_eval_instance}]

    print(f"problem: {problem}")
    print(f"time limit: {time_limit} s")

    print("running GUROBI...")

    fieldnames = [
        'problem',
        'policy',
        'seed',
        'instance',
        'nnodes',
        'stime',
        'gap',
        'objVal',
        'status'
    ]

    with open(result_file, 'a', newline='') as csvfile:
        writer = csv.DictWriter(csvfile, fieldnames=fieldnames)
        writer.writeheader()

        for seed, instance in enumerate(instances):
            print(f"{instance['type']}: {instance['path']}...")

            m = gp.read(instance['path'])

            if limit_para == 'time':
                m.Params.timeLimit = time_limit
            elif limit_para == 'gap':
                m.Params.MIPGap = gap_limit
            m.Params.seed = seed

            m.optimize()

            res={
                'problem': problem,
                'policy': "GUROBI",
                'seed': seed,
                'instance': instance['path'],
                "status": 'optimal' if m.status==2 else 'unknown',
                'nnodes':m.nodecount,
                'stime':m.runtime,
                'gap':m.mipgap,
                'objVal':m.objval,
            }

            writer.writerow(res)
            csvfile.flush()
            print(res)

    result_df = pd.read_csv(result_file)

    return result_df


if __name__ == '__main__':
    problem = 'item_placement'
    lp_path = 'instances/'+problem+'/test/'
    result_df = gurobi_main(problem, lp_path)