from pathlib import Path
import argparse
import os
import sys
import time
import json
import urllib
import random
import math
from torch import nn, optim
from torchvision import models, datasets, transforms
import torch
import torchvision


parser = argparse.ArgumentParser(description='Evaluate resnet50 features on ImageNet')
parser.add_argument('--data', type=Path, metavar='DIR',
                    help='path to dataset')
parser.add_argument('--workers', default=8, type=int, metavar='N',
                    help='number of data loader workers')
parser.add_argument('--epochs', default=100, type=int, metavar='N',
                    help='number of total epochs to run')
parser.add_argument('--batch-size', default=256, type=int, metavar='N',
                    help='mini-batch size')
parser.add_argument('--lr', default=1.0, type=float, metavar='LR',
                    help='classifier base learning rate')
parser.add_argument('--weight-decay', default=1e-6, type=float, metavar='W',
                    help='weight decay')
parser.add_argument('--print-freq', default=100, type=int, metavar='N',
                    help='print frequency')
parser.add_argument('--pretrained', type=str, default="./")
parser.add_argument('--name', default="test", type=str)
parser.add_argument('--aug', action="store_true")


def main():
    args = parser.parse_args()
    args.ngpus_per_node = torch.cuda.device_count()
    args.rank = 0
    args.dist_url = f'tcp://localhost:{random.randrange(49152, 65535)}'
    args.world_size = args.ngpus_per_node
    torch.multiprocessing.spawn(main_worker, (args,), args.ngpus_per_node)


def main_worker(gpu, args):
    args.rank += gpu
    torch.distributed.init_process_group(
        backend='nccl', init_method=args.dist_url,
        world_size=args.world_size, rank=args.rank)

    torch.cuda.set_device(gpu)
    torch.backends.cudnn.benchmark = True

    model = models.resnet50().cuda(gpu)
    model.fc = nn.Identity()

    # loading path
    args.pretrained = args.pretrained + args.name + ".pth"

    state_dict = torch.load(args.pretrained, map_location='cpu')
    model.load_state_dict(state_dict["backbone"])
    model.requires_grad_(False)

    classifier = nn.Linear(2048, 1000).cuda(gpu)
    classifier.weight.data.normal_(mean=0.0, std=0.01)
    classifier.bias.data.zero_()
    classifier = torch.nn.parallel.DistributedDataParallel(classifier, device_ids=[gpu])

    criterion = nn.CrossEntropyLoss().cuda(gpu)

    param_groups = [dict(params=classifier.parameters(), lr=args.lr)]
    optimizer = optim.SGD(param_groups, lr=args.lr, momentum=0.9, weight_decay=args.weight_decay)

    start_epoch = 0
    best_acc = argparse.Namespace(top1=0, top5=0)
    

    # Data loading code
    traindir = args.data / 'train'
    valdir = args.data / 'val'
    normalize = transforms.Normalize(mean=[0.485, 0.456, 0.406],
                                     std=[0.229, 0.224, 0.225])

    train_dataset = datasets.ImageFolder(traindir, transforms.Compose([
            transforms.RandomResizedCrop(224),
            transforms.RandomHorizontalFlip(),
            transforms.ToTensor(),
            normalize,
            HighPassFilter() if not args.aug else Identity(),
        ]))
    val_dataset = datasets.ImageFolder(valdir, transforms.Compose([
            transforms.Resize(256),
            transforms.CenterCrop(224),
            transforms.ToTensor(),
            normalize,
            HighPassFilter() if not args.aug else Identity(),
        ]))

    train_sampler = torch.utils.data.distributed.DistributedSampler(train_dataset)
    kwargs = dict(batch_size=args.batch_size // args.world_size, num_workers=args.workers, pin_memory=True)
    train_loader = torch.utils.data.DataLoader(train_dataset, sampler=train_sampler, **kwargs)
    val_loader = torch.utils.data.DataLoader(val_dataset, **kwargs)

    start_time = time.time()
    model.eval()
    for epoch in range(start_epoch, args.epochs):
        train_sampler.set_epoch(epoch)
        for step, (images, target) in enumerate(train_loader, start=epoch * len(train_loader)):
            images = images.cuda(gpu, non_blocking=True)
            features = model(images)
            output = classifier(features)
            loss = criterion(output, target.cuda(gpu, non_blocking=True))
            lr = adjust_learning_rate(args, optimizer, train_loader, step)
            optimizer.zero_grad()
            loss.backward()
            optimizer.step()
            if step % args.print_freq == 0:
                torch.distributed.reduce(loss.div_(args.world_size), 0)
                if args.rank == 0:
                    pg = optimizer.param_groups
                    stats = dict(epoch=epoch, step=step,
                                 lr=lr, loss=loss.item(),
                                 time=int(time.time() - start_time))
                    print(json.dumps(stats))

        # evaluate
        if args.rank == 0:
            top1 = AverageMeter('Acc@1')
            top5 = AverageMeter('Acc@5')
            with torch.no_grad():
                for images, target in val_loader:
                    images = images.cuda(gpu, non_blocking=True)
                    features = model(images)
                    output = classifier(features)
                    acc1, acc5 = accuracy(output, target.cuda(gpu, non_blocking=True), topk=(1, 5))
                    top1.update(acc1[0].item(), images.size(0))
                    top5.update(acc5[0].item(), images.size(0))
            best_acc.top1 = max(best_acc.top1, top1.avg)
            best_acc.top5 = max(best_acc.top5, top5.avg)
            stats = dict(epoch=epoch, acc1=top1.avg, acc5=top5.avg, best_acc1=best_acc.top1, best_acc5=best_acc.top5)
            print(json.dumps(stats))


        # sanity check
        state_dict = torch.load(args.pretrained, map_location='cpu')
        for k, v in model.state_dict().items():
            assert torch.equal(v.cpu(), state_dict["backbone"][k]), k


class AverageMeter(object):
    """Computes and stores the average and current value"""
    def __init__(self, name, fmt=':f'):
        self.name = name
        self.fmt = fmt
        self.reset()

    def reset(self):
        self.val = 0
        self.avg = 0
        self.sum = 0
        self.count = 0

    def update(self, val, n=1):
        self.val = val
        self.sum += val * n
        self.count += n
        self.avg = self.sum / self.count

    def __str__(self):
        fmtstr = '{name} {val' + self.fmt + '} ({avg' + self.fmt + '})'
        return fmtstr.format(**self.__dict__)


def accuracy(output, target, topk=(1,)):
    """Computes the accuracy over the k top predictions for the specified values of k"""
    with torch.no_grad():
        maxk = max(topk)
        batch_size = target.size(0)

        _, pred = output.topk(maxk, 1, True, True)
        pred = pred.t()
        correct = pred.eq(target.view(1, -1).expand_as(pred))

        res = []
        for k in topk:
            correct_k = correct[:k].reshape(-1).float().sum(0, keepdim=True)
            res.append(correct_k.mul_(100.0 / batch_size))
        return res


class HighPassFilter(object):
    def __init__(self, kernel_size=13, sigma=5):
        self.kernel_size = kernel_size
        self.sigma = sigma

    def __call__(self, img):
        return img - transforms.functional.gaussian_blur(
            img, kernel_size=self.kernel_size, sigma=self.sigma
        )

class Identity(object):
    def __init__(self):
        pass

    def __call__(self, img):
        return img


def adjust_learning_rate(args, optimizer, loader, step):
    max_steps = args.epochs * len(loader)
    base_lr = args.lr * args.batch_size / 256
    end_lr = base_lr * 0.001

    warmup_steps = 5 * len(loader)
    if step < warmup_steps:
        lr = base_lr * step / warmup_steps
    else:
        step -= warmup_steps
        max_steps -= warmup_steps
        q = 0.5 * (1 + math.cos(math.pi * step / max_steps))
        lr = base_lr * q + end_lr * (1 - q)
    
    for param_group in optimizer.param_groups:
        param_group['lr'] = lr
    return lr


if __name__ == '__main__':
    main()
