# coding=utf-8
# pylint: skip-file
"""
Codes adapted from
https://github.com/yang-song/score_sde_pytorch/blob/main/models/ncsnpp.py
"""


from . import utils, layers, layerspp
import torch.nn as nn
import functools
import torch
import numpy as np


ResnetBlockDDPM = layerspp.ResnetBlockDDPMpp
ResnetBlockBigGAN = layerspp.ResnetBlockBigGANpp
Combine = layerspp.Combine
conv3x3 = layerspp.conv3x3
conv1x1 = layerspp.conv1x1
get_act = layers.get_act
default_initializer = layers.default_init


@utils.register_model(name='ncsnpp')
class NCSNpp(nn.Module):
  """NCSN++ model"""

  def __init__(self, flags):
    super().__init__()
    self.nf = nf = flags.ngf
    self.ch_mult = ch_mult = flags.ch_mult
    self.num_resolutions = num_resolutions = len(ch_mult)
    self.attn_resolutions = attn_resolutions = flags.attn
    self.num_res_blocks = num_res_blocks = flags.num_res_blocks
    dropout = flags.dropout
    self.act = act = get_act(flags.act)
    resamp_with_conv = flags.resamp_with_conv
    self.all_resolutions = all_resolutions = [flags.img_size // (2 ** i)
                                              for i in range(num_resolutions)]
    fir = flags.fir
    fir_kernel = flags.fir_kernel
    init_scale = flags.init_scale
    self.skip_rescale = skip_rescale = flags.skip_rescale
    self.resblock_type = resblock_type = flags.resblock_type.lower()
    self.progressive = progressive = flags.progressive.lower()
    self.progressive_input = progressive_input = flags.progressive_input.lower()
    self.embedding_type = embedding_type = flags.embedding_type.lower()
    combine_method = flags.progressive_combine.lower()
    assert progressive in ['none', 'output_skip', 'residual']
    assert progressive_input in ['none', 'input_skip', 'residual']
    assert embedding_type in ['fourier', 'positional']
    assert combine_method in ['sum', 'cat']

    # ===================== preparing blocks =====================
    combiner = functools.partial(Combine, method=combine_method)
    AttnBlock = functools.partial(layerspp.AttnBlockpp,
                                  init_scale=init_scale,
                                  skip_rescale=skip_rescale)
    Upsample = functools.partial(layerspp.Upsample,
                                 with_conv=resamp_with_conv,
                                 fir=fir, fir_kernel=fir_kernel)
    Downsample = functools.partial(layerspp.Downsample,
                                   with_conv=resamp_with_conv,
                                   fir=fir, fir_kernel=fir_kernel)

    if progressive == 'output_skip':
      self.pyramid_upsample = layerspp.Upsample(with_conv=False,
                                                fir=fir, fir_kernel=fir_kernel)
    elif progressive == 'residual':
      pyramid_upsample = functools.partial(layerspp.Upsample,
                                           with_conv=True,
                                           fir=fir, fir_kernel=fir_kernel)
    if progressive_input == 'input_skip':
      self.pyramid_downsample = layerspp.Downsample(with_conv=False,
                                                    fir=fir, fir_kernel=fir_kernel)
    elif progressive_input == 'residual':
      pyramid_downsample = functools.partial(layerspp.Downsample,
                                             with_conv=True,
                                             fir=fir, fir_kernel=fir_kernel)
    if resblock_type == 'ddpm':
      ResnetBlock = functools.partial(ResnetBlockDDPM,
                                      act=act,
                                      dropout=dropout,
                                      init_scale=init_scale,
                                      skip_rescale=skip_rescale,
                                      temb_dim=nf * 4)
    elif resblock_type == 'biggan':
      ResnetBlock = functools.partial(ResnetBlockBigGAN,
                                      act=act,
                                      dropout=dropout,
                                      fir=fir,
                                      fir_kernel=fir_kernel,
                                      init_scale=init_scale,
                                      skip_rescale=skip_rescale,
                                      temb_dim=nf * 4)
    else:
      raise ValueError(f'resblock type {resblock_type} unrecognized.')

    # ===================== model construction =====================
    # Timestep embedding
    if embedding_type == 'fourier':  # Gaussian Fourier features embeddings.
      self.time_embedding = layerspp.GaussianFourierTimeEmbedding(
        act=act, embedding_dim=nf, out_dim=nf * 4, scale=flags.fourier_scale
      )
    elif embedding_type == 'positional':  # Positional time embedding used in ddpm
      self.time_embedding = layerspp.PositionalTimeEmbedding(
        T=flags.T, act=act, embedding_dim=nf, out_dim=nf * 4
      )
    else:
      raise ValueError(f'embedding type {embedding_type} unknown.')

    # Head
    channels = flags.img_channels
    if progressive_input != 'none':
      input_pyramid_ch = channels
    self.head = conv3x3(channels, nf)
    chs = [nf]

    # Downsampling blocks
    self.downblocks = nn.ModuleList()
    now_ch = nf
    for i_level, mult in enumerate(ch_mult):
      out_ch = nf * mult

      for _ in range(num_res_blocks):
        self.downblocks.append(ResnetBlock(in_ch=now_ch, out_ch=out_ch))
        now_ch = out_ch
        if all_resolutions[i_level] in attn_resolutions:
          self.downblocks.append(AttnBlock(channels=now_ch))
        chs.append(now_ch)

      if i_level != num_resolutions - 1:
        if resblock_type == 'ddpm':
          self.downblocks.append(Downsample(in_ch=now_ch))
        else:
          self.downblocks.append(ResnetBlock(down=True, in_ch=now_ch))
        if progressive_input == 'input_skip':
          self.downblocks.append(combiner(dim1=input_pyramid_ch, dim2=now_ch))
          if combine_method == 'cat':
            now_ch *= 2
        elif progressive_input == 'residual':
          self.downblocks.append(pyramid_downsample(in_ch=input_pyramid_ch, out_ch=now_ch))
          input_pyramid_ch = now_ch
        chs.append(now_ch)

    # Middle blocks
    now_ch = chs[-1]
    self.middleblocks = nn.ModuleList(
      [ResnetBlock(in_ch=now_ch),
       AttnBlock(channels=now_ch),
       ResnetBlock(in_ch=now_ch),]
    )

    # Upsampling blocks
    self.upblocks = nn.ModuleList()
    pyramid_ch = 0
    for i_level, mult in reversed(list(enumerate(ch_mult))):
      out_ch = nf * mult

      for _ in range(num_res_blocks + 1):
        self.upblocks.append(ResnetBlock(in_ch=now_ch + chs.pop(), out_ch=out_ch))
        now_ch = out_ch

      if all_resolutions[i_level] in attn_resolutions:
        self.upblocks.append(AttnBlock(channels=now_ch))

      if progressive != 'none':
        if i_level == num_resolutions - 1:
          if progressive == 'output_skip':
            self.upblocks.append(nn.GroupNorm(num_groups=min(now_ch // 4, 32),
                                              num_channels=now_ch, eps=1e-6))
            self.upblocks.append(conv3x3(now_ch, channels, init_scale=init_scale))
            pyramid_ch = channels
          elif progressive == 'residual':
            self.upblocks.append(nn.GroupNorm(num_groups=min(now_ch // 4, 32),
                                              num_channels=now_ch, eps=1e-6))
            self.upblocks.append(conv3x3(now_ch, now_ch, bias=True))
            pyramid_ch = now_ch
          else:
            raise ValueError(f'{progressive} is not a valid name.')
        else:
          if progressive == 'output_skip':
            self.upblocks.append(nn.GroupNorm(num_groups=min(now_ch // 4, 32),
                                              num_channels=now_ch, eps=1e-6))
            self.upblocks.append(conv3x3(now_ch, channels, bias=True, init_scale=init_scale))
            pyramid_ch = channels
          elif progressive == 'residual':
            self.upblocks.append(pyramid_upsample(in_ch=pyramid_ch, out_ch=now_ch))
            pyramid_ch = now_ch
          else:
            raise ValueError(f'{progressive} is not a valid name')

      if i_level != 0:
        if resblock_type == 'ddpm':
          self.upblocks.append(Upsample(in_ch=now_ch))
        else:
          self.upblocks.append(ResnetBlock(in_ch=now_ch, up=True))

    assert not chs

    # Tail
    if progressive != 'output_skip':
      self.tail = nn.Sequential(
        nn.GroupNorm(num_groups=min(now_ch // 4, 32), num_channels=now_ch, eps=1e-6),
        act,
        conv3x3(now_ch, channels, init_scale=init_scale),
      )

  def forward(self, x, time_cond):
    # Timestep embedding
    temb = self.time_embedding(time_cond)

    # Head
    input_pyramid = None
    if self.progressive_input != 'none':
      input_pyramid = x
    hs = [self.head(x)]

    # Downsampling
    idx = 0
    for i_level in range(self.num_resolutions):

      for _ in range(self.num_res_blocks):
        h = self.downblocks[idx](hs[-1], temb)
        idx += 1
        if h.shape[-1] in self.attn_resolutions:
          h = self.downblocks[idx](h)
          idx += 1
        hs.append(h)

      if i_level != self.num_resolutions - 1:
        if self.resblock_type == 'ddpm':
          h = self.downblocks[idx](hs[-1])
          idx += 1
        else:
          h = self.downblocks[idx](hs[-1], temb)
          idx += 1
        if self.progressive_input == 'input_skip':
          input_pyramid = self.pyramid_downsample(input_pyramid)
          h = self.downblocks[idx](input_pyramid, h)
          idx += 1
        elif self.progressive_input == 'residual':
          input_pyramid = self.downblocks[idx](input_pyramid)
          idx += 1
          if self.skip_rescale:
            input_pyramid = (input_pyramid + h) / np.sqrt(2.)
          else:
            input_pyramid = input_pyramid + h
          h = input_pyramid
        hs.append(h)

    assert idx == len(self.downblocks)

    # Middle
    h = hs[-1]
    for layer in self.middleblocks:
      h = layer(h, temb)

    # Upsampling
    pyramid = None
    idx = 0
    for i_level in reversed(range(self.num_resolutions)):

      for _ in range(self.num_res_blocks + 1):
        h = self.upblocks[idx](torch.cat([h, hs.pop()], dim=1), temb)
        idx += 1

      if h.shape[-1] in self.attn_resolutions:
        h = self.upblocks[idx](h)
        idx += 1

      if self.progressive != 'none':
        if i_level == self.num_resolutions - 1:
          if self.progressive == 'output_skip' or 'residual':
            pyramid = self.act(self.upblocks[idx](h))
            idx += 1
            pyramid = self.upblocks[idx](pyramid)
            idx += 1
          else:
            raise ValueError(f'{self.progressive} is not a valid name.')
        else:
          if self.progressive == 'output_skip':
            pyramid = self.pyramid_upsample(pyramid)
            pyramid_h = self.act(self.upblocks[idx](h))
            idx += 1
            pyramid_h = self.upblocks[idx](pyramid_h)
            idx += 1
            pyramid = pyramid + pyramid_h
          elif self.progressive == 'residual':
            pyramid = self.upblocks[idx](pyramid)
            idx += 1
            if self.skip_rescale:
              pyramid = (pyramid + h) / np.sqrt(2.)
            else:
              pyramid = pyramid + h
            h = pyramid
          else:
            raise ValueError(f'{self.progressive} is not a valid name')

      if i_level != 0:
        if self.resblock_type == 'ddpm':
          h = self.upblocks[idx](h)
          idx += 1
        else:
          h = self.upblocks[idx](h, temb)
          idx += 1

    assert idx == len(self.upblocks)
    assert not hs

    # Tail
    if self.progressive == 'output_skip':
      h = pyramid
    else:
      h = self.tail(h)

    return h
