"""
Inference with bad questions as inputs
"""

import sys
sys.path.append('./')

import csv

import fire
import torch
import os
import copy
from typing import List

from peft import PeftModel, PeftConfig
from transformers import LlamaConfig, LlamaTokenizer, LlamaForCausalLM
from eval_utils.model_utils import load_model, load_peft_model
from eval_utils.prompt_utils import apply_prompt_template
import json
from collections import defaultdict
from eval_utils.chat_utils import question_read


def main(
    model_name,
    peft_model: str=None,
    quantization: bool=False,
    max_new_tokens = 512, #The maximum numbers of tokens to generate
    prompt_file: str='openai_finetuning/customized_data/manual_harmful_instructions.csv',
    prompt_template_style: str='base',
    seed: int=42, #seed value for reproducibility
    do_sample: bool=True, #Whether or not to use sampling ; use greedy decoding otherwise.
    use_cache: bool=True,  #[optional] Whether or not the model should use the past last key/values attentions Whether or not the model should use the past last key/values attentions (if applicable to the model) to speed up decoding.
    top_p: float=0.0, # [optional] If set to float < 1, only the smallest set of most probable tokens with probabilities that add up to top_p or higher are kept for generation.
    temperature: float=1.0, # [optional] The value used to modulate the next token probabilities.
    top_k: int=50, # [optional] The number of highest probability vocabulary tokens to keep for top-k-filtering.
    repetition_penalty: float=1.0, #The parameter for repetition penalty. 1.0 means no penalty.
    length_penalty: int=1, #[optional] Exponential penalty to the length that is used with beam-based generation. 
    use_fast_kernels: bool = False, # Enable using SDPA from PyTroch Accelerated Transformers, make use Flash Attention and Xformer memory-efficient kernels
    output_file: str = None,
    save_activation: bool = True,
    **kwargs
):
    

    ## Set the seeds for reproducibility
    torch.cuda.manual_seed(seed)
    torch.manual_seed(seed)
    
    model = load_model(model_name, quantization)
    if peft_model:
        model = load_peft_model(model, peft_model)

    model.eval()
    
    if use_fast_kernels:
        """
        Setting 'use_fast_kernels' will enable
        using of Flash Attention or Xformer memory-efficient kernels 
        based on the hardware being used. This would speed up inference when used for batched inputs.
        """
        try:
            from optimum.bettertransformer import BetterTransformer
            model = BetterTransformer.transform(model)    
        except ImportError:
            print("Module 'optimum' not found. Please install 'optimum' it before proceeding.")

    tokenizer = LlamaTokenizer.from_pretrained(model_name)
    # tokenizer.add_special_tokens(
    #     {
         
    #         "pad_token": "<PAD>",
    #     }
    # )
    # model.resize_token_embeddings(model.config.vocab_size + 1) 
    
    question_dataset = question_read(prompt_file)
    
    
    # Apply prompt template
    chats = apply_prompt_template(prompt_template_style, question_dataset, tokenizer)
    
    out = []

    act_all = defaultdict(dict)
    if save_activation:
        activation = {}
        def get_activation(name):
            def hook(model, input, output):
                activation[name] = output.detach()
            return hook
                
        for l in range(32):
            model.model.layers[l].mlp.register_forward_hook(get_activation('layer.%d.mlp' % l))
            model.model.layers[l].post_attention_layernorm.register_forward_hook(get_activation('layer.%d.post_attention_layernorm' % l))
                      
    with torch.no_grad():
        
        for idx, chat in enumerate(chats):
            
            tokens= torch.tensor(chat).long()
            tokens= tokens.unsqueeze(0)
            tokens= tokens.to("cuda:0")
            
            input_token_length = tokens.shape[1]
                
            outputs = model.generate(
                input_ids = tokens,
                max_new_tokens=max_new_tokens,
                do_sample=do_sample,
                top_p=top_p,
                temperature=temperature,
                use_cache=use_cache,
                top_k=top_k,
                repetition_penalty=repetition_penalty,
                length_penalty=length_penalty,
                **kwargs
            )
            
            output_text = tokenizer.decode(outputs[0][input_token_length:], skip_special_tokens=True)
            
            if save_activation:
                for l in range(32):
                    if idx not in act_all['mlp']:
                        act_all['mlp'][idx] = []
                    if idx not in act_all['post_attention_layernorm']:
                        act_all['post_attention_layernorm'][idx] = []
                        
                    act_all['mlp'][idx].append(activation['layer.%d.mlp' % l])
                    act_all['post_attention_layernorm'][idx].append(activation['layer.%d.post_attention_layernorm' % l])
            
            out.append({'prompt': question_dataset[idx], 'answer': output_text})
            print('\n\n\n')
            print('>>> sample - %d' % idx)
            print('prompt = ', question_dataset[idx])
            print('answer = ', output_text)
    
    if output_file is not None:
        with open(output_file, 'w') as f:
            for li in out:
                f.write(json.dumps(li))
                f.write("\n")

    if save_activation:
        import pickle
        
        act_file = output_file.replace('.jsonl', '_act.pkl')
        with open(act_file, 'wb') as handle:
            pickle.dump(act_all, handle, protocol=pickle.HIGHEST_PROTOCOL)
        print('activation saved at', act_file)
    
    # torch.save(model.state_dict(), './temp/bn_normeval.ckpt')

if __name__ == "__main__":
    fire.Fire(main)