#ifndef SPECTRUM_FRAGMENT_UTILS_H
#define SPECTRUM_FRAGMENT_UTILS_H

template <typename T>
std::vector<T> get_random_elements(const std::vector<T>& input_vector, size_t N) {
    // Create a random device and a random number generator
    std::random_device rd;
    std::mt19937 gen(rd());

    // Create a vector to store the indices
    std::vector<size_t> indices(input_vector.size());
    std::iota(indices.begin(), indices.end(), 0); // Fill with 0, 1, ..., input_vector.size()-1

    // Shuffle the indices
    std::shuffle(indices.begin(), indices.end(), gen);

    // Select the first N indices
    indices.resize(N);

    // Create a result vector to store the random elements
    std::vector<T> result;
    result.reserve(N);
    for (size_t index : indices) {
        result.push_back(input_vector[index]);
    }

    return result;
}

template<typename T>
std::vector<T> order_fragment(const std::vector<T>& fragment) {
    std::vector<T> reversed_fragment = fragment;
    std::reverse(reversed_fragment.begin(), reversed_fragment.end());
    if(fragment < reversed_fragment){
        return fragment;
    }else{
        return reversed_fragment;
    }
}

template<typename T>
vector<pair<vector<T>, vector<T>>> find_bodies_of_cyclic_fragment(const std::vector<T>&fragment, const std::vector<T> &location){
    if(fragment.size()==6){
        std::vector<pair<std::vector<T>,vector<T>>> bodies_and_locations;
        std::vector<int>location_template(6,-1);
        int k = 0;
        for(int j=0; j<fragment.size(); j++){
            if(fragment[j] != 0){
                location_template[j] = location[k];
                k++;
            }
        }
        for(int i = 0; i<fragment.size(); i++){
            std::vector<T> body = {fragment[(i+1)%6], fragment[(i+2)%6], fragment[(i+3)%6], fragment[(i+4)%6], fragment[(i+5)%6]};
            vector<T> body_location;
            for(int j = i+1; j<i+6; j++){
                if(location_template[j%6] != -1){
                    body_location.push_back(location_template[j%6]);
                }
            }
            if(i%2==0){
                body.insert(body.begin(),0);
                body.push_back(0);
            }
            vector<T> reversed_body = body;
            std::reverse(reversed_body.begin(), reversed_body.end());
            if (body < reversed_body) {
                bodies_and_locations.push_back(std::make_pair(body, body_location));
            } else if(body > reversed_body){
                vector<NodeId> reversed_location = body_location;
                std::reverse(reversed_location.begin(), reversed_location.end());
                bodies_and_locations.push_back(std::make_pair(reversed_body, reversed_location));
            }else{
                vector<NodeId> reversed_location = body_location;
                std::reverse(reversed_location.begin(), reversed_location.end());
                if(reversed_location<body_location){
                    bodies_and_locations.push_back(std::make_pair(reversed_body, reversed_location));
                }else{
                    bodies_and_locations.push_back(std::make_pair(body, body_location));
                }
            }

        }
        return bodies_and_locations;
    }
}

template<typename T>
std::vector<std::vector<size_t>> find_canonical_order(std::vector<T> fragment) {
    if (fragment.size() == 6) {
        std::vector<size_t> indices = {0, 1, 2, 3, 4, 5};

        auto compare_indices = [&fragment](size_t a, size_t b) {
            return fragment[a] < fragment[b];
        };

        std::vector<std::vector<size_t>> permutations = {
                {0, 1, 2, 3, 4, 5},
                {0, 5, 4, 3, 2, 1},
                {2, 1, 0, 5, 4, 3},
                {2, 3, 4, 5, 0, 1},
                {4, 5, 0, 1, 2, 3},
                {4, 3, 2, 1, 0, 5}
        };

        // Sort permutations to find the canonical order
        std::sort(permutations.begin(), permutations.end(),
                  [&fragment, &compare_indices](const std::vector<size_t> &a, const std::vector<size_t> &b) {
                      for (size_t i = 0; i < a.size(); ++i) {
                          if (fragment[a[i]] != fragment[b[i]]) {
                              return compare_indices(a[i], b[i]);
                          }
                      }
                      return false;
                  });

        // Find all permutations that are equivalent to the canonical order
        std::vector<std::vector<size_t>> result;
        std::vector<size_t> canonical = permutations.front();

        for (const auto& perm : permutations) {
            bool is_equivalent = true;
            for (size_t i = 0; i < perm.size(); ++i) {
                if (fragment[perm[i]] != fragment[canonical[i]]) {
                    is_equivalent = false;
                    break;
                }
            }
            if (is_equivalent) {
                result.push_back(perm);
            }
        }

        return result;
    }

    return {}; // Return an empty vector if the size is not 6
}
template<typename T>
std::pair<std::vector<T>, std::vector<T>> reorder_cyclic_fragment(const std::vector<T>& fragment, const std::vector<T>& location, const vector<std::vector<T>>& possible_orders){
    if(fragment.size()==6){
        vector<vector<T>> cleaned_locations;
        vector<vector<T>> ordered_fragments;
        for(auto order:possible_orders){
            std::vector<T>ordered_fragment(6,0);
            std::vector<int>location_template(6,-1);
            int k = 0;
            for(int i=0; i<fragment.size(); i++){
                if(fragment[i] != 0){
                    location_template[i] = location[k];
                    k++;
                }
            }
            for(int i = 0; i<order.size(); ++i){
                ordered_fragment[i] = fragment[order[i]];
            }
            ordered_fragments.push_back(ordered_fragment);
            std::vector<int> ordered_location(6,0);
            for(int i = 0; i<order.size(); i ++){
                ordered_location[i] = location_template[order[i]];
            }
            std::vector<T> cleaned_location{};
            for(int i = 0; i<ordered_location.size(); i ++){
                if(ordered_location[i] != -1){
                    cleaned_location.push_back(ordered_location[i]);
                }
            }
            cleaned_locations.push_back(cleaned_location);
        }
        vector<T> best_order = cleaned_locations[0];
        size_t best_order_index = 0;
        for(size_t i = 1; i<cleaned_locations.size(); i++){
            if(cleaned_locations[i] < best_order){
                best_order = cleaned_locations[i];
                best_order_index = i;
            }
        }
        return std::make_pair(ordered_fragments[best_order_index],cleaned_locations[best_order_index]);
    }
}

template<typename T>
std::pair<std::vector<T>, std::vector<T>> order_cyclic_fragment_and_location(std::vector<T> fragment, std::vector<T> location) {
    vector<std::vector<T>> ordered_indices_possibilities = find_canonical_order(fragment);
    if(fragment.size()==6){
        return reorder_cyclic_fragment(fragment,location,ordered_indices_possibilities);
    }else{
        return reorder_cyclic_fragment(fragment,location,ordered_indices_possibilities);
    }

}

#endif //SPECTRUM_FRAGMENT_UTILS_H
