import numpy as np
import time
import matplotlib.pyplot as plt

def generate_annotator_data(num_responses, num_annotators):
    np.random.seed(42)
    preferences = np.zeros((num_responses, num_responses))
    for _ in range(num_annotators):
        for i in range(num_responses):
            for j in range(i + 1, num_responses):
                if np.random.rand() < 0.5:
                    preferences[i, j] += 1
                else:
                    preferences[j, i] += 1
    return preferences

# Zermelo's Algorithm for annotator preferences
def zermelo_algorithm_annotators(preferences, max_iter=1000, tol=1e-6):
    num_responses = len(preferences)
    strengths = np.ones(num_responses)
    
    for iteration in range(max_iter):
        strengths_new = np.zeros(num_responses)
        for i in range(num_responses):
            wins = np.sum(preferences[i])
            denominator = 0
            for j in range(num_responses):
                if i != j:
                    denominator += (preferences[i, j] + preferences[j, i]) / (strengths[i] + strengths[j])
            strengths_new[i] = wins / denominator if denominator != 0 else 0

        # Convergence check
        if np.linalg.norm(strengths_new - strengths, ord=1) < tol:
            break

        strengths = strengths_new

    return strengths

# Newman's Enhanced Iterative Algorithm for annotator preferences
def newman_algorithm_annotators(preferences, max_iter=1000, tol=1e-6):
    num_responses = len(preferences)
    strengths = np.ones(num_responses)
    
    for iteration in range(max_iter):
        strengths_new = np.zeros(num_responses)
        for i in range(num_responses):
            numerator = 0
            denominator = 0
            for j in range(num_responses):
                if i != j:
                    numerator += preferences[i, j] * strengths[j] / (strengths[i] + strengths[j])
                    denominator += preferences[j, i] / (strengths[i] + strengths[j])
            strengths_new[i] = numerator / denominator if denominator != 0 else 0

        # Convergence check
        if np.linalg.norm(strengths_new - strengths, ord=1) < tol:
            break

        strengths = strengths_new

    return strengths

# Function to retrieve preference matrix from a list of rankings by multiple annotators for each sample
def retrieve_preference_matrix(rank_list):
    """
    rank_list: List of rankings provided by different annotators for a given set of responses.
               rank_list[i] contains the rankings of annotator i.
               The shape is (num_annotators, num_responses), with each row indicating the rank of each response.
    """
    num_annotators = len(rank_list)
    num_responses = len(rank_list[0])
    preference_matrix = np.zeros((num_responses, num_responses))

    # Iterate through each annotator's ranking
    for annotator_ranks in rank_list:
        for i in range(num_responses):
            for j in range(num_responses):
                if i != j:
                    # If response i ranks higher than response j, increment preference count
                    if annotator_ranks[i] < annotator_ranks[j]:
                        preference_matrix[i, j] += 1

    return preference_matrix

# Defining the function to compute the logit value as per the given formula
def compute_logit(preferences):
    # Identifying the index of responses with strongest and lowest strengths
    strengths = newman_algorithm_annotators(preferences)

    strongest_index = np.argmax(strengths)
    lowest_index = np.argmin(strengths)

    # Extracting the strengths for the strongest and lowest responses
    w_s = strengths[strongest_index]
    w_l = strengths[lowest_index]

    # Computing the logit value
    r_star = np.log(w_s / (w_s + w_l))

    return strongest_index, lowest_index, w_s, w_l, r_star

def compute_logit_2(w_s, w_l):

    r_star = np.log(w_s / (w_s + w_l))

    return r_star