import numpy as np
import cv2

def convert_color_factory(src: str, dst: str):
    code = getattr(cv2, f"COLOR_{src.upper()}2{dst.upper()}")

    def convert_color(img: np.ndarray) -> np.ndarray:
        out_img = cv2.cvtColor(img, code)
        return out_img

    convert_color.__doc__ = f"""Convert a {src.upper()} image to {dst.upper()}
    image.

    Args:
        img (ndarray or str): The input image.

    Returns:
        ndarray: The converted {dst.upper()} image.
    """

    return convert_color


bgr2rgb = convert_color_factory("bgr", "rgb")

rgb2bgr = convert_color_factory("rgb", "bgr")

bgr2hsv = convert_color_factory("bgr", "hsv")

hsv2bgr = convert_color_factory("hsv", "bgr")

bgr2hls = convert_color_factory("bgr", "hls")

hls2bgr = convert_color_factory("hls", "bgr")