from typing import Sequence, Union

import torch.nn as nn

from allenact.base_abstractions.preprocessor import Preprocessor
from allenact.utils.experiment_utils import Builder, TrainingPipeline
from allenact_plugins.clip_plugin.clip_preprocessors import (
    ClipViTPreprocessor,
)
from projects.plugins.ithor_plugin.ithor_sensors import (
    GoalObjectTypeThorSensor,
    RGBSensorThor,
)
from projects.object_navigation.baseline_configs.clip.zeroshot_mixins import (
    DataGenPreprocessGRUActorCriticMixin,
)
from projects.object_navigation.baseline_configs.robothor.zeroshot_objectnav_robothor_base import (
    ZeroshotObjectNavRoboThorBaseConfig,
)
from projects.object_navigation.baseline_configs.robothor.clip.objectnav_robothor_rgb_clipresnet50gru_ddppo import (
    ObjectNavRoboThorClipRGBPPOExperimentConfig,
)
from projects.object_navigation.baseline_configs.ithor.objectnav_ithor_base import (
    ObjectNaviThorBaseConfig,
)
from projects.object_navigation.baseline_configs.navigation_base import ObjectNavPPOMixin


from allenact.base_abstractions.sensor import ExpertActionSensor
from projects.plugins.robothor_plugin.robothor_tasks import ObjectNavTask


class ObjectNaviThorClipViTRGBPPOExperimentConfig(ObjectNaviThorBaseConfig):
    """A CLIP Object Navigation experiment configuration in RoboThor
    with RGB input."""

    CLIP_MODEL_TYPE = "ViT-B/32"

    SENSORS = [
        RGBSensorThor(
            height=ObjectNaviThorBaseConfig.SCREEN_SIZE,
            width=ObjectNaviThorBaseConfig.SCREEN_SIZE,
            use_resnet_normalization=True,
            mean=ClipViTPreprocessor.CLIP_RGB_MEANS,
            stdev=ClipViTPreprocessor.CLIP_RGB_STDS,
            uuid="rgb_lowres",
        ),
        GoalObjectTypeThorSensor(object_types=ObjectNaviThorBaseConfig.TARGET_TYPES,),
        ExpertActionSensor(len(ObjectNavTask.class_action_names())),
    ]
    
    PROMPT = False

    def __init__(self, **kwargs):
        super().__init__(**kwargs)

        self.preprocessing_and_model = DataGenPreprocessGRUActorCriticMixin(
            sensors=self.SENSORS,
            clip_model_type=self.CLIP_MODEL_TYPE,
            screen_size=self.SCREEN_SIZE,
            goal_sensor_type=GoalObjectTypeThorSensor,
            pool=False,
            pooling_type='',
            target_types=self.TARGET_TYPES,
            prompt = self.PROMPT
        )

    def training_pipeline(self, **kwargs) -> TrainingPipeline:
        return ObjectNavPPOMixin.training_pipeline(
            auxiliary_uuids=[],
            multiple_beliefs=False,
            advance_scene_rollout_period=self.ADVANCE_SCENE_ROLLOUT_PERIOD,
        )

    def preprocessors(self) -> Sequence[Union[Preprocessor, Builder[Preprocessor]]]:
        return self.preprocessing_and_model.preprocessors()

    def create_model(self, **kwargs) -> nn.Module:
        return self.preprocessing_and_model.create_model(
            num_actions=self.ACTION_SPACE.n, **kwargs
        )

    def tag(cls):
        return "ObjectNav-iTHOR-RGB-DATA-GERNERATION"
